const axios = require('axios');

/**
 * Amadeus API Configuration
 * Test Environment Configuration with provided credentials
 */
class AmadeusConfig {
  constructor() {
    // Environment-based configuration
    const environment = process.env.AMADEUS_ENVIRONMENT || 'test';

    if (environment === 'production') {
      // Production environment credentials
      this.apiKey = process.env.AMADEUS_API_KEY;
      this.apiSecret = process.env.AMADEUS_API_SECRET;
      this.baseUrl = 'https://api.amadeus.com';

      if (!this.apiKey || !this.apiSecret) {
        throw new Error('Production Amadeus API credentials not found in environment variables');
      }
    } else {
      // Test environment credentials (fallback)
      this.apiKey = 'sjGyAD8I7zrCQQSa7NDttuzi8D1asMoY';
      this.apiSecret = 'ihHqqJ3jjK3kVf1b';
      this.baseUrl = 'https://test.api.amadeus.com';
    }

    this.environment = environment;

    // Token management
    this.accessToken = null;
    this.tokenExpiresAt = null;

    // API endpoints
    this.endpoints = {
      auth: '/v1/security/oauth2/token',
      flightOffers: '/v2/shopping/flight-offers',
      flightOffersPrice: '/v1/shopping/flight-offers/pricing',
      flightCreate: '/v1/booking/flight-orders',
      flightStatus: '/v1/schedule/flights',
      airportLocation: '/v1/reference-data/locations',
      airlines: '/v1/reference-data/airlines',
      checkinLinks: '/v1/reference-data/urls/checkin-links'
    };
  }

  /**
   * Get or refresh access token
   */
  async getAccessToken() {
    // Check if token is still valid
    if (this.accessToken && this.tokenExpiresAt && Date.now() < this.tokenExpiresAt) {
      return this.accessToken;
    }

    try {
      const response = await axios.post(`${this.baseUrl}${this.endpoints.auth}`, {
        grant_type: 'client_credentials',
        client_id: this.apiKey,
        client_secret: this.apiSecret
      }, {
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded'
        }
      });

      this.accessToken = response.data.access_token;
      // Set expiration time with 5 minutes buffer
      this.tokenExpiresAt = Date.now() + ((response.data.expires_in - 300) * 1000);

      console.log('✅ Amadeus access token retrieved successfully');
      return this.accessToken;

    } catch (error) {
      console.error('❌ Failed to get Amadeus access token:', error.response?.data || error.message);
      throw new Error('Amadeus authentication failed');
    }
  }

  /**
   * Make authenticated API request
   */
  async makeRequest(endpoint, method = 'GET', data = null, params = null) {
    const token = await this.getAccessToken();

    const config = {
      method,
      url: `${this.baseUrl}${endpoint}`,
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    };

    if (params) {
      config.params = params;
    }

    if (data && (method === 'POST' || method === 'PUT')) {
      config.data = data;
    }

    try {
      const response = await axios(config);
      return response.data;
    } catch (error) {
      console.error(`❌ Amadeus API request failed (${method} ${endpoint}):`, error.response?.data || error.message);
      throw error;
    }
  }

  /**
   * Test API connection
   */
  async testConnection() {
    try {
      await this.getAccessToken();
      console.log('✅ Amadeus API connection test successful');
      return true;
    } catch (error) {
      console.error('❌ Amadeus API connection test failed:', error.message);
      return false;
    }
  }
}

module.exports = AmadeusConfig;