/**
 * Permission Middleware
 * Kullanıcının belirli bir işlem için yetkisi olup olmadığını kontrol eder
 */

// Token'dan kullanıcı bilgisini çıkar (basit Base64 decode)
function getUserFromToken(token) {
  if (!token) return null;

  try {
    // Bearer prefix'ini kaldır
    const cleanToken = token.replace('Bearer ', '');

    // Base64 decode
    const decoded = Buffer.from(cleanToken, 'base64').toString('utf-8');
    const user = JSON.parse(decoded);

    return user;
  } catch (error) {
    console.error('Token decode error:', error);
    return null;
  }
}

// Kullanıcının belirli bir izne sahip olup olmadığını kontrol et
function hasPermission(user, requiredPermissions) {
  if (!user || !user.permissions) {
    return false;
  }

  const userPermissions = Array.isArray(user.permissions)
    ? user.permissions
    : [];

  // Wildcard (*) iznine sahipse her şeye erişebilir
  if (userPermissions.includes('*')) {
    return true;
  }

  // system_admin veya agency_manager rolleri çoğu şeye erişebilir
  const adminRoles = ['system_admin', 'agency_manager'];
  if (adminRoles.includes(user.role)) {
    return true;
  }

  // Eğer gerekli izin listesi boşsa (herkese açık)
  if (!requiredPermissions || requiredPermissions.length === 0) {
    return true;
  }

  // En az bir gerekli izne sahip mi kontrol et
  return requiredPermissions.some(permission =>
    userPermissions.includes(permission)
  );
}

// Middleware factory - belirli izinleri kontrol eder
function requirePermissions(...requiredPermissions) {
  return (req, res, next) => {
    try {
      // Authorization header'dan token al
      const authHeader = req.headers.authorization;

      if (!authHeader) {
        return res.status(401).json({
          error: 'Yetkilendirme gerekli',
          code: 'UNAUTHORIZED'
        });
      }

      // Token'dan kullanıcı bilgisini çıkar
      const user = getUserFromToken(authHeader);

      if (!user) {
        return res.status(401).json({
          error: 'Geçersiz token',
          code: 'INVALID_TOKEN'
        });
      }

      // Kullanıcının yetkisini kontrol et
      if (!hasPermission(user, requiredPermissions)) {
        return res.status(403).json({
          error: 'Bu işlem için yetkiniz yok',
          code: 'FORBIDDEN',
          requiredPermissions,
          userRole: user.role
        });
      }

      // Kullanıcı bilgisini request'e ekle
      req.user = user;

      next();
    } catch (error) {
      console.error('Permission middleware error:', error);
      return res.status(500).json({
        error: 'Yetki kontrolünde hata oluştu',
        code: 'INTERNAL_ERROR'
      });
    }
  };
}

// Sadece belirli rollere izin ver
function requireRole(...allowedRoles) {
  return (req, res, next) => {
    try {
      const authHeader = req.headers.authorization;

      if (!authHeader) {
        return res.status(401).json({
          error: 'Yetkilendirme gerekli',
          code: 'UNAUTHORIZED'
        });
      }

      const user = getUserFromToken(authHeader);

      if (!user) {
        return res.status(401).json({
          error: 'Geçersiz token',
          code: 'INVALID_TOKEN'
        });
      }

      if (!allowedRoles.includes(user.role)) {
        return res.status(403).json({
          error: 'Bu işlem için yetkiniz yok',
          code: 'FORBIDDEN',
          requiredRoles: allowedRoles,
          userRole: user.role
        });
      }

      req.user = user;
      next();
    } catch (error) {
      console.error('Role middleware error:', error);
      return res.status(500).json({
        error: 'Yetki kontrolünde hata oluştu',
        code: 'INTERNAL_ERROR'
      });
    }
  };
}

// Kullanıcı kendi kaydını mı düzenliyor kontrol et
function requireOwnership(getUserIdFromRequest) {
  return (req, res, next) => {
    try {
      const authHeader = req.headers.authorization;

      if (!authHeader) {
        return res.status(401).json({
          error: 'Yetkilendirme gerekli',
          code: 'UNAUTHORIZED'
        });
      }

      const user = getUserFromToken(authHeader);

      if (!user) {
        return res.status(401).json({
          error: 'Geçersiz token',
          code: 'INVALID_TOKEN'
        });
      }

      const targetUserId = getUserIdFromRequest(req);

      // Admin'ler her şeyi düzenleyebilir
      const adminRoles = ['system_admin', 'agency_manager'];
      if (adminRoles.includes(user.role)) {
        req.user = user;
        return next();
      }

      // Kendi kaydını düzenliyorsa izin ver
      if (user.id === targetUserId) {
        req.user = user;
        return next();
      }

      return res.status(403).json({
        error: 'Bu kaydı düzenleme yetkiniz yok',
        code: 'FORBIDDEN'
      });
    } catch (error) {
      console.error('Ownership middleware error:', error);
      return res.status(500).json({
        error: 'Yetki kontrolünde hata oluştu',
        code: 'INTERNAL_ERROR'
      });
    }
  };
}

module.exports = {
  requirePermissions,
  requireRole,
  requireOwnership,
  getUserFromToken,
  hasPermission
};
