# Ticket Request API Routes

RESTful API endpoints for managing ticket requests in the vTravel Panel system.

## Base URL
```
http://localhost:3002/api/ticket-requests
```

## Endpoints

### 1. Create Ticket Request
**POST** `/api/ticket-requests`

Creates a new ticket request with optional passenger associations.

**Request Body:**
```json
{
  "customerId": "uuid",
  "airline": "THY|PC|AJET|OTHER",
  "departureAirport": "IST",
  "arrivalAirport": "AYT",
  "departureDate": "2025-10-15",
  "departureTime": "14:30",
  "status": "pending|confirmed|ticketed|cancelled|rejected",
  "requestDate": "2025-10-09",
  "flightNumber": "TK123",
  "ticketPrice": 1500.00,
  "commission": 150.00,
  "paymentStatus": "paid|pending|cancelled",
  "paymentMethod": "cash|credit_card|bank_transfer",
  "notes": "Window seat preferred",
  "passengers": [
    {
      "firstName": "John",
      "lastName": "Doe",
      "phone": "5551234567",
      "email": "john@example.com",
      "age": 35,
      "isChild": false
    }
  ]
}
```

**Response:**
```json
{
  "success": true,
  "message": "Ticket request created successfully",
  "data": {
    "id": "uuid",
    "customerId": "uuid",
    "airline": "THY",
    "pnr": null,
    "departureAirport": "IST",
    "arrivalAirport": "AYT",
    "departureDate": "2025-10-15",
    "status": "pending",
    "passengers": [...]
  }
}
```

### 2. List Ticket Requests
**GET** `/api/ticket-requests`

Retrieves filtered and paginated list of ticket requests.

**Query Parameters:**
- `customerId` - Filter by customer UUID
- `status` - Filter by status (pending, confirmed, ticketed, cancelled, rejected)
- `airline` - Filter by airline (THY, PC, AJET, OTHER)
- `departureDate` - Filter by departure date (YYYY-MM-DD)
- `search` - Search by PNR or passenger name
- `limit` - Results per page (default: 20)
- `offset` - Pagination offset (default: 0)

**Example:**
```
GET /api/ticket-requests?status=confirmed&airline=THY&limit=10&offset=0
```

**Response:**
```json
{
  "success": true,
  "data": [...],
  "pagination": {
    "limit": 10,
    "offset": 0,
    "total": 25
  }
}
```

### 3. Get Upcoming Tickets
**GET** `/api/ticket-requests/upcoming`

Retrieves tickets departing within the next 24 hours.

**Response:**
```json
{
  "success": true,
  "data": [...],
  "count": 5
}
```

### 4. Get Ticket Details
**GET** `/api/ticket-requests/:id`

Retrieves detailed information about a specific ticket including all passengers.

**Response:**
```json
{
  "success": true,
  "data": {
    "id": "uuid",
    "customerId": "uuid",
    "airline": "THY",
    "pnr": "ABC123",
    "departureAirport": "IST",
    "arrivalAirport": "AYT",
    "departureDate": "2025-10-15",
    "departureTime": "14:30",
    "status": "ticketed",
    "passengers": [
      {
        "id": "uuid",
        "firstName": "John",
        "lastName": "Doe",
        "phone": "5551234567",
        "email": "john@example.com"
      }
    ]
  }
}
```

### 5. Update Ticket Request
**PUT** `/api/ticket-requests/:id`

Updates ticket request information.

**Allowed Update Fields:**
- departureAirport
- arrivalAirport
- departureDate
- departureTime
- status
- notes
- airline
- flightNumber
- ticketPrice
- commission
- paymentStatus
- paymentMethod

**Request Body:**
```json
{
  "status": "confirmed",
  "flightNumber": "TK456",
  "notes": "Updated notes"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Ticket request updated successfully",
  "data": {...}
}
```

### 6. Cancel Ticket Request
**DELETE** `/api/ticket-requests/:id`

Soft deletes a ticket request by setting status to 'cancelled'.

**Response:**
```json
{
  "success": true,
  "message": "Ticket request cancelled successfully",
  "data": {
    "id": "uuid",
    "status": "cancelled"
  }
}
```

### 7. Assign PNR to Ticket
**POST** `/api/ticket-requests/:id/assign-pnr`

Assigns a PNR code to a ticket and updates status to 'ticketed'.

**Request Body:**
```json
{
  "pnr": "ABC123"
}
```

**Validation:**
- PNR must be exactly 6 alphanumeric characters
- PNR cannot be assigned to multiple tickets

**Response:**
```json
{
  "success": true,
  "message": "PNR assigned successfully",
  "data": {...}
}
```

### 8. Add Passenger to Ticket
**POST** `/api/ticket-requests/:ticketId/passengers`

Associates a passenger with a ticket request.

**Request Body (New Passenger):**
```json
{
  "firstName": "Jane",
  "lastName": "Smith",
  "phone": "5559876543",
  "email": "jane@example.com",
  "age": 28,
  "isChild": false
}
```

**Request Body (Existing Passenger):**
```json
{
  "id": "existing-passenger-uuid"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Passenger added to ticket successfully",
  "data": {
    "ticket": {...},
    "passenger": {...}
  }
}
```

### 9. Remove Passenger from Ticket
**DELETE** `/api/ticket-requests/:ticketId/passengers/:passengerId`

Removes a passenger association from a ticket request.

**Response:**
```json
{
  "success": true,
  "message": "Passenger removed from ticket successfully",
  "data": {...}
}
```

## Error Responses

### Validation Error (400)
```json
{
  "error": "Validation failed",
  "details": [
    "customerId is required",
    "airline is required"
  ]
}
```

### Not Found (404)
```json
{
  "error": "Not found",
  "message": "Ticket request not found"
}
```

### Conflict (409)
```json
{
  "error": "Conflict",
  "message": "This PNR is already assigned to another ticket"
}
```

### Internal Server Error (500)
```json
{
  "error": "Internal server error",
  "message": "Failed to create ticket request"
}
```

## Real-time Updates (Socket.IO)

The API emits real-time events via Socket.IO:

- `ticket_updated` - When a ticket is updated
- `ticket_cancelled` - When a ticket is cancelled
- `pnr_assigned` - When a PNR is assigned
- `passenger_added` - When a passenger is added
- `passenger_removed` - When a passenger is removed

**Event Format:**
```json
{
  "ticketId": "uuid",
  "timestamp": "2025-10-09T10:30:00.000Z",
  // ... additional event-specific data
}
```

## Database Functions Used

The route integrates with the following database functions:

- `createTicketRequest(data)`
- `getTicketRequest(id)`
- `listTicketRequests(filters, pagination)`
- `updateTicketRequest(id, updates)`
- `deleteTicketRequest(id)`
- `assignPnrToTicket(ticketId, pnr)`
- `getTicketWithPassengers(ticketId)`
- `getUpcomingTickets()`
- `addPassengerToTicket(ticketRequestId, passengerId)`
- `removePassengerFromTicket(ticketRequestId, passengerId)`
- `createPassenger(data)`
- `getPassenger(id)`

## Integration with Server

The routes are registered in `server/index.js`:

```javascript
const ticketRequestRoutes = require('./routes/ticket-requests');
app.use('/api/ticket-requests', timeoutMiddleware(ticketRequestRoutes));
```

Features:
- Timeout protection (25 seconds)
- Error handling middleware
- Real-time Socket.IO integration
- Request/response logging

## Testing Examples

### Create a ticket request:
```bash
curl -X POST http://localhost:3002/api/ticket-requests \
  -H "Content-Type: application/json" \
  -d '{
    "customerId": "customer-uuid",
    "airline": "THY",
    "departureAirport": "IST",
    "arrivalAirport": "AYT",
    "departureDate": "2025-10-15",
    "status": "pending",
    "requestDate": "2025-10-09",
    "passengers": [
      {
        "firstName": "John",
        "lastName": "Doe"
      }
    ]
  }'
```

### List confirmed tickets:
```bash
curl "http://localhost:3002/api/ticket-requests?status=confirmed&limit=10"
```

### Assign PNR:
```bash
curl -X POST http://localhost:3002/api/ticket-requests/{id}/assign-pnr \
  -H "Content-Type: application/json" \
  -d '{"pnr": "ABC123"}'
```
