const express = require('express');
const router = express.Router();
const db = require('../services/database');

// Helper function to validate date
function isValidDate(dateString) {
  const date = new Date(dateString);
  return date instanceof Date && !isNaN(date);
}

// Helper function to calculate nights between dates
function calculateNights(checkIn, checkOut) {
  const checkInDate = new Date(checkIn);
  const checkOutDate = new Date(checkOut);
  const diffTime = Math.abs(checkOutDate - checkInDate);
  return Math.ceil(diffTime / (1000 * 60 * 60 * 24));
}

// Helper function to validate status
function isValidStatus(status) {
  const validStatuses = ['pending', 'searching', 'quoted', 'confirmed', 'booked', 'cancelled', 'completed'];
  return validStatuses.includes(status);
}

// ============================================
// POST /api/accommodation-requests
// Create new accommodation request
// ============================================
router.post('/', async (req, res) => {
  try {
    const {
      customerId,
      requestedByUserId,
      destination,
      checkInDate,
      checkOutDate,
      nights,
      totalGuests,
      roomType,
      roomCount,
      budgetMin,
      budgetMax,
      currency,
      specialRequests,
      mealPlan,
      hotelCategory,
      status,
      agentNotes,
      guests
    } = req.body;

    // Required field validation
    if (!customerId) {
      return res.status(400).json({
        success: false,
        error: 'Customer ID is required'
      });
    }

    if (!destination || !destination.trim()) {
      return res.status(400).json({
        success: false,
        error: 'Destination is required'
      });
    }

    if (!checkInDate) {
      return res.status(400).json({
        success: false,
        error: 'Check-in date is required'
      });
    }

    if (!checkOutDate) {
      return res.status(400).json({
        success: false,
        error: 'Check-out date is required'
      });
    }

    if (!totalGuests || totalGuests < 1) {
      return res.status(400).json({
        success: false,
        error: 'Total guests must be at least 1'
      });
    }

    if (!roomCount || roomCount < 1) {
      return res.status(400).json({
        success: false,
        error: 'Room count must be at least 1'
      });
    }

    if (!roomType || !roomType.trim()) {
      return res.status(400).json({
        success: false,
        error: 'Room type is required'
      });
    }

    // Validate dates
    if (!isValidDate(checkInDate)) {
      return res.status(400).json({
        success: false,
        error: 'Invalid check-in date format'
      });
    }

    if (!isValidDate(checkOutDate)) {
      return res.status(400).json({
        success: false,
        error: 'Invalid check-out date format'
      });
    }

    // Check if check-out is after check-in
    if (new Date(checkOutDate) <= new Date(checkInDate)) {
      return res.status(400).json({
        success: false,
        error: 'Check-out date must be after check-in date'
      });
    }

    // Validate nights match date difference
    const calculatedNights = calculateNights(checkInDate, checkOutDate);
    if (nights && nights !== calculatedNights) {
      return res.status(400).json({
        success: false,
        error: `Nights (${nights}) do not match date difference (${calculatedNights} nights)`
      });
    }

    // Validate status if provided
    if (status && !isValidStatus(status)) {
      return res.status(400).json({
        success: false,
        error: 'Invalid status. Must be one of: pending, searching, quoted, confirmed, booked, cancelled, completed'
      });
    }

    // Validate guests array
    if (!guests || !Array.isArray(guests) || guests.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'At least one guest is required'
      });
    }

    // Validate each guest
    for (let i = 0; i < guests.length; i++) {
      const guest = guests[i];
      if (!guest.firstName || !guest.firstName.trim()) {
        return res.status(400).json({
          success: false,
          error: `Guest ${i + 1}: First name is required`
        });
      }
      if (!guest.lastName || !guest.lastName.trim()) {
        return res.status(400).json({
          success: false,
          error: `Guest ${i + 1}: Last name is required`
        });
      }
    }

    // Verify customer exists
    const customer = await db.getCustomer(customerId);
    if (!customer) {
      return res.status(404).json({
        success: false,
        error: 'Customer not found'
      });
    }

    // Create accommodation request
    const accommodationData = {
      customerId,
      requestedByUserId: requestedByUserId || null,
      destination: destination.trim(),
      checkInDate,
      checkOutDate,
      nights: nights || calculatedNights,
      totalGuests,
      roomType: roomType.trim(),
      roomCount,
      budgetMin: budgetMin || null,
      budgetMax: budgetMax || null,
      currency: currency || 'TRY',
      specialRequests: specialRequests ? specialRequests.trim() : null,
      mealPlan: mealPlan || null,
      hotelCategory: hotelCategory || null,
      status: status || 'pending',
      agentNotes: agentNotes ? agentNotes.trim() : null
    };

    const accommodationRequest = await db.createAccommodationRequest(accommodationData);

    // Add guests to the accommodation request
    const guestPromises = guests.map(guest =>
      db.addGuestToAccommodation(accommodationRequest.id, {
        firstName: guest.firstName.trim(),
        lastName: guest.lastName.trim(),
        age: guest.age || null,
        isChild: guest.isChild || false,
        specialNeeds: guest.specialNeeds ? guest.specialNeeds.trim() : null
      })
    );

    await Promise.all(guestPromises);

    // Get the complete request with guests
    const completeRequest = await db.getAccommodationWithGuests(accommodationRequest.id);

    console.log(`✅ Accommodation request created: ${accommodationRequest.id} for ${destination}`);

    res.status(201).json({
      success: true,
      data: completeRequest
    });

  } catch (error) {
    console.error('❌ Create accommodation request error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to create accommodation request',
      message: error.message
    });
  }
});

// ============================================
// GET /api/accommodation-requests/:id
// Get accommodation request details with guests
// ============================================
router.get('/:id', async (req, res) => {
  try {
    const { id } = req.params;

    const request = await db.getAccommodationWithGuests(id);

    if (!request) {
      return res.status(404).json({
        success: false,
        error: 'Accommodation request not found'
      });
    }

    res.json({
      success: true,
      data: request
    });

  } catch (error) {
    console.error('❌ Get accommodation request error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to retrieve accommodation request',
      message: error.message
    });
  }
});

// ============================================
// GET /api/accommodation-requests
// List accommodation requests with filters
// ============================================
router.get('/', async (req, res) => {
  try {
    const {
      customerId,
      status,
      destination,
      checkInDateFrom,
      checkInDateTo,
      limit = 20,
      offset = 0
    } = req.query;

    // Build filters
    const filters = {};
    if (customerId) filters.customerId = customerId;
    if (status) {
      if (!isValidStatus(status)) {
        return res.status(400).json({
          success: false,
          error: 'Invalid status. Must be one of: pending, searching, quoted, confirmed, booked, cancelled, completed'
        });
      }
      filters.status = status;
    }
    if (destination) filters.destination = destination;

    // Note: Date range filtering would require additional database function support
    // For now, we'll list and filter in application layer if needed
    if (checkInDateFrom || checkInDateTo) {
      console.log('⚠️ Date range filtering will be applied post-query');
    }

    const pagination = {
      limit: Math.min(parseInt(limit) || 20, 100), // Max 100 per page
      offset: parseInt(offset) || 0
    };

    const result = await db.listAccommodationRequests(filters, pagination);

    // Apply date range filtering if specified
    let filteredData = result.data;
    if (checkInDateFrom || checkInDateTo) {
      filteredData = result.data.filter(request => {
        const checkInDate = new Date(request.checkInDate);
        if (checkInDateFrom && checkInDate < new Date(checkInDateFrom)) return false;
        if (checkInDateTo && checkInDate > new Date(checkInDateTo)) return false;
        return true;
      });
    }

    res.json({
      success: true,
      data: filteredData,
      total: result.total,
      limit: result.limit,
      offset: result.offset,
      filters: {
        customerId,
        status,
        destination,
        checkInDateFrom,
        checkInDateTo
      }
    });

  } catch (error) {
    console.error('❌ List accommodation requests error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to list accommodation requests',
      message: error.message
    });
  }
});

// ============================================
// PUT /api/accommodation-requests/:id
// Update accommodation request
// ============================================
router.put('/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const updates = req.body;

    // Check if request exists
    const existingRequest = await db.getAccommodationRequest(id);
    if (!existingRequest) {
      return res.status(404).json({
        success: false,
        error: 'Accommodation request not found'
      });
    }

    // Validate dates if being updated
    if (updates.checkInDate || updates.checkOutDate) {
      const checkInDate = updates.checkInDate || existingRequest.checkInDate;
      const checkOutDate = updates.checkOutDate || existingRequest.checkOutDate;

      if (!isValidDate(checkInDate) || !isValidDate(checkOutDate)) {
        return res.status(400).json({
          success: false,
          error: 'Invalid date format'
        });
      }

      if (new Date(checkOutDate) <= new Date(checkInDate)) {
        return res.status(400).json({
          success: false,
          error: 'Check-out date must be after check-in date'
        });
      }

      // Auto-calculate nights if dates changed
      if (updates.checkInDate || updates.checkOutDate) {
        updates.nights = calculateNights(checkInDate, checkOutDate);
      }
    }

    // Validate status if being updated
    if (updates.status && !isValidStatus(updates.status)) {
      return res.status(400).json({
        success: false,
        error: 'Invalid status. Must be one of: pending, searching, quoted, confirmed, booked, cancelled, completed'
      });
    }

    // Validate numeric fields
    if (updates.totalGuests !== undefined && updates.totalGuests < 1) {
      return res.status(400).json({
        success: false,
        error: 'Total guests must be at least 1'
      });
    }

    if (updates.roomCount !== undefined && updates.roomCount < 1) {
      return res.status(400).json({
        success: false,
        error: 'Room count must be at least 1'
      });
    }

    // Trim string fields
    if (updates.destination) updates.destination = updates.destination.trim();
    if (updates.roomType) updates.roomType = updates.roomType.trim();
    if (updates.specialRequests) updates.specialRequests = updates.specialRequests.trim();
    if (updates.agentNotes) updates.agentNotes = updates.agentNotes.trim();

    // Update the request
    const updatedRequest = await db.updateAccommodationRequest(id, updates);

    console.log(`✅ Accommodation request updated: ${id}`);

    res.json({
      success: true,
      data: updatedRequest
    });

  } catch (error) {
    console.error('❌ Update accommodation request error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to update accommodation request',
      message: error.message
    });
  }
});

// ============================================
// DELETE /api/accommodation-requests/:id
// Cancel accommodation request (soft delete)
// ============================================
router.delete('/:id', async (req, res) => {
  try {
    const { id } = req.params;

    // Check if request exists
    const existingRequest = await db.getAccommodationRequest(id);
    if (!existingRequest) {
      return res.status(404).json({
        success: false,
        error: 'Accommodation request not found'
      });
    }

    // Check if already cancelled
    if (existingRequest.status === 'cancelled') {
      return res.status(400).json({
        success: false,
        error: 'Accommodation request is already cancelled'
      });
    }

    // Soft delete by setting status to cancelled
    const success = await db.deleteAccommodationRequest(id);

    if (success) {
      console.log(`✅ Accommodation request cancelled: ${id}`);

      res.json({
        success: true,
        message: 'Accommodation request cancelled successfully'
      });
    } else {
      res.status(500).json({
        success: false,
        error: 'Failed to cancel accommodation request'
      });
    }

  } catch (error) {
    console.error('❌ Delete accommodation request error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to cancel accommodation request',
      message: error.message
    });
  }
});

// ============================================
// PUT /api/accommodation-requests/:id/status
// Update accommodation request status with booking details
// ============================================
router.put('/:id/status', async (req, res) => {
  try {
    const { id } = req.params;
    const { status, hotelName, confirmationNumber, totalPrice, agentNotes } = req.body;

    // Check if request exists
    const existingRequest = await db.getAccommodationRequest(id);
    if (!existingRequest) {
      return res.status(404).json({
        success: false,
        error: 'Accommodation request not found'
      });
    }

    // Validate status
    if (!status) {
      return res.status(400).json({
        success: false,
        error: 'Status is required'
      });
    }

    if (!isValidStatus(status)) {
      return res.status(400).json({
        success: false,
        error: 'Invalid status. Must be one of: pending, searching, quoted, confirmed, booked, cancelled, completed'
      });
    }

    // Build status update details
    const details = {};

    // For 'booked' status, require booking details
    if (status === 'booked') {
      if (!hotelName || !hotelName.trim()) {
        return res.status(400).json({
          success: false,
          error: 'Hotel name is required for booked status'
        });
      }
      if (!confirmationNumber || !confirmationNumber.trim()) {
        return res.status(400).json({
          success: false,
          error: 'Confirmation number is required for booked status'
        });
      }
      details.hotelName = hotelName.trim();
      details.confirmationNumber = confirmationNumber.trim();
      if (totalPrice) details.totalPrice = totalPrice;
    }

    // Add optional fields if provided
    if (hotelName && status !== 'booked') details.hotelName = hotelName.trim();
    if (confirmationNumber && status !== 'booked') details.confirmationNumber = confirmationNumber.trim();
    if (totalPrice) details.totalPrice = totalPrice;
    if (agentNotes) details.agentNotes = agentNotes.trim();

    // Update status
    const updatedRequest = await db.updateAccommodationStatus(id, status, details);

    console.log(`✅ Accommodation request status updated: ${id} -> ${status}`);

    res.json({
      success: true,
      data: updatedRequest,
      message: `Status updated to ${status}`
    });

  } catch (error) {
    console.error('❌ Update accommodation status error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to update accommodation status',
      message: error.message
    });
  }
});

// ============================================
// POST /api/accommodation-requests/:id/guests
// Add guest to accommodation request
// ============================================
router.post('/:id/guests', async (req, res) => {
  try {
    const { id } = req.params;
    const { firstName, lastName, age, isChild, specialNeeds } = req.body;

    // Check if request exists
    const existingRequest = await db.getAccommodationRequest(id);
    if (!existingRequest) {
      return res.status(404).json({
        success: false,
        error: 'Accommodation request not found'
      });
    }

    // Validate guest data
    if (!firstName || !firstName.trim()) {
      return res.status(400).json({
        success: false,
        error: 'Guest first name is required'
      });
    }

    if (!lastName || !lastName.trim()) {
      return res.status(400).json({
        success: false,
        error: 'Guest last name is required'
      });
    }

    // Add guest
    const guestData = {
      firstName: firstName.trim(),
      lastName: lastName.trim(),
      age: age || null,
      isChild: isChild || false,
      specialNeeds: specialNeeds ? specialNeeds.trim() : null
    };

    const guest = await db.addGuestToAccommodation(id, guestData);

    console.log(`✅ Guest added to accommodation request ${id}: ${firstName} ${lastName}`);

    res.status(201).json({
      success: true,
      data: guest,
      message: 'Guest added successfully'
    });

  } catch (error) {
    console.error('❌ Add guest error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to add guest',
      message: error.message
    });
  }
});

module.exports = router;
