const express = require('express');
const router = express.Router();
const db = require('../services/db');
const crypto = require('crypto');

// Generate UUID
function generateId() {
  return crypto.randomUUID();
}

// GET /api/agency-users - Tüm kullanıcıları listele
router.get('/', async (req, res) => {
  try {
    const { department_id, role, is_active } = req.query;

    let query = `
      SELECT
        au.*,
        d.name as department_name,
        d.color as department_color,
        m.first_name as manager_first_name,
        m.last_name as manager_last_name
      FROM agency_users au
      LEFT JOIN departments d ON d.id = au.department_id
      LEFT JOIN agency_users m ON m.id = au.manager_id
      WHERE 1=1
    `;

    const params = [];

    if (department_id) {
      query += ' AND au.department_id = ?';
      params.push(department_id);
    }

    if (role) {
      query += ' AND au.role = ?';
      params.push(role);
    }

    if (is_active !== undefined) {
      query += ' AND au.is_active = ?';
      params.push(is_active === 'true' ? 1 : 0);
    }
    // If is_active is not specified, show all users (both active and inactive)

    query += ' ORDER BY au.created_at DESC';

    const users = await db.all(query, params);

    res.json(users);
  } catch (error) {
    console.error('Error fetching agency users:', error);
    res.status(500).json({ error: 'Failed to fetch agency users' });
  }
});

// GET /api/agency-users/:id - Tek kullanıcı detayı
router.get('/:id', async (req, res) => {
  try {
    const user = await db.get(`
      SELECT
        au.*,
        d.name as department_name,
        d.color as department_color,
        m.first_name as manager_first_name,
        m.last_name as manager_last_name
      FROM agency_users au
      LEFT JOIN departments d ON d.id = au.department_id
      LEFT JOIN agency_users m ON m.id = au.manager_id
      WHERE au.id = ?
    `, [req.params.id]);

    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    res.json(user);
  } catch (error) {
    console.error('Error fetching user:', error);
    res.status(500).json({ error: 'Failed to fetch user' });
  }
});

// POST /api/agency-users - Yeni kullanıcı oluştur
router.post('/', async (req, res) => {
  try {
    const {
      first_name,
      last_name,
      email,
      phone,
      avatar,
      department_id,
      position,
      role,
      permissions,
      manager_id,
      is_active
    } = req.body;

    // Validation
    if (!first_name || !last_name || !email || !department_id || !position || !role) {
      return res.status(400).json({
        error: 'First name, last name, email, department, position, and role are required'
      });
    }

    // Check if email already exists
    const existingUser = await db.get('SELECT id FROM agency_users WHERE email = ?', [email]);
    if (existingUser) {
      return res.status(400).json({ error: 'Email already exists' });
    }

    const id = generateId();

    await db.run(`
      INSERT INTO agency_users (
        id, first_name, last_name, email, phone, avatar,
        department_id, position, role, permissions,
        is_active, manager_id,
        total_tickets, total_revenue,
        created_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 0, 0, CURRENT_TIMESTAMP)
    `, [
      id,
      first_name,
      last_name,
      email,
      phone || null,
      avatar || null,
      department_id,
      position,
      role,
      permissions ? JSON.stringify(permissions) : null,
      is_active !== undefined ? is_active : 1,
      manager_id || null
    ]);

    // Update department user count
    await db.run(`
      UPDATE departments
      SET user_count = (
        SELECT COUNT(*) FROM agency_users WHERE department_id = departments.id AND is_active = 1
      )
      WHERE id = ?
    `, [department_id]);

    const user = await db.get('SELECT * FROM agency_users WHERE id = ?', [id]);

    res.status(201).json(user);
  } catch (error) {
    console.error('Error creating user:', error);
    res.status(500).json({ error: 'Failed to create user' });
  }
});

// PUT /api/agency-users/:id - Kullanıcı güncelle
router.put('/:id', async (req, res) => {
  try {
    const {
      first_name,
      last_name,
      email,
      phone,
      avatar,
      department_id,
      position,
      role,
      permissions,
      manager_id,
      is_active,
      is_online
    } = req.body;

    // Get old user data for department count updates
    const oldUser = await db.get('SELECT department_id FROM agency_users WHERE id = ?', [req.params.id]);

    if (!oldUser) {
      return res.status(404).json({ error: 'User not found' });
    }

    // Build dynamic UPDATE query for partial updates
    const updates = [];
    const params = [];

    if (first_name !== undefined) {
      updates.push('first_name = ?');
      params.push(first_name);
    }
    if (last_name !== undefined) {
      updates.push('last_name = ?');
      params.push(last_name);
    }
    if (email !== undefined) {
      updates.push('email = ?');
      params.push(email);
    }
    if (phone !== undefined) {
      updates.push('phone = ?');
      params.push(phone || null);
    }
    if (avatar !== undefined) {
      updates.push('avatar = ?');
      params.push(avatar || null);
    }
    if (department_id !== undefined) {
      updates.push('department_id = ?');
      params.push(department_id);
    }
    if (position !== undefined) {
      updates.push('position = ?');
      params.push(position);
    }
    if (role !== undefined) {
      updates.push('role = ?');
      params.push(role);
    }
    if (permissions !== undefined) {
      updates.push('permissions = ?');
      params.push(permissions ? JSON.stringify(permissions) : null);
    }
    if (manager_id !== undefined) {
      updates.push('manager_id = ?');
      params.push(manager_id || null);
    }
    if (is_active !== undefined) {
      updates.push('is_active = ?');
      params.push(is_active ? 1 : 0);
    }
    if (is_online !== undefined) {
      updates.push('is_online = ?');
      params.push(is_online ? 1 : 0);
    }

    // Always update timestamp
    updates.push('updated_at = CURRENT_TIMESTAMP');

    // Add user ID for WHERE clause
    params.push(req.params.id);

    if (updates.length === 1) {
      return res.status(400).json({ error: 'No fields to update' });
    }

    await db.run(`
      UPDATE agency_users
      SET ${updates.join(', ')}
      WHERE id = ?
    `, params);

    // Update department user counts if department changed
    const newDepartmentId = department_id !== undefined ? department_id : oldUser.department_id;
    if (oldUser.department_id !== newDepartmentId) {
      await db.run(`
        UPDATE departments
        SET user_count = (
          SELECT COUNT(*) FROM agency_users WHERE department_id = departments.id AND is_active = 1
        )
        WHERE id IN (?, ?)
      `, [oldUser.department_id, newDepartmentId]);
    } else if (is_active !== undefined) {
      // If only is_active changed, still update the count
      await db.run(`
        UPDATE departments
        SET user_count = (
          SELECT COUNT(*) FROM agency_users WHERE department_id = departments.id AND is_active = 1
        )
        WHERE id = ?
      `, [oldUser.department_id]);
    }

    const user = await db.get('SELECT * FROM agency_users WHERE id = ?', [req.params.id]);

    res.json(user);
  } catch (error) {
    console.error('Error updating user:', error);
    res.status(500).json({ error: 'Failed to update user' });
  }
});

// PATCH /api/agency-users/:id/deactivate - Kullanıcıyı pasifleştir (soft delete)
router.patch('/:id/deactivate', async (req, res) => {
  try {
    const user = await db.get('SELECT department_id FROM agency_users WHERE id = ?', [req.params.id]);

    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    await db.run(
      'UPDATE agency_users SET is_active = 0, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
      [req.params.id]
    );

    // Update department user count
    await db.run(`
      UPDATE departments
      SET user_count = (
        SELECT COUNT(*) FROM agency_users WHERE department_id = departments.id AND is_active = 1
      )
      WHERE id = ?
    `, [user.department_id]);

    res.json({ message: 'User deactivated successfully' });
  } catch (error) {
    console.error('Error deactivating user:', error);
    res.status(500).json({ error: 'Failed to deactivate user' });
  }
});

// DELETE /api/agency-users/:id/permanent - Kullanıcıyı kalıcı olarak sil (hard delete)
router.delete('/:id/permanent', async (req, res) => {
  try {
    const user = await db.get('SELECT department_id FROM agency_users WHERE id = ?', [req.params.id]);

    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    // Permanently delete user from database
    await db.run('DELETE FROM agency_users WHERE id = ?', [req.params.id]);

    // Update department user count
    await db.run(`
      UPDATE departments
      SET user_count = (
        SELECT COUNT(*) FROM agency_users WHERE department_id = departments.id AND is_active = 1
      )
      WHERE id = ?
    `, [user.department_id]);

    res.json({ message: 'User permanently deleted successfully' });
  } catch (error) {
    console.error('Error permanently deleting user:', error);
    res.status(500).json({ error: 'Failed to permanently delete user' });
  }
});

// GET /api/agency-users/stats/overview - Kullanıcı istatistikleri
router.get('/stats/overview', async (req, res) => {
  try {
    const stats = await db.get(`
      SELECT
        COUNT(*) as total_users,
        COUNT(CASE WHEN is_active = 1 THEN 1 END) as active_users,
        COUNT(CASE WHEN is_online = 1 THEN 1 END) as online_users,
        COUNT(CASE WHEN role = 'super_admin' THEN 1 END) as super_admins,
        COUNT(CASE WHEN role = 'admin' THEN 1 END) as admins,
        COUNT(CASE WHEN role = 'manager' THEN 1 END) as managers,
        COUNT(CASE WHEN role IN ('senior_agent', 'agent') THEN 1 END) as agents
      FROM agency_users
    `);

    res.json(stats);
  } catch (error) {
    console.error('Error fetching user stats:', error);
    res.status(500).json({ error: 'Failed to fetch user stats' });
  }
});

module.exports = router;
