const express = require('express');
const router = express.Router();
const db = require('../services/db');
const crypto = require('crypto');
const bcrypt = require('bcrypt');

// Simple JWT token generation (in production, use proper JWT library)
function generateToken(user) {
  const payload = {
    userId: user.id,
    email: user.email,
    role: user.role,
    timestamp: Date.now()
  };
  return Buffer.from(JSON.stringify(payload)).toString('base64');
}

// POST /api/auth/login - Kullanıcı girişi
router.post('/login', async (req, res) => {
  try {
    const { email, password } = req.body;

    // Validation
    if (!email || !password) {
      return res.status(400).json({
        error: 'E-posta ve şifre gereklidir'
      });
    }

    // Find user by email
    const user = await db.get(`
      SELECT
        au.*,
        d.name as department_name,
        d.color as department_color
      FROM agency_users au
      LEFT JOIN departments d ON d.id = au.department_id
      WHERE au.email = ? AND au.is_active = 1
    `, [email]);

    if (!user) {
      return res.status(401).json({
        error: 'E-posta veya şifre hatalı'
      });
    }

    // Verify password with bcrypt
    if (!user.password) {
      // If user has no password set, reject login
      return res.status(401).json({
        error: 'Bu kullanıcı için şifre ayarlanmamış. Lütfen yöneticinize başvurun.'
      });
    }

    const isValidPassword = await bcrypt.compare(password, user.password);
    if (!isValidPassword) {
      return res.status(401).json({
        error: 'E-posta veya şifre hatalı'
      });
    }

    // Update last login
    await db.run(
      'UPDATE agency_users SET is_online = 1, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
      [user.id]
    );

    // Generate token
    const token = generateToken(user);

    // Prepare user response (exclude sensitive data)
    let permissions = [];
    try {
      if (user.permissions) {
        permissions = typeof user.permissions === 'string'
          ? JSON.parse(user.permissions)
          : user.permissions;
      }
    } catch (error) {
      console.error('Error parsing permissions:', error);
      permissions = [];
    }

    const userResponse = {
      id: user.id,
      firstName: user.first_name,
      lastName: user.last_name,
      email: user.email,
      phone: user.phone,
      avatar: user.avatar,
      role: user.role,
      departmentId: user.department_id,
      department: {
        name: user.department_name,
        color: user.department_color
      },
      permissions: permissions,
      isOnline: true
    };

    res.json({
      success: true,
      message: 'Giriş başarılı',
      token,
      user: userResponse
    });
  } catch (error) {
    console.error('Error during login:', error);
    res.status(500).json({ error: 'Giriş sırasında bir hata oluştu' });
  }
});

// POST /api/auth/logout - Kullanıcı çıkışı
router.post('/logout', async (req, res) => {
  try {
    const { userId } = req.body;

    if (userId) {
      await db.run(
        'UPDATE agency_users SET is_online = 0, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
        [userId]
      );
    }

    res.json({
      success: true,
      message: 'Çıkış başarılı'
    });
  } catch (error) {
    console.error('Error during logout:', error);
    res.status(500).json({ error: 'Çıkış sırasında bir hata oluştu' });
  }
});

// GET /api/auth/me - Mevcut kullanıcı bilgisi
router.get('/me', async (req, res) => {
  try {
    // In production, verify token from Authorization header
    const authHeader = req.headers.authorization;
    if (!authHeader) {
      return res.status(401).json({ error: 'Yetkilendirme gerekli' });
    }

    const token = authHeader.replace('Bearer ', '');
    const payload = JSON.parse(Buffer.from(token, 'base64').toString());

    const user = await db.get(`
      SELECT
        au.*,
        d.name as department_name,
        d.color as department_color
      FROM agency_users au
      LEFT JOIN departments d ON d.id = au.department_id
      WHERE au.id = ? AND au.is_active = 1
    `, [payload.userId]);

    if (!user) {
      return res.status(401).json({ error: 'Kullanıcı bulunamadı' });
    }

    const userResponse = {
      id: user.id,
      firstName: user.first_name,
      lastName: user.last_name,
      email: user.email,
      phone: user.phone,
      avatar: user.avatar,
      role: user.role,
      departmentId: user.department_id,
      department: {
        name: user.department_name,
        color: user.department_color
      },
      permissions: user.permissions ? JSON.parse(user.permissions) : [],
      isOnline: user.is_online
    };

    res.json({
      success: true,
      user: userResponse
    });
  } catch (error) {
    console.error('Error getting current user:', error);
    res.status(500).json({ error: 'Kullanıcı bilgisi alınamadı' });
  }
});

module.exports = router;
