const express = require('express');
const router = express.Router();

// In-memory storage (gerçek projede database kullanılacak)
let customers = [
  {
    id: '1',
    firstName: 'Ahmet',
    lastName: 'Yılmaz',
    email: 'ahmet@example.com',
    phone: '05321234567',
    isVip: true,
    notes: 'Cam kenarı koltuk tercih ediyor',
    createdAt: new Date('2025-01-01'),
    updatedAt: new Date('2025-01-01')
  },
  {
    id: '2',
    firstName: 'Fatma',
    lastName: 'Demir',
    email: 'fatma@example.com',
    phone: '05329876543',
    isVip: false,
    notes: '',
    createdAt: new Date('2025-01-02'),
    updatedAt: new Date('2025-01-02')
  },
  {
    id: '3',
    firstName: 'Mehmet',
    lastName: 'Kaya',
    email: 'mehmet@example.com',
    phone: '05325556677',
    isVip: false,
    notes: 'Koridor koltuk',
    createdAt: new Date('2025-01-03'),
    updatedAt: new Date('2025-01-03')
  }
];

let nextId = 4;

// GET /api/customers - Tüm müşterileri listele
router.get('/', (req, res) => {
  const { search, vip } = req.query;
  let filteredCustomers = customers;

  // Arama filtresi
  if (search) {
    const searchLower = search.toLowerCase();
    filteredCustomers = filteredCustomers.filter(customer => 
      customer.firstName.toLowerCase().includes(searchLower) ||
      customer.lastName.toLowerCase().includes(searchLower) ||
      customer.email.toLowerCase().includes(searchLower) ||
      customer.phone.includes(search)
    );
  }

  // VIP filtresi
  if (vip !== undefined) {
    filteredCustomers = filteredCustomers.filter(customer => 
      customer.isVip === (vip === 'true')
    );
  }

  res.json({
    customers: filteredCustomers,
    total: filteredCustomers.length
  });
});

// GET /api/customers/:id - Tek müşteri detayı
router.get('/:id', (req, res) => {
  const customer = customers.find(c => c.id === req.params.id);
  
  if (!customer) {
    return res.status(404).json({ error: 'Müşteri bulunamadı' });
  }

  res.json(customer);
});

// POST /api/customers - Yeni müşteri ekle
router.post('/', (req, res) => {
  const { firstName, lastName, email, phone, notes, isVip } = req.body;

  // Validation
  if (!firstName || !lastName || !phone) {
    return res.status(400).json({
      error: 'Ad, soyad ve telefon zorunludur'
    });
  }

  // Duplicate check
  const existingCustomer = customers.find(c => 
    c.phone === phone || (email && c.email === email)
  );

  if (existingCustomer) {
    return res.status(409).json({
      error: 'Bu telefon numarası veya email ile müşteri zaten mevcut'
    });
  }

  const newCustomer = {
    id: nextId.toString(),
    firstName: firstName.trim(),
    lastName: lastName.trim(),
    email: email?.trim() || '',
    phone: phone.trim(),
    notes: notes?.trim() || '',
    isVip: !!isVip,
    createdAt: new Date(),
    updatedAt: new Date()
  };

  customers.push(newCustomer);
  nextId++;

  res.status(201).json(newCustomer);
});

// PUT /api/customers/:id - Müşteri güncelle
router.put('/:id', (req, res) => {
  const customerIndex = customers.findIndex(c => c.id === req.params.id);
  
  if (customerIndex === -1) {
    return res.status(404).json({ error: 'Müşteri bulunamadı' });
  }

  const { firstName, lastName, email, phone, notes, isVip } = req.body;

  // Validation
  if (!firstName || !lastName || !phone) {
    return res.status(400).json({
      error: 'Ad, soyad ve telefon zorunludur'
    });
  }

  // Duplicate check (excluding current customer)
  const existingCustomer = customers.find(c => 
    c.id !== req.params.id && 
    (c.phone === phone || (email && c.email === email))
  );

  if (existingCustomer) {
    return res.status(409).json({
      error: 'Bu telefon numarası veya email ile başka müşteri mevcut'
    });
  }

  const updatedCustomer = {
    ...customers[customerIndex],
    firstName: firstName.trim(),
    lastName: lastName.trim(),
    email: email?.trim() || '',
    phone: phone.trim(),
    notes: notes?.trim() || '',
    isVip: !!isVip,
    updatedAt: new Date()
  };

  customers[customerIndex] = updatedCustomer;
  res.json(updatedCustomer);
});

// DELETE /api/customers/:id - Müşteri sil
router.delete('/:id', (req, res) => {
  const customerIndex = customers.findIndex(c => c.id === req.params.id);
  
  if (customerIndex === -1) {
    return res.status(404).json({ error: 'Müşteri bulunamadı' });
  }

  const deletedCustomer = customers.splice(customerIndex, 1)[0];
  res.json({ 
    message: 'Müşteri silindi', 
    customer: deletedCustomer 
  });
});

// GET /api/customers/:id/history - Müşteri check-in geçmişi
router.get('/:id/history', (req, res) => {
  const customer = customers.find(c => c.id === req.params.id);
  
  if (!customer) {
    return res.status(404).json({ error: 'Müşteri bulunamadı' });
  }

  // Mock check-in history data
  const history = [
    {
      id: '1',
      pnr: 'ABC123',
      flightNumber: 'TK 1234',
      from: 'İstanbul (IST)',
      to: 'Antalya (AYT)',
      date: '2025-08-15',
      status: 'completed',
      seat: '12A',
      createdAt: new Date('2025-08-14')
    },
    {
      id: '2',
      pnr: 'DEF456',
      flightNumber: 'TK 5678',
      from: 'Antalya (AYT)',
      to: 'İstanbul (IST)',
      date: '2025-08-20',
      status: 'completed',
      seat: '15C',
      createdAt: new Date('2025-08-19')
    }
  ];

  res.json({
    customer,
    history,
    total: history.length
  });
});

module.exports = router;