const express = require('express');
const router = express.Router();

// Import database functions
const {
  createCustomer,
  getCustomer,
  listCustomers,
  updateCustomer,
  deleteCustomer,
  searchCustomers,
  getCustomerStats,
  createCorporateUser,
  getCorporateUser,
  listCorporateUsers,
  updateCorporateUser,
  deleteCorporateUser
} = require('../services/database');

// ============================================
// CUSTOMER ROUTES
// ============================================

/**
 * POST /api/customers
 * Create a new customer
 *
 * Body:
 * - type: 'individual' | 'corporate' | 'agency' (required)
 * - phone: string (required)
 * - email: string (optional)
 * - firstName: string (for individual)
 * - lastName: string (for individual)
 * - companyName: string (for corporate/agency)
 * - taxNumber: string (optional)
 * - address: string (optional)
 * - isVip: boolean (optional)
 * - notes: string (optional)
 */
router.post('/', async (req, res) => {
  try {
    const { type, phone, email, firstName, lastName, companyName } = req.body;

    // Validation
    if (!type || !phone) {
      return res.status(400).json({
        success: false,
        error: 'Type and phone are required'
      });
    }

    // Validate customer type
    if (!['individual', 'corporate', 'agency'].includes(type)) {
      return res.status(400).json({
        success: false,
        error: 'Type must be individual, corporate, or agency'
      });
    }

    // Type-specific validation
    if (type === 'individual' && (!firstName || !lastName)) {
      return res.status(400).json({
        success: false,
        error: 'firstName and lastName are required for individual customers'
      });
    }

    if ((type === 'corporate' || type === 'agency') && !companyName) {
      return res.status(400).json({
        success: false,
        error: 'companyName is required for corporate/agency customers'
      });
    }

    // Create customer
    const customer = await createCustomer(req.body);

    res.status(201).json({
      success: true,
      data: customer
    });
  } catch (error) {
    console.error('Error creating customer:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/customers/:id
 * Get customer details by ID
 */
router.get('/:id', async (req, res) => {
  try {
    const customer = await getCustomer(req.params.id);

    if (!customer) {
      return res.status(404).json({
        success: false,
        error: 'Customer not found'
      });
    }

    res.json({
      success: true,
      data: customer
    });
  } catch (error) {
    console.error('Error getting customer:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * PUT /api/customers/:id
 * Update customer
 *
 * Body: Same as POST, all fields optional except those that maintain data integrity
 */
router.put('/:id', async (req, res) => {
  try {
    const { id } = req.params;

    // Check if customer exists
    const existingCustomer = await getCustomer(id);
    if (!existingCustomer) {
      return res.status(404).json({
        success: false,
        error: 'Customer not found'
      });
    }

    // Type-specific validation if type is being updated
    const { type, firstName, lastName, companyName } = req.body;
    const customerType = type || existingCustomer.type;

    if (customerType === 'individual' && firstName !== undefined && lastName !== undefined) {
      if (!firstName || !lastName) {
        return res.status(400).json({
          success: false,
          error: 'firstName and lastName cannot be empty for individual customers'
        });
      }
    }

    if ((customerType === 'corporate' || customerType === 'agency') && companyName !== undefined) {
      if (!companyName) {
        return res.status(400).json({
          success: false,
          error: 'companyName cannot be empty for corporate/agency customers'
        });
      }
    }

    // Update customer
    const updatedCustomer = await updateCustomer(id, req.body);

    res.json({
      success: true,
      data: updatedCustomer
    });
  } catch (error) {
    console.error('Error updating customer:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * DELETE /api/customers/:id
 * Delete customer
 */
router.delete('/:id', async (req, res) => {
  try {
    const { id } = req.params;

    // Check if customer exists
    const existingCustomer = await getCustomer(id);
    if (!existingCustomer) {
      return res.status(404).json({
        success: false,
        error: 'Customer not found'
      });
    }

    await deleteCustomer(id);

    res.json({
      success: true,
      message: 'Customer deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting customer:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/customers
 * List customers with filters and pagination
 *
 * Query params:
 * - type: 'individual' | 'corporate' | 'agency'
 * - status: 'active' | 'inactive'
 * - isVip: 'true' | 'false'
 * - search: string (searches in name, email, phone, company)
 * - limit: number (default: 20, max: 100)
 * - offset: number (default: 0)
 */
router.get('/', async (req, res) => {
  try {
    const { type, status, isVip, search, limit, offset } = req.query;

    // Build filters object
    const filters = {};
    if (type) filters.type = type;
    if (status) filters.status = status;
    if (isVip !== undefined) filters.isVip = isVip === 'true';
    if (search) filters.search = search;

    // Build pagination object
    const pagination = {
      limit: Math.min(parseInt(limit) || 20, 100),
      offset: parseInt(offset) || 0
    };

    // Get customers
    const result = await listCustomers(filters, pagination);

    res.json({
      success: true,
      data: result.customers,
      total: result.total,
      limit: pagination.limit,
      offset: pagination.offset
    });
  } catch (error) {
    console.error('Error listing customers:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/customers/search
 * Search customers by query
 *
 * Query params:
 * - q: string (required) - search query
 * - limit: number (default: 20)
 */
router.get('/search', async (req, res) => {
  try {
    const { q, limit } = req.query;

    if (!q) {
      return res.status(400).json({
        success: false,
        error: 'Search query (q) is required'
      });
    }

    const results = await searchCustomers(q);

    // Apply limit if specified
    const limitValue = Math.min(parseInt(limit) || 20, 100);
    const limitedResults = results.slice(0, limitValue);

    res.json({
      success: true,
      data: limitedResults,
      total: results.length
    });
  } catch (error) {
    console.error('Error searching customers:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/customers/:id/stats
 * Get customer statistics
 *
 * Returns ticket counts, total spent, etc.
 */
router.get('/:id/stats', async (req, res) => {
  try {
    const { id } = req.params;

    // Check if customer exists
    const customer = await getCustomer(id);
    if (!customer) {
      return res.status(404).json({
        success: false,
        error: 'Customer not found'
      });
    }

    const stats = await getCustomerStats(id);

    res.json({
      success: true,
      data: stats
    });
  } catch (error) {
    console.error('Error getting customer stats:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// ============================================
// CORPORATE USER ROUTES (Nested under customers)
// ============================================

/**
 * POST /api/customers/:customerId/users
 * Create a corporate user for a customer
 *
 * Body:
 * - firstName: string (required)
 * - lastName: string (required)
 * - email: string (required)
 * - phone: string (required)
 * - departmentId: string (optional)
 * - role: string (optional)
 * - isActive: boolean (optional, default: true)
 */
router.post('/:customerId/users', async (req, res) => {
  try {
    const { customerId } = req.params;
    const { firstName, lastName, email, phone } = req.body;

    // Check if customer exists and is corporate/agency
    const customer = await getCustomer(customerId);
    if (!customer) {
      return res.status(404).json({
        success: false,
        error: 'Customer not found'
      });
    }

    if (customer.type !== 'corporate' && customer.type !== 'agency') {
      return res.status(400).json({
        success: false,
        error: 'Corporate users can only be created for corporate or agency customers'
      });
    }

    // Validation
    if (!firstName || !lastName || !email || !phone) {
      return res.status(400).json({
        success: false,
        error: 'firstName, lastName, email, and phone are required'
      });
    }

    // Create corporate user
    const userData = {
      ...req.body,
      customerId
    };
    const user = await createCorporateUser(userData);

    res.status(201).json({
      success: true,
      data: user
    });
  } catch (error) {
    console.error('Error creating corporate user:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/customers/:customerId/users
 * List corporate users for a customer
 *
 * Query params:
 * - status: 'active' | 'inactive'
 * - departmentId: string
 * - search: string
 * - limit: number (default: 20)
 * - offset: number (default: 0)
 */
router.get('/:customerId/users', async (req, res) => {
  try {
    const { customerId } = req.params;
    const { status, departmentId, search, limit, offset } = req.query;

    // Check if customer exists
    const customer = await getCustomer(customerId);
    if (!customer) {
      return res.status(404).json({
        success: false,
        error: 'Customer not found'
      });
    }

    // Build filters object
    const filters = { customerId };
    if (status) filters.status = status;
    if (departmentId) filters.departmentId = departmentId;
    if (search) filters.search = search;

    // Build pagination object
    const pagination = {
      limit: Math.min(parseInt(limit) || 20, 100),
      offset: parseInt(offset) || 0
    };

    // Get corporate users
    const result = await listCorporateUsers(filters, pagination);

    res.json({
      success: true,
      data: result.users,
      total: result.total,
      limit: pagination.limit,
      offset: pagination.offset
    });
  } catch (error) {
    console.error('Error listing corporate users:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/customers/:customerId/users/:id
 * Get corporate user details
 */
router.get('/:customerId/users/:id', async (req, res) => {
  try {
    const { customerId, id } = req.params;

    // Check if customer exists
    const customer = await getCustomer(customerId);
    if (!customer) {
      return res.status(404).json({
        success: false,
        error: 'Customer not found'
      });
    }

    // Get corporate user
    const user = await getCorporateUser(id);

    if (!user) {
      return res.status(404).json({
        success: false,
        error: 'Corporate user not found'
      });
    }

    // Verify user belongs to customer
    if (user.customer_id !== customerId) {
      return res.status(404).json({
        success: false,
        error: 'Corporate user not found for this customer'
      });
    }

    res.json({
      success: true,
      data: user
    });
  } catch (error) {
    console.error('Error getting corporate user:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * PUT /api/customers/:customerId/users/:id
 * Update corporate user
 *
 * Body: Same as POST, all fields optional
 */
router.put('/:customerId/users/:id', async (req, res) => {
  try {
    const { customerId, id } = req.params;

    // Check if customer exists
    const customer = await getCustomer(customerId);
    if (!customer) {
      return res.status(404).json({
        success: false,
        error: 'Customer not found'
      });
    }

    // Check if user exists
    const existingUser = await getCorporateUser(id);
    if (!existingUser) {
      return res.status(404).json({
        success: false,
        error: 'Corporate user not found'
      });
    }

    // Verify user belongs to customer
    if (existingUser.customer_id !== customerId) {
      return res.status(404).json({
        success: false,
        error: 'Corporate user not found for this customer'
      });
    }

    // Update corporate user
    const updatedUser = await updateCorporateUser(id, req.body);

    res.json({
      success: true,
      data: updatedUser
    });
  } catch (error) {
    console.error('Error updating corporate user:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * DELETE /api/customers/:customerId/users/:id
 * Delete corporate user
 */
router.delete('/:customerId/users/:id', async (req, res) => {
  try {
    const { customerId, id } = req.params;

    // Check if customer exists
    const customer = await getCustomer(customerId);
    if (!customer) {
      return res.status(404).json({
        success: false,
        error: 'Customer not found'
      });
    }

    // Check if user exists
    const existingUser = await getCorporateUser(id);
    if (!existingUser) {
      return res.status(404).json({
        success: false,
        error: 'Corporate user not found'
      });
    }

    // Verify user belongs to customer
    if (existingUser.customer_id !== customerId) {
      return res.status(404).json({
        success: false,
        error: 'Corporate user not found for this customer'
      });
    }

    await deleteCorporateUser(id);

    res.json({
      success: true,
      message: 'Corporate user deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting corporate user:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

module.exports = router;
