const express = require('express');
const router = express.Router();
const db = require('../services/db');
const crypto = require('crypto');

// Generate UUID
function generateId() {
  return crypto.randomUUID();
}

// GET /api/departments - Tüm departmanları listele
router.get('/', async (req, res) => {
  try {
    const departments = await db.all(`
      SELECT
        d.*,
        COUNT(DISTINCT au.id) as user_count,
        au2.first_name as manager_first_name,
        au2.last_name as manager_last_name
      FROM departments d
      LEFT JOIN agency_users au ON au.department_id = d.id AND au.is_active = 1
      LEFT JOIN agency_users au2 ON au2.id = d.manager_id
      WHERE d.is_active = 1
      GROUP BY d.id
      ORDER BY d.name
    `);

    res.json(departments);
  } catch (error) {
    console.error('Error fetching departments:', error);
    res.status(500).json({ error: 'Failed to fetch departments' });
  }
});

// GET /api/departments/:id - Tek departman detayı
router.get('/:id', async (req, res) => {
  try {
    const department = await db.get(`
      SELECT
        d.*,
        COUNT(DISTINCT au.id) as user_count
      FROM departments d
      LEFT JOIN agency_users au ON au.department_id = d.id AND au.is_active = 1
      WHERE d.id = ? AND d.is_active = 1
      GROUP BY d.id
    `, [req.params.id]);

    if (!department) {
      return res.status(404).json({ error: 'Department not found' });
    }

    // Get users in this department
    const users = await db.all(`
      SELECT id, first_name, last_name, email, role, phone, avatar
      FROM agency_users
      WHERE department_id = ? AND is_active = 1
      ORDER BY first_name, last_name
    `, [req.params.id]);

    res.json({ ...department, users });
  } catch (error) {
    console.error('Error fetching department:', error);
    res.status(500).json({ error: 'Failed to fetch department' });
  }
});

// POST /api/departments - Yeni departman oluştur
router.post('/', async (req, res) => {
  try {
    const {
      name,
      description,
      color,
      parent_department_id,
      manager_id,
      permissions
    } = req.body;

    // Validation
    if (!name || !color) {
      return res.status(400).json({ error: 'Name and color are required' });
    }

    const id = generateId();

    // Convert permissions array to JSON string
    const permissionsJson = permissions ? JSON.stringify(permissions) : null;

    await db.run(`
      INSERT INTO departments (
        id, name, description, color, parent_department_id, manager_id, permissions, is_active
      ) VALUES (?, ?, ?, ?, ?, ?, ?, 1)
    `, [id, name, description || null, color, parent_department_id || null, manager_id || null, permissionsJson]);

    const department = await db.get('SELECT * FROM departments WHERE id = ?', [id]);

    res.status(201).json(department);
  } catch (error) {
    console.error('Error creating department:', error);
    res.status(500).json({ error: 'Failed to create department' });
  }
});

// PUT /api/departments/:id - Departman güncelle
router.put('/:id', async (req, res) => {
  try {
    const {
      name,
      description,
      color,
      parent_department_id,
      manager_id,
      permissions,
      is_active
    } = req.body;

    // Build dynamic UPDATE query for partial updates
    const updates = [];
    const params = [];

    if (name !== undefined) {
      updates.push('name = ?');
      params.push(name);
    }
    if (description !== undefined) {
      updates.push('description = ?');
      params.push(description || null);
    }
    if (color !== undefined) {
      updates.push('color = ?');
      params.push(color);
    }
    if (parent_department_id !== undefined) {
      updates.push('parent_department_id = ?');
      params.push(parent_department_id || null);
    }
    if (manager_id !== undefined) {
      updates.push('manager_id = ?');
      params.push(manager_id || null);
    }
    if (permissions !== undefined) {
      updates.push('permissions = ?');
      params.push(permissions ? JSON.stringify(permissions) : null);
    }
    if (is_active !== undefined) {
      updates.push('is_active = ?');
      params.push(is_active ? 1 : 0);
    }

    // Always update timestamp
    updates.push('updated_at = CURRENT_TIMESTAMP');

    // Add department ID for WHERE clause
    params.push(req.params.id);

    if (updates.length === 1) {
      return res.status(400).json({ error: 'No fields to update' });
    }

    await db.run(`
      UPDATE departments
      SET ${updates.join(', ')}
      WHERE id = ?
    `, params);

    // If permissions were updated, cascade to all users in this department
    if (permissions !== undefined) {
      const permissionsJson = permissions ? JSON.stringify(permissions) : null;
      await db.run(`
        UPDATE agency_users
        SET permissions = ?, updated_at = CURRENT_TIMESTAMP
        WHERE department_id = ? AND is_active = 1
      `, [permissionsJson, req.params.id]);

      console.log(`Updated permissions for all users in department ${req.params.id}`);
    }

    const department = await db.get('SELECT * FROM departments WHERE id = ?', [req.params.id]);

    if (!department) {
      return res.status(404).json({ error: 'Department not found' });
    }

    res.json(department);
  } catch (error) {
    console.error('Error updating department:', error);
    res.status(500).json({ error: 'Failed to update department' });
  }
});

// DELETE /api/departments/:id - Departman sil (soft delete)
router.delete('/:id', async (req, res) => {
  try {
    // Check if department has users
    const userCount = await db.get(
      'SELECT COUNT(*) as count FROM agency_users WHERE department_id = ? AND is_active = 1',
      [req.params.id]
    );

    if (userCount.count > 0) {
      return res.status(400).json({
        error: 'Cannot delete department with active users. Please reassign users first.'
      });
    }

    await db.run(
      'UPDATE departments SET is_active = 0, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
      [req.params.id]
    );

    res.json({ message: 'Department deleted successfully' });
  } catch (error) {
    console.error('Error deleting department:', error);
    res.status(500).json({ error: 'Failed to delete department' });
  }
});

// GET /api/departments/stats - Departman istatistikleri
router.get('/stats/overview', async (req, res) => {
  try {
    const stats = await db.get(`
      SELECT
        COUNT(DISTINCT d.id) as total_departments,
        COUNT(DISTINCT au.id) as total_users,
        COUNT(DISTINCT CASE WHEN d.parent_department_id IS NULL THEN d.id END) as top_level_departments
      FROM departments d
      LEFT JOIN agency_users au ON au.department_id = d.id AND au.is_active = 1
      WHERE d.is_active = 1
    `);

    res.json(stats);
  } catch (error) {
    console.error('Error fetching department stats:', error);
    res.status(500).json({ error: 'Failed to fetch department stats' });
  }
});

module.exports = router;
