const express = require('express');
const router = express.Router();
const db = require('../services/database');

// Validation helpers
const validateTicketRequestData = (data) => {
  const errors = [];

  if (!data.customerId) errors.push('customerId is required');
  if (!data.airline) errors.push('airline is required');
  if (!data.departureAirport) errors.push('departureAirport is required');
  if (!data.arrivalAirport) errors.push('arrivalAirport is required');
  if (!data.departureDate) errors.push('departureDate is required');
  if (!data.status) errors.push('status is required');
  if (!data.requestDate) errors.push('requestDate is required');

  // Validate status
  const validStatuses = ['pending', 'confirmed', 'ticketed', 'cancelled', 'rejected'];
  if (data.status && !validStatuses.includes(data.status)) {
    errors.push(`status must be one of: ${validStatuses.join(', ')}`);
  }

  // Validate airline
  const validAirlines = ['THY', 'PC', 'AJET', 'OTHER'];
  if (data.airline && !validAirlines.includes(data.airline)) {
    errors.push(`airline must be one of: ${validAirlines.join(', ')}`);
  }

  // Validate passengers if provided
  if (data.passengers && Array.isArray(data.passengers)) {
    if (data.passengers.length === 0) {
      errors.push('At least one passenger is required');
    }
    data.passengers.forEach((passenger, index) => {
      if (!passenger.firstName) errors.push(`Passenger ${index + 1}: firstName is required`);
      if (!passenger.lastName) errors.push(`Passenger ${index + 1}: lastName is required`);
    });
  }

  return errors;
};

const validatePnr = (pnr) => {
  if (!pnr) return 'PNR is required';
  if (!/^[A-Z0-9]{6}$/.test(pnr.toUpperCase())) {
    return 'PNR must be 6 alphanumeric characters';
  }
  return null;
};

// POST /api/ticket-requests - Create new ticket request
router.post('/', async (req, res) => {
  try {
    const errors = validateTicketRequestData(req.body);
    if (errors.length > 0) {
      return res.status(400).json({
        error: 'Validation failed',
        details: errors
      });
    }

    const { passengers, ...ticketData } = req.body;

    // Create ticket request
    const ticket = await db.createTicketRequest(ticketData);

    // Add passengers if provided
    if (passengers && Array.isArray(passengers) && passengers.length > 0) {
      for (const passengerData of passengers) {
        // Check if passenger already exists
        let passenger = await db.getPassenger(passengerData.id);

        if (!passenger) {
          // Create new passenger
          passenger = await db.createPassenger(passengerData);
        }

        // Link passenger to ticket
        await db.addPassengerToTicket(ticket.id, passenger.id);
      }

      // Fetch the complete ticket with passengers
      const completeTicket = await db.getTicketWithPassengers(ticket.id);

      return res.status(201).json({
        success: true,
        message: 'Ticket request created successfully',
        data: completeTicket
      });
    }

    res.status(201).json({
      success: true,
      message: 'Ticket request created successfully',
      data: ticket
    });

  } catch (error) {
    console.error('Error creating ticket request:', error);

    if (error.message && error.message.includes('UNIQUE constraint')) {
      return res.status(409).json({
        error: 'Duplicate ticket request',
        message: 'A ticket request with this information already exists'
      });
    }

    res.status(500).json({
      error: 'Internal server error',
      message: process.env.NODE_ENV === 'development' ? error.message : 'Failed to create ticket request'
    });
  }
});

// GET /api/ticket-requests - List ticket requests with filters and pagination
router.get('/', async (req, res) => {
  try {
    const {
      customerId,
      status,
      airline,
      departureDate,
      search,
      limit = 20,
      offset = 0
    } = req.query;

    const filters = {};
    if (customerId) filters.customerId = customerId;
    if (status) filters.status = status;
    if (airline) filters.airline = airline;
    if (departureDate) filters.dateFrom = departureDate;

    const pagination = {
      limit: parseInt(limit, 10),
      offset: parseInt(offset, 10)
    };

    // Get filtered tickets
    let tickets = await db.listTicketRequests(filters, pagination);

    // Apply search filter if provided (PNR or passenger name)
    if (search) {
      const searchLower = search.toLowerCase();

      // For each ticket, check PNR or fetch passengers to search by name
      const filteredTickets = [];
      for (const ticket of tickets) {
        // Check PNR match
        if (ticket.pnr && ticket.pnr.toLowerCase().includes(searchLower)) {
          filteredTickets.push(ticket);
          continue;
        }

        // Check passenger names
        const ticketWithPassengers = await db.getTicketWithPassengers(ticket.id);
        if (ticketWithPassengers && ticketWithPassengers.passengers) {
          const hasMatchingPassenger = ticketWithPassengers.passengers.some(passenger =>
            passenger.firstName.toLowerCase().includes(searchLower) ||
            passenger.lastName.toLowerCase().includes(searchLower)
          );

          if (hasMatchingPassenger) {
            filteredTickets.push(ticket);
          }
        }
      }

      tickets = filteredTickets;
    }

    res.json({
      success: true,
      data: tickets,
      pagination: {
        limit: pagination.limit,
        offset: pagination.offset,
        total: tickets.length
      }
    });

  } catch (error) {
    console.error('Error listing ticket requests:', error);
    res.status(500).json({
      error: 'Internal server error',
      message: process.env.NODE_ENV === 'development' ? error.message : 'Failed to list ticket requests'
    });
  }
});

// GET /api/ticket-requests/upcoming - Get upcoming tickets
router.get('/upcoming', async (req, res) => {
  try {
    const upcomingTickets = await db.getUpcomingTickets();

    // Enrich with passenger information
    const enrichedTickets = await Promise.all(
      upcomingTickets.map(async (ticket) => {
        return await db.getTicketWithPassengers(ticket.id);
      })
    );

    res.json({
      success: true,
      data: enrichedTickets,
      count: enrichedTickets.length
    });

  } catch (error) {
    console.error('Error fetching upcoming tickets:', error);
    res.status(500).json({
      error: 'Internal server error',
      message: process.env.NODE_ENV === 'development' ? error.message : 'Failed to fetch upcoming tickets'
    });
  }
});

// GET /api/ticket-requests/:id - Get ticket details with passengers
router.get('/:id', async (req, res) => {
  try {
    const { id } = req.params;

    const ticket = await db.getTicketWithPassengers(id);

    if (!ticket) {
      return res.status(404).json({
        error: 'Not found',
        message: 'Ticket request not found'
      });
    }

    res.json({
      success: true,
      data: ticket
    });

  } catch (error) {
    console.error('Error fetching ticket request:', error);
    res.status(500).json({
      error: 'Internal server error',
      message: process.env.NODE_ENV === 'development' ? error.message : 'Failed to fetch ticket request'
    });
  }
});

// PUT /api/ticket-requests/:id - Update ticket request
router.put('/:id', async (req, res) => {
  try {
    const { id } = req.params;

    // Check if ticket exists
    const existingTicket = await db.getTicketRequest(id);
    if (!existingTicket) {
      return res.status(404).json({
        error: 'Not found',
        message: 'Ticket request not found'
      });
    }

    // Validate update data
    const allowedUpdates = [
      'departureAirport',
      'arrivalAirport',
      'departureDate',
      'departureTime',
      'status',
      'notes',
      'airline',
      'flightNumber',
      'ticketPrice',
      'commission',
      'paymentStatus',
      'paymentMethod'
    ];

    const updates = {};
    Object.keys(req.body).forEach(key => {
      if (allowedUpdates.includes(key)) {
        updates[key] = req.body[key];
      }
    });

    if (Object.keys(updates).length === 0) {
      return res.status(400).json({
        error: 'Bad request',
        message: 'No valid update fields provided'
      });
    }

    // Validate status if being updated
    if (updates.status) {
      const validStatuses = ['pending', 'confirmed', 'ticketed', 'cancelled', 'rejected'];
      if (!validStatuses.includes(updates.status)) {
        return res.status(400).json({
          error: 'Validation failed',
          message: `status must be one of: ${validStatuses.join(', ')}`
        });
      }
    }

    // Update ticket
    await db.updateTicketRequest(id, updates);

    // Fetch updated ticket with passengers
    const updatedTicket = await db.getTicketWithPassengers(id);

    // Emit real-time update via Socket.IO
    if (req.io) {
      req.io.emit('ticket_updated', {
        ticketId: id,
        updates,
        timestamp: new Date().toISOString()
      });
    }

    res.json({
      success: true,
      message: 'Ticket request updated successfully',
      data: updatedTicket
    });

  } catch (error) {
    console.error('Error updating ticket request:', error);
    res.status(500).json({
      error: 'Internal server error',
      message: process.env.NODE_ENV === 'development' ? error.message : 'Failed to update ticket request'
    });
  }
});

// DELETE /api/ticket-requests/:id - Cancel ticket request
router.delete('/:id', async (req, res) => {
  try {
    const { id } = req.params;

    // Check if ticket exists
    const existingTicket = await db.getTicketRequest(id);
    if (!existingTicket) {
      return res.status(404).json({
        error: 'Not found',
        message: 'Ticket request not found'
      });
    }

    // Soft delete - set status to cancelled
    await db.deleteTicketRequest(id);

    // Emit real-time update via Socket.IO
    if (req.io) {
      req.io.emit('ticket_cancelled', {
        ticketId: id,
        timestamp: new Date().toISOString()
      });
    }

    res.json({
      success: true,
      message: 'Ticket request cancelled successfully',
      data: { id, status: 'cancelled' }
    });

  } catch (error) {
    console.error('Error cancelling ticket request:', error);
    res.status(500).json({
      error: 'Internal server error',
      message: process.env.NODE_ENV === 'development' ? error.message : 'Failed to cancel ticket request'
    });
  }
});

// POST /api/ticket-requests/:id/assign-pnr - Assign PNR to ticket
router.post('/:id/assign-pnr', async (req, res) => {
  try {
    const { id } = req.params;
    const { pnr } = req.body;

    // Validate PNR
    const pnrError = validatePnr(pnr);
    if (pnrError) {
      return res.status(400).json({
        error: 'Validation failed',
        message: pnrError
      });
    }

    // Check if ticket exists
    const existingTicket = await db.getTicketRequest(id);
    if (!existingTicket) {
      return res.status(404).json({
        error: 'Not found',
        message: 'Ticket request not found'
      });
    }

    // Check if PNR is already assigned to another ticket
    const tickets = await db.listTicketRequests({}, { limit: 1000, offset: 0 });
    const duplicatePnr = tickets.find(t =>
      t.pnr &&
      t.pnr.toUpperCase() === pnr.toUpperCase() &&
      t.id !== id
    );

    if (duplicatePnr) {
      return res.status(409).json({
        error: 'Conflict',
        message: 'This PNR is already assigned to another ticket'
      });
    }

    // Assign PNR and update status to 'ticketed'
    await db.assignPnrToTicket(id, pnr.toUpperCase());

    // Fetch updated ticket with passengers
    const updatedTicket = await db.getTicketWithPassengers(id);

    // Emit real-time update via Socket.IO
    if (req.io) {
      req.io.emit('pnr_assigned', {
        ticketId: id,
        pnr: pnr.toUpperCase(),
        timestamp: new Date().toISOString()
      });
    }

    res.json({
      success: true,
      message: 'PNR assigned successfully',
      data: updatedTicket
    });

  } catch (error) {
    console.error('Error assigning PNR:', error);
    res.status(500).json({
      error: 'Internal server error',
      message: process.env.NODE_ENV === 'development' ? error.message : 'Failed to assign PNR'
    });
  }
});

// POST /api/ticket-requests/:ticketId/passengers - Add passenger to ticket
router.post('/:ticketId/passengers', async (req, res) => {
  try {
    const { ticketId } = req.params;
    const passengerData = req.body;

    // Validate passenger data
    if (!passengerData.firstName || !passengerData.lastName) {
      return res.status(400).json({
        error: 'Validation failed',
        message: 'firstName and lastName are required'
      });
    }

    // Check if ticket exists
    const ticket = await db.getTicketRequest(ticketId);
    if (!ticket) {
      return res.status(404).json({
        error: 'Not found',
        message: 'Ticket request not found'
      });
    }

    let passenger;

    // If passenger ID is provided, use existing passenger
    if (passengerData.id) {
      passenger = await db.getPassenger(passengerData.id);
      if (!passenger) {
        return res.status(404).json({
          error: 'Not found',
          message: 'Passenger not found'
        });
      }
    } else {
      // Create new passenger
      passenger = await db.createPassenger(passengerData);
    }

    // Link passenger to ticket
    await db.addPassengerToTicket(ticketId, passenger.id);

    // Fetch updated ticket with all passengers
    const updatedTicket = await db.getTicketWithPassengers(ticketId);

    // Emit real-time update via Socket.IO
    if (req.io) {
      req.io.emit('passenger_added', {
        ticketId,
        passengerId: passenger.id,
        timestamp: new Date().toISOString()
      });
    }

    res.status(201).json({
      success: true,
      message: 'Passenger added to ticket successfully',
      data: {
        ticket: updatedTicket,
        passenger
      }
    });

  } catch (error) {
    console.error('Error adding passenger to ticket:', error);

    if (error.message && error.message.includes('UNIQUE constraint')) {
      return res.status(409).json({
        error: 'Conflict',
        message: 'This passenger is already added to this ticket'
      });
    }

    res.status(500).json({
      error: 'Internal server error',
      message: process.env.NODE_ENV === 'development' ? error.message : 'Failed to add passenger to ticket'
    });
  }
});

// DELETE /api/ticket-requests/:ticketId/passengers/:passengerId - Remove passenger from ticket
router.delete('/:ticketId/passengers/:passengerId', async (req, res) => {
  try {
    const { ticketId, passengerId } = req.params;

    // Check if ticket exists
    const ticket = await db.getTicketRequest(ticketId);
    if (!ticket) {
      return res.status(404).json({
        error: 'Not found',
        message: 'Ticket request not found'
      });
    }

    // Check if passenger exists
    const passenger = await db.getPassenger(passengerId);
    if (!passenger) {
      return res.status(404).json({
        error: 'Not found',
        message: 'Passenger not found'
      });
    }

    // Remove passenger from ticket
    const removed = await db.removePassengerFromTicket(ticketId, passengerId);

    if (!removed) {
      return res.status(404).json({
        error: 'Not found',
        message: 'Passenger is not associated with this ticket'
      });
    }

    // Fetch updated ticket with remaining passengers
    const updatedTicket = await db.getTicketWithPassengers(ticketId);

    // Emit real-time update via Socket.IO
    if (req.io) {
      req.io.emit('passenger_removed', {
        ticketId,
        passengerId,
        timestamp: new Date().toISOString()
      });
    }

    res.json({
      success: true,
      message: 'Passenger removed from ticket successfully',
      data: updatedTicket
    });

  } catch (error) {
    console.error('Error removing passenger from ticket:', error);
    res.status(500).json({
      error: 'Internal server error',
      message: process.env.NODE_ENV === 'development' ? error.message : 'Failed to remove passenger from ticket'
    });
  }
});

module.exports = router;
