const AmadeusService = require('./AmadeusService');

class FlightStatusService {
  constructor() {
    this.flightCache = new Map();
    this.cacheTimeout = 30 * 60 * 1000; // 30 minutes cache
    this.amadeusService = new AmadeusService();
    this.useAmadeus = true; // Flag to switch between Amadeus and mock data
  }

  // Check if check-in is open for a flight
  isCheckinOpen(departureDate, departureTime) {
    const now = new Date();
    const departure = new Date(`${departureDate}T${departureTime}`);
    const checkinOpenTime = new Date(departure.getTime() - 24 * 60 * 60 * 1000); // 24 hours before

    return {
      isOpen: now >= checkinOpenTime && now < departure,
      checkinOpenTime,
      departure,
      timeUntilOpen: now < checkinOpenTime ? checkinOpenTime.getTime() - now.getTime() : 0,
      timeUntilClosure: now < departure ? departure.getTime() - now.getTime() : 0
    };
  }

  // Get optimal check-in time (just after check-in opens)
  getOptimalCheckinTime(departureDate, departureTime) {
    const departure = new Date(`${departureDate}T${departureTime}`);
    const checkinOpenTime = new Date(departure.getTime() - 24 * 60 * 60 * 1000);
    
    // Schedule check-in 30 seconds after it opens to ensure system is ready
    return new Date(checkinOpenTime.getTime() + 30 * 1000);
  }

  // Determine current check-in status for a ticket
  determineCheckinStatus(ticket) {
    const now = new Date();
    const departure = new Date(`${ticket.departureDate}T${ticket.departureTime}`);
    const checkinOpenTime = new Date(departure.getTime() - 24 * 60 * 60 * 1000);

    // If check-in was already completed or failed
    if (ticket.checkinStatus === 'completed' || ticket.checkinStatus === 'failed') {
      return ticket.checkinStatus;
    }

    // If currently processing
    if (ticket.checkinStatus === 'processing') {
      return 'processing';
    }

    // If flight has already departed
    if (now >= departure) {
      return 'expired';
    }

    // If check-in is open
    if (now >= checkinOpenTime) {
      return 'ready';
    }

    // If waiting for check-in to open
    return 'waiting';
  }

  // Get flight status information (Amadeus + fallback to mock)
  async getFlightStatus(airline, flightNumber, departureDate) {
    const cacheKey = `${airline}-${flightNumber}-${departureDate}`;
    const cached = this.flightCache.get(cacheKey);

    if (cached && Date.now() - cached.timestamp < this.cacheTimeout) {
      return cached.data;
    }

    let flightStatus;

    // Try Amadeus API first
    if (this.useAmadeus) {
      try {
        const amadeusResult = await this.amadeusService.getFlightStatus(
          airline,
          flightNumber,
          departureDate
        );

        if (amadeusResult.success) {
          flightStatus = this.formatAmadeusFlightStatus(amadeusResult.flight);
          console.log(`✅ Flight status from Amadeus: ${airline}${flightNumber}`);
        } else {
          console.log(`⚠️ Amadeus flight status failed, using mock: ${amadeusResult.message}`);
          flightStatus = await this.mockFlightStatusApi(airline, flightNumber, departureDate);
        }
      } catch (error) {
        console.error(`❌ Amadeus flight status error: ${error.message}`);
        flightStatus = await this.mockFlightStatusApi(airline, flightNumber, departureDate);
      }
    } else {
      // Use mock data
      flightStatus = await this.mockFlightStatusApi(airline, flightNumber, departureDate);
    }

    this.flightCache.set(cacheKey, {
      data: flightStatus,
      timestamp: Date.now()
    });

    return flightStatus;
  }

  // Mock flight status API (replace with real airline API calls)
  async mockFlightStatusApi(airline, flightNumber, departureDate) {
    // Simulate API delay
    await new Promise(resolve => setTimeout(resolve, Math.random() * 1000));

    // Mock flight status
    const statuses = ['on_time', 'delayed', 'cancelled', 'boarding', 'departed'];
    const randomStatus = statuses[Math.floor(Math.random() * statuses.length)];

    const baseDelay = Math.floor(Math.random() * 120); // 0-120 minutes delay

    return {
      airline,
      flightNumber,
      departureDate,
      status: randomStatus,
      scheduledDeparture: `${departureDate}T${this.getRandomTime()}`,
      estimatedDeparture: randomStatus === 'delayed' 
        ? this.addMinutes(`${departureDate}T${this.getRandomTime()}`, baseDelay)
        : `${departureDate}T${this.getRandomTime()}`,
      gate: randomStatus !== 'cancelled' ? this.getRandomGate() : null,
      terminal: randomStatus !== 'cancelled' ? this.getRandomTerminal() : null,
      checkinStatus: this.getCheckinStatusFromFlightStatus(randomStatus),
      delay: randomStatus === 'delayed' ? baseDelay : 0,
      reason: randomStatus === 'delayed' ? 'Weather conditions' : 
              randomStatus === 'cancelled' ? 'Technical issues' : null,
      lastUpdated: new Date().toISOString()
    };
  }

  // Helper methods for mock data
  getRandomTime() {
    const hours = String(Math.floor(Math.random() * 24)).padStart(2, '0');
    const minutes = String(Math.floor(Math.random() * 60)).padStart(2, '0');
    return `${hours}:${minutes}`;
  }

  getRandomGate() {
    const gates = ['A1', 'A2', 'A3', 'B1', 'B2', 'B3', 'C1', 'C2', 'C3', 'D1', 'D2'];
    return gates[Math.floor(Math.random() * gates.length)];
  }

  getRandomTerminal() {
    const terminals = ['1', '2', '3', 'International', 'Domestic'];
    return terminals[Math.floor(Math.random() * terminals.length)];
  }

  getCheckinStatusFromFlightStatus(flightStatus) {
    switch (flightStatus) {
      case 'cancelled':
        return 'unavailable';
      case 'departed':
        return 'closed';
      case 'boarding':
        return 'closing_soon';
      default:
        return 'open';
    }
  }

  addMinutes(dateTimeString, minutes) {
    const date = new Date(dateTimeString);
    date.setMinutes(date.getMinutes() + minutes);
    return date.toISOString().slice(0, 16); // Return in YYYY-MM-DDTHH:MM format
  }

  // Update ticket status based on flight status
  async updateTicketWithFlightStatus(ticket) {
    try {
      const flightStatus = await this.getFlightStatus(
        ticket.airline,
        ticket.flightNumber,
        ticket.departureDate.toISOString().split('T')[0]
      );

      // Update ticket with flight information
      const updatedTicket = {
        ...ticket,
        flightStatus: flightStatus.status,
        estimatedDeparture: flightStatus.estimatedDeparture,
        gate: flightStatus.gate,
        terminal: flightStatus.terminal,
        delay: flightStatus.delay,
        delayReason: flightStatus.reason,
        lastFlightUpdate: new Date(flightStatus.lastUpdated),
        updatedAt: new Date()
      };

      // Adjust check-in status based on flight status
      if (flightStatus.status === 'cancelled') {
        updatedTicket.checkinStatus = 'cancelled';
        updatedTicket.errorMessage = 'Flight cancelled';
      } else if (flightStatus.status === 'delayed' && flightStatus.delay > 0) {
        // Recalculate check-in timing if flight is significantly delayed
        const newDeparture = new Date(flightStatus.estimatedDeparture);
        updatedTicket.departureDate = newDeparture;
        updatedTicket.departureTime = newDeparture.toTimeString().slice(0, 5);
        
        // Recalculate check-in open time
        updatedTicket.checkinOpenTime = new Date(newDeparture.getTime() - 24 * 60 * 60 * 1000);
        
        // Update status based on new timing
        updatedTicket.checkinStatus = this.determineCheckinStatus(updatedTicket);
      }

      return updatedTicket;

    } catch (error) {
      console.error(`Error updating flight status for ${ticket.pnr}:`, error);
      return ticket; // Return unchanged ticket on error
    }
  }

  // Get check-in window information
  getCheckinWindow(departureDate, departureTime, airline = 'THY') {
    const departure = new Date(`${departureDate}T${departureTime}`);
    
    // Different airlines have different check-in windows
    const windows = {
      THY: { hours: 24, closeMinutes: 45 }, // THY: 24 hours before, closes 45 min before
      PC: { hours: 24, closeMinutes: 40 },  // Pegasus: 24 hours before, closes 40 min before
      OTHER: { hours: 24, closeMinutes: 60 } // Other airlines: closes 1 hour before
    };

    const window = windows[airline] || windows.OTHER;
    
    const openTime = new Date(departure.getTime() - window.hours * 60 * 60 * 1000);
    const closeTime = new Date(departure.getTime() - window.closeMinutes * 60 * 1000);

    const now = new Date();

    return {
      openTime,
      closeTime,
      departure,
      isOpen: now >= openTime && now < closeTime,
      isPast: now >= closeTime,
      isFuture: now < openTime,
      timeUntilOpen: now < openTime ? openTime.getTime() - now.getTime() : 0,
      timeUntilClose: now < closeTime ? closeTime.getTime() - now.getTime() : 0,
      windowDurationHours: window.hours,
      closeMinutesBeforeDeparture: window.closeMinutes
    };
  }

  // Format time remaining in human readable format
  formatTimeRemaining(milliseconds) {
    if (milliseconds <= 0) return 'Geçti';

    const seconds = Math.floor(milliseconds / 1000);
    const minutes = Math.floor(seconds / 60);
    const hours = Math.floor(minutes / 60);
    const days = Math.floor(hours / 24);

    if (days > 0) {
      return `${days} gün ${hours % 24} saat`;
    } else if (hours > 0) {
      return `${hours} saat ${minutes % 60} dakika`;
    } else if (minutes > 0) {
      return `${minutes} dakika`;
    } else {
      return `${seconds} saniye`;
    }
  }

  // Clear expired cache entries
  cleanCache() {
    const now = Date.now();
    for (const [key, value] of this.flightCache.entries()) {
      if (now - value.timestamp > this.cacheTimeout) {
        this.flightCache.delete(key);
      }
    }
  }

  // Get comprehensive ticket timing info
  getTicketTimingInfo(ticket) {
    const checkinWindow = this.getCheckinWindow(
      ticket.departureDate.toISOString().split('T')[0],
      ticket.departureTime,
      ticket.airline
    );

    const optimalTime = this.getOptimalCheckinTime(
      ticket.departureDate.toISOString().split('T')[0],
      ticket.departureTime
    );

    return {
      ...checkinWindow,
      optimalCheckinTime: optimalTime,
      currentStatus: this.determineCheckinStatus(ticket),
      recommendations: this.getCheckinRecommendations(ticket, checkinWindow)
    };
  }

  // Get check-in recommendations
  getCheckinRecommendations(ticket, window) {
    const now = new Date();
    const recommendations = [];

    if (window.isFuture) {
      const timeUntil = this.formatTimeRemaining(window.timeUntilOpen);
      recommendations.push(`Check-in ${timeUntil} sonra açılacak`);
    } else if (window.isOpen) {
      const timeLeft = this.formatTimeRemaining(window.timeUntilClose);
      recommendations.push(`Check-in ${timeLeft} sonra kapanacak`);
      
      if (ticket.checkinStatus === 'ready') {
        recommendations.push('Check-in için uygun zaman');
      }
    } else if (window.isPast) {
      recommendations.push('Check-in süresi geçti');
    }

    // VIP recommendations
    if (ticket.customer && ticket.customer.isVip) {
      recommendations.push('VIP müşteri - öncelik ver');
    }

    // Retry recommendations
    if (ticket.checkinAttempts > 0 && ticket.checkinStatus === 'failed') {
      recommendations.push(`${ticket.checkinAttempts} deneme yapıldı - manuel kontrol et`);
    }

    return recommendations;
  }

  // Format Amadeus flight status to our internal format
  formatAmadeusFlightStatus(amadeusFlightData) {
    const {
      carrierCode,
      flightNumber,
      scheduledDepartureDate,
      flightStatus,
      departure,
      arrival,
      aircraft,
      operating
    } = amadeusFlightData;

    // Map Amadeus status to our internal status
    const statusMapping = {
      'SCH': 'on_time',     // Scheduled
      'ACT': 'on_time',     // Active
      'DEL': 'delayed',     // Delayed
      'CNX': 'cancelled',   // Cancelled
      'ROU': 'diverted',    // Rerouted
      'UNK': 'unknown'      // Unknown
    };

    const mappedStatus = statusMapping[flightStatus] || 'unknown';

    // Calculate delay if applicable
    let delay = 0;
    let estimatedDeparture = departure?.scheduledTimeLocal;

    if (departure?.actualTimeLocal && departure?.scheduledTimeLocal) {
      const scheduled = new Date(departure.scheduledTimeLocal);
      const actual = new Date(departure.actualTimeLocal);
      delay = Math.max(0, Math.floor((actual - scheduled) / (1000 * 60))); // minutes
      estimatedDeparture = departure.actualTimeLocal;
    } else if (departure?.estimatedTimeLocal) {
      const scheduled = new Date(departure.scheduledTimeLocal);
      const estimated = new Date(departure.estimatedTimeLocal);
      delay = Math.max(0, Math.floor((estimated - scheduled) / (1000 * 60))); // minutes
      estimatedDeparture = departure.estimatedTimeLocal;
    }

    return {
      airline: carrierCode,
      flightNumber: flightNumber,
      departureDate: scheduledDepartureDate,
      status: mappedStatus,
      scheduledDeparture: departure?.scheduledTimeLocal,
      estimatedDeparture: estimatedDeparture,
      actualDeparture: departure?.actualTimeLocal,
      gate: departure?.gate,
      terminal: departure?.terminal,
      checkinStatus: this.getCheckinStatusFromFlightStatus(mappedStatus),
      delay: delay,
      reason: mappedStatus === 'delayed' ? 'Flight delayed' :
              mappedStatus === 'cancelled' ? 'Flight cancelled' : null,
      aircraft: aircraft ? `${aircraft.code}` : null,
      destination: {
        airport: arrival?.iataCode,
        scheduledTime: arrival?.scheduledTimeLocal,
        estimatedTime: arrival?.estimatedTimeLocal,
        actualTime: arrival?.actualTimeLocal,
        gate: arrival?.gate,
        terminal: arrival?.terminal
      },
      operating: operating ? {
        carrierCode: operating.carrierCode,
        flightNumber: operating.flightNumber
      } : null,
      lastUpdated: new Date().toISOString(),
      dataSource: 'amadeus'
    };
  }

  // Enable/disable Amadeus integration
  setAmadeusEnabled(enabled) {
    this.useAmadeus = enabled;
    console.log(`📡 Amadeus integration ${enabled ? 'enabled' : 'disabled'}`);
  }

  // Test Amadeus connection
  async testAmadeusConnection() {
    try {
      const result = await this.amadeusService.testConnection();
      console.log(`🔧 Amadeus connection test: ${result ? 'SUCCESS' : 'FAILED'}`);
      return result;
    } catch (error) {
      console.error('🔧 Amadeus connection test error:', error.message);
      return false;
    }
  }

  // Get check-in links using Amadeus
  async getCheckinLinks(airlineCode, language = 'en-GB') {
    if (!this.useAmadeus) {
      return { success: false, message: 'Amadeus integration disabled' };
    }

    try {
      const result = await this.amadeusService.getCheckinLinks(airlineCode, language);

      if (result.success && result.checkinLinks.length > 0) {
        return {
          success: true,
          links: result.checkinLinks.map(link => ({
            type: link.type,
            href: link.href,
            methods: link.methods || ['GET']
          }))
        };
      } else {
        return {
          success: false,
          message: 'No check-in links found for this airline'
        };
      }
    } catch (error) {
      console.error(`Error getting check-in links for ${airlineCode}:`, error);
      return {
        success: false,
        message: error.message
      };
    }
  }

  // Search airports using Amadeus
  async searchAirports(keyword) {
    if (!this.useAmadeus) {
      return { success: false, message: 'Amadeus integration disabled' };
    }

    try {
      const result = await this.amadeusService.getAirportInfo(keyword);

      if (result.success) {
        return {
          success: true,
          airports: result.airports.map(airport => ({
            iataCode: airport.iataCode,
            name: airport.name,
            city: airport.address?.cityName,
            country: airport.address?.countryName,
            timezone: airport.timeZoneOffset
          }))
        };
      } else {
        return {
          success: false,
          message: result.message
        };
      }
    } catch (error) {
      console.error(`Error searching airports for ${keyword}:`, error);
      return {
        success: false,
        message: error.message
      };
    }
  }
}

module.exports = FlightStatusService;