const axios = require('axios');
const xml2js = require('xml2js');

class PegasusNDCService {
  constructor() {
    this.baseURL = process.env.PEGASUS_NDC_URL || 'https://api.flypgs.com/ndc'; // Test URL
    this.credentials = {
      username: process.env.PEGASUS_USERNAME,
      password: process.env.PEGASUS_PASSWORD,
      agentId: process.env.PEGASUS_AGENT_ID
    };
    this.timeout = 30000; // 30 seconds
  }

  async performCheckin({ pnr, lastName, firstName, phone, flightType, passportNumber, visaNumber }) {
    const result = {
      success: false,
      message: '',
      seat: null,
      boardingPassUrl: null,
      flightInfo: null
    };

    try {
      console.log(`🚀 Starting Pegasus NDC check-in for PNR: ${pnr}, Passenger: ${lastName}`);

      // 1. OrderRetrieve - Rezervasyonu getir
      const orderData = await this.retrieveOrder(pnr, lastName);
      if (!orderData) {
        result.message = 'Rezervasyon bulunamadı';
        return result;
      }

      // 2. Check-in eligibility kontrolü
      if (!this.isCheckinEligible(orderData)) {
        result.message = 'Check-in henüz mevcut değil veya zaten tamamlanmış';
        return result;
      }

      // 3. OrderChange ile check-in işlemi
      const checkinResult = await this.executeCheckin(orderData, {
        pnr,
        lastName,
        firstName,
        phone,
        flightType,
        passportNumber,
        visaNumber
      });

      if (checkinResult.success) {
        result.success = true;
        result.message = 'Check-in başarılı';
        result.seat = checkinResult.seat;
        result.boardingPassUrl = checkinResult.boardingPassUrl;
        result.flightInfo = checkinResult.flightInfo;
      } else {
        result.message = checkinResult.message || 'Check-in başarısız';
      }

    } catch (error) {
      console.error('Pegasus NDC check-in error:', error);
      result.message = `Check-in hatası: ${error.message}`;
    }

    return result;
  }

  async retrieveOrder(pnr, lastName) {
    try {
      const requestXML = this.buildOrderRetrieveXML(pnr, lastName);
      
      const response = await axios.post(this.baseURL + '/OrderRetrieve', requestXML, {
        headers: {
          'Content-Type': 'application/xml',
          'Authorization': this.getAuthHeader()
        },
        timeout: this.timeout
      });

      const parsedResponse = await this.parseXMLResponse(response.data);
      
      if (parsedResponse.success) {
        return parsedResponse.orderData;
      } else {
        console.error('OrderRetrieve failed:', parsedResponse.error);
        return null;
      }

    } catch (error) {
      console.error('OrderRetrieve API error:', error);
      throw new Error('Rezervasyon sorgulanamadı');
    }
  }

  async executeCheckin(orderData, passengerInfo) {
    try {
      const requestXML = this.buildOrderChangeXML(orderData, passengerInfo);
      
      const response = await axios.post(this.baseURL + '/OrderChange', requestXML, {
        headers: {
          'Content-Type': 'application/xml',
          'Authorization': this.getAuthHeader()
        },
        timeout: this.timeout
      });

      const parsedResponse = await this.parseXMLResponse(response.data);
      
      if (parsedResponse.success) {
        return {
          success: true,
          seat: parsedResponse.seat,
          boardingPassUrl: parsedResponse.boardingPassUrl,
          flightInfo: parsedResponse.flightInfo
        };
      } else {
        return {
          success: false,
          message: parsedResponse.error || 'Check-in işlemi başarısız'
        };
      }

    } catch (error) {
      console.error('OrderChange API error:', error);
      throw new Error('Check-in işlemi gerçekleştirilemedi');
    }
  }

  buildOrderRetrieveXML(pnr, lastName) {
    return `<?xml version="1.0" encoding="UTF-8"?>
<soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/"
               xmlns:ndc="http://www.iata.org/IATA/EDIST/2024.1">
  <soap:Header>
    <ndc:MessageHeader>
      <ndc:MessageId>${this.generateMessageId()}</ndc:MessageId>
      <ndc:Timestamp>${new Date().toISOString()}</ndc:Timestamp>
      <ndc:Source>
        <ndc:AgentId>${this.credentials.agentId}</ndc:AgentId>
      </ndc:Source>
    </ndc:MessageHeader>
  </soap:Header>
  <soap:Body>
    <ndc:OrderRetrieveRQ>
      <ndc:Document>
        <ndc:Name>NDC OrderRetrieve</ndc:Name>
        <ndc:ReferenceVersion>2024.1</ndc:ReferenceVersion>
      </ndc:Document>
      <ndc:Party>
        <ndc:Sender>
          <ndc:AgentId>${this.credentials.agentId}</ndc:AgentId>
        </ndc:Sender>
      </ndc:Party>
      <ndc:Query>
        <ndc:Filters>
          <ndc:OrderFilter>
            <ndc:OrderId>${pnr}</ndc:OrderId>
            <ndc:PassengerFilter>
              <ndc:Surname>${lastName}</ndc:Surname>
            </ndc:PassengerFilter>
          </ndc:OrderFilter>
        </ndc:Filters>
      </ndc:Query>
    </ndc:OrderRetrieveRQ>
  </soap:Body>
</soap:Envelope>`;
  }

  buildOrderChangeXML(orderData, passengerInfo) {
    const { pnr, lastName, firstName, phone, flightType, passportNumber, visaNumber } = passengerInfo;
    
    // International flight için pasaport bilgilerini ekle
    const documentsSection = flightType === 'international' ? `
      <ndc:Documents>
        <ndc:Passport>
          <ndc:Number>${passportNumber}</ndc:Number>
        </ndc:Passport>
        ${visaNumber ? `<ndc:Visa><ndc:Number>${visaNumber}</ndc:Number></ndc:Visa>` : ''}
      </ndc:Documents>
    ` : '';

    return `<?xml version="1.0" encoding="UTF-8"?>
<soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/"
               xmlns:ndc="http://www.iata.org/IATA/EDIST/2024.1">
  <soap:Header>
    <ndc:MessageHeader>
      <ndc:MessageId>${this.generateMessageId()}</ndc:MessageId>
      <ndc:Timestamp>${new Date().toISOString()}</ndc:Timestamp>
      <ndc:Source>
        <ndc:AgentId>${this.credentials.agentId}</ndc:AgentId>
      </ndc:Source>
    </ndc:MessageHeader>
  </soap:Header>
  <soap:Body>
    <ndc:OrderChangeRQ>
      <ndc:Document>
        <ndc:Name>NDC OrderChange</ndc:Name>
        <ndc:ReferenceVersion>2024.1</ndc:ReferenceVersion>
      </ndc:Document>
      <ndc:Party>
        <ndc:Sender>
          <ndc:AgentId>${this.credentials.agentId}</ndc:AgentId>
        </ndc:Sender>
      </ndc:Party>
      <ndc:Query>
        <ndc:OrderId>${pnr}</ndc:OrderId>
        <ndc:OrderItems>
          <ndc:CheckInRequest>
            <ndc:Passenger>
              <ndc:GivenName>${firstName}</ndc:GivenName>
              <ndc:Surname>${lastName}</ndc:Surname>
              <ndc:ContactInfo>
                <ndc:Phone>${phone}</ndc:Phone>
              </ndc:ContactInfo>
              ${documentsSection}
            </ndc:Passenger>
          </ndc:CheckInRequest>
        </ndc:OrderItems>
      </ndc:Query>
    </ndc:OrderChangeRQ>
  </soap:Body>
</soap:Envelope>`;
  }

  async parseXMLResponse(xmlData) {
    try {
      const parser = new xml2js.Parser({ explicitArray: false });
      const result = await parser.parseStringPromise(xmlData);
      
      // Response parsing logic - NDC response yapısına göre
      if (result['soap:Envelope']) {
        const body = result['soap:Envelope']['soap:Body'];
        
        if (body.OrderRetrieveRS) {
          return this.parseOrderRetrieveResponse(body.OrderRetrieveRS);
        } else if (body.OrderChangeRS) {
          return this.parseOrderChangeResponse(body.OrderChangeRS);
        } else if (body['soap:Fault']) {
          return {
            success: false,
            error: body['soap:Fault'].faultstring || 'API Error'
          };
        }
      }
      
      return {
        success: false,
        error: 'Invalid response format'
      };
      
    } catch (error) {
      console.error('XML parsing error:', error);
      return {
        success: false,
        error: 'Response parsing failed'
      };
    }
  }

  parseOrderRetrieveResponse(response) {
    try {
      // NDC OrderRetrieve response parsing
      if (response.Success) {
        return {
          success: true,
          orderData: {
            orderId: response.Order?.OrderId,
            flights: response.Order?.OrderItems?.Flight || [],
            passengers: response.Order?.Passengers || [],
            status: response.Order?.Status
          }
        };
      } else {
        return {
          success: false,
          error: response.Errors?.Error?.Description || 'Order retrieve failed'
        };
      }
    } catch (error) {
      return {
        success: false,
        error: 'Failed to parse order retrieve response'
      };
    }
  }

  parseOrderChangeResponse(response) {
    try {
      // NDC OrderChange response parsing
      if (response.Success) {
        const checkinData = response.Order?.OrderItems?.CheckIn;
        return {
          success: true,
          seat: checkinData?.Seat?.Number || null,
          boardingPassUrl: checkinData?.BoardingPass?.URL || null,
          flightInfo: {
            flight: checkinData?.Flight?.FlightNumber,
            from: checkinData?.Flight?.DepartureAirport,
            to: checkinData?.Flight?.ArrivalAirport,
            date: checkinData?.Flight?.DepartureDate,
            gate: checkinData?.Flight?.Gate
          }
        };
      } else {
        return {
          success: false,
          error: response.Errors?.Error?.Description || 'Check-in failed'
        };
      }
    } catch (error) {
      return {
        success: false,
        error: 'Failed to parse check-in response'
      };
    }
  }

  isCheckinEligible(orderData) {
    // Check-in eligibility kontrolleri
    if (!orderData || !orderData.flights) {
      return false;
    }

    // Uçuş tarihinden 24 saat öncesi kontrolü
    const now = new Date();
    const flights = Array.isArray(orderData.flights) ? orderData.flights : [orderData.flights];
    
    return flights.some(flight => {
      const departureDate = new Date(flight.DepartureDate);
      const checkinOpenTime = new Date(departureDate.getTime() - 24 * 60 * 60 * 1000);
      
      return now >= checkinOpenTime && now < departureDate;
    });
  }

  getAuthHeader() {
    // Basic Authentication for Pegasus API
    const credentials = Buffer.from(`${this.credentials.username}:${this.credentials.password}`).toString('base64');
    return `Basic ${credentials}`;
  }

  generateMessageId() {
    return `MSG_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
  }

  async testConnection() {
    try {
      // Test endpoint'i çağır
      console.log('🧪 Testing Pegasus NDC API connection...');
      
      // Basit bir health check
      const response = await axios.get(this.baseURL + '/health', {
        headers: {
          'Authorization': this.getAuthHeader()
        },
        timeout: 10000
      });

      return {
        success: true,
        message: 'Pegasus NDC API connection successful'
      };
    } catch (error) {
      return {
        success: false,
        message: `Connection failed: ${error.message}`
      };
    }
  }
}

module.exports = PegasusNDCService;