const { parentPort, workerData } = require('worker_threads');
const SeleniumService = require('./SeleniumService');

// Worker thread for executing check-in operations
async function executeCheckin() {
  const { jobId, ticketData } = workerData;
  let seleniumService = null;

  try {
    console.log(`🤖 Worker ${jobId}: Starting check-in for ${ticketData.pnr}`);

    seleniumService = new SeleniumService();

    // Execute check-in with timeout protection
    const result = await Promise.race([
      seleniumService.performCheckin(ticketData),
      new Promise((_, reject) =>
        setTimeout(() => reject(new Error('Selenium operation timeout')), 25000)
      )
    ]);

    // Send result back to main thread
    parentPort.postMessage({
      success: result.success,
      message: result.message,
      seat: result.flightInfo?.seat,
      boardingPassUrl: result.boardingPassUrl,
      screenshots: result.screenshots
    });

    console.log(`✅ Worker ${jobId}: Check-in completed for ${ticketData.pnr}`);

  } catch (error) {
    console.error(`❌ Worker ${jobId}: Error for ${ticketData.pnr}:`, error.message);

    // Send error back to main thread
    parentPort.postMessage({
      success: false,
      message: error.message,
      error: error.stack
    });

  } finally {
    // Always cleanup Selenium resources
    if (seleniumService) {
      try {
        await seleniumService.close();
      } catch (cleanupError) {
        console.error(`⚠️ Worker ${jobId}: Cleanup error:`, cleanupError.message);
      }
    }

    // Exit worker thread
    process.exit(0);
  }
}

// Handle worker termination signals
process.on('SIGTERM', async () => {
  console.log(`🛑 Worker ${workerData.jobId}: Received termination signal`);
  process.exit(1);
});

process.on('SIGINT', async () => {
  console.log(`🛑 Worker ${workerData.jobId}: Received interrupt signal`);
  process.exit(1);
});

// Start the worker
executeCheckin().catch((error) => {
  console.error(`💥 Worker ${workerData.jobId}: Fatal error:`, error);
  process.exit(1);
});