require('dotenv').config({ path: '.env.local' });
const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');

let pool;

// MySQL connection pool oluştur
async function initializeMySQL() {
  try {
    pool = mysql.createPool({
      host: process.env.DB_HOST || 'localhost',
      port: process.env.DB_PORT || 3306,
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'vtravel_local',
      waitForConnections: true,
      connectionLimit: 10,
      queueLimit: 0,
      enableKeepAlive: true,
      keepAliveInitialDelay: 0
    });

    // Test connection
    const connection = await pool.getConnection();
    console.log(`✅ Connected to MySQL database: ${process.env.DB_NAME}`);
    connection.release();

    // Initialize schema if needed
    await initializeSchema();
    await logTableCounts();

    return pool;
  } catch (error) {
    console.error('❌ MySQL connection error:', error.message);
    throw error;
  }
}

// Schema'yı initialize et
async function initializeSchema() {
  try {
    const schemaPath = path.join(__dirname, 'database-schema-mysql.sql');

    if (!fs.existsSync(schemaPath)) {
      console.log('⚠️  MySQL schema file not found, skipping initialization');
      return;
    }

    const schema = fs.readFileSync(schemaPath, 'utf8');

    // SQL statements'ları ayır
    const statements = schema
      .split(';')
      .map(s => s.trim())
      .filter(s => s.length > 0 && !s.startsWith('--'));

    const connection = await pool.getConnection();

    try {
      for (const statement of statements) {
        if (statement.includes('CREATE TABLE')) {
          await connection.query(statement);
        }
      }
      console.log('✅ MySQL schema initialized');
    } finally {
      connection.release();
    }
  } catch (error) {
    if (error.code !== 'ER_TABLE_EXISTS_ERROR') {
      console.error('❌ Error initializing schema:', error.message);
    }
  }
}

// Tablo sayılarını logla
async function logTableCounts() {
  try {
    const [tables] = await pool.query('SHOW TABLES');
    console.log(`📊 Database tables (${tables.length} total):`);

    for (const table of tables) {
      const tableName = Object.values(table)[0];
      const [result] = await pool.query(`SELECT COUNT(*) as count FROM ${tableName}`);
      console.log(`   - ${tableName}: ${result[0].count} records`);
    }
  } catch (error) {
    console.error('❌ Error logging table counts:', error.message);
  }
}

// MySQL query wrapper (SQLite-like interface için)
const db = {
  // Promise-based query
  async query(sql, params = []) {
    try {
      const [rows] = await pool.query(sql, params);
      return rows;
    } catch (error) {
      console.error('❌ Query error:', error.message);
      throw error;
    }
  },

  // Single row query
  async get(sql, params = []) {
    const rows = await this.query(sql, params);
    return rows[0] || null;
  },

  // All rows query
  async all(sql, params = []) {
    return await this.query(sql, params);
  },

  // Insert/Update/Delete
  async run(sql, params = []) {
    try {
      const [result] = await pool.query(sql, params);
      return {
        lastID: result.insertId,
        changes: result.affectedRows
      };
    } catch (error) {
      console.error('❌ Run error:', error.message);
      throw error;
    }
  },

  // Close connection
  async close() {
    if (pool) {
      await pool.end();
      console.log('✅ MySQL connection closed');
    }
  }
};

// Initialize on module load
initializeMySQL().catch(err => {
  console.error('Failed to initialize MySQL:', err);
  process.exit(1);
});

// ============================================
// CHECK-IN TICKETS FUNCTIONS
// ============================================

// Helper: Convert snake_case to camelCase
function toCamelCase(str) {
  return str.replace(/_([a-z])/g, (g) => g[1].toUpperCase());
}

function rowToCamelCase(row) {
  if (!row) return null;
  const camelRow = {};
  for (const key in row) {
    camelRow[toCamelCase(key)] = row[key];
  }
  return camelRow;
}

// Generate UUID
function generateUUID() {
  return require('crypto').randomUUID();
}

// Create check-in ticket
async function createCheckinTicket(data) {
  const id = generateUUID();
  const timestamp = new Date().toISOString().slice(0, 19).replace('T', ' '); // MySQL datetime format

  // Convert departureDate to MySQL datetime format if it's ISO string
  let departureDate = data.departureDate;
  if (departureDate && typeof departureDate === 'string' && departureDate.includes('T')) {
    departureDate = departureDate.slice(0, 19).replace('T', ' ');
  }

  // Convert checkinOpenTime to MySQL datetime format if it's ISO string
  let checkinOpenTime = data.checkinOpenTime;
  if (checkinOpenTime && typeof checkinOpenTime === 'string' && checkinOpenTime.includes('T')) {
    checkinOpenTime = checkinOpenTime.slice(0, 19).replace('T', ' ');
  }

  await db.run(
    `INSERT INTO checkin_tickets (
      id, ticket_request_id, pnr, customer_id, requested_by_user_id, airline,
      flight_number, departure_airport, arrival_airport, departure_date,
      departure_time, checkin_status, checkin_open_time, is_auto_checkin_enabled,
      notes, created_at, updated_at
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
    [
      id, data.ticketRequestId || null, data.pnr, data.customerId || null,
      data.requestedByUserId || null, data.airline, data.flightNumber,
      data.departureAirport, data.arrivalAirport, departureDate,
      data.departureTime, data.checkinStatus || 'waiting',
      checkinOpenTime, data.isAutoCheckinEnabled ? 1 : 0,
      data.notes || null, timestamp, timestamp
    ]
  );

  return { id, ...data, createdAt: timestamp, updatedAt: timestamp };
}

// Get single check-in ticket
async function getCheckinTicket(id) {
  const row = await db.get('SELECT * FROM checkin_tickets WHERE id = ?', [id]);
  return rowToCamelCase(row);
}

// List check-in tickets with filters
async function listCheckinTickets(filters = {}, pagination = {}) {
  const { customerId, status, airline, pnr, ticketRequestId, scheduledDate, search } = filters;
  const { limit = 50, offset = 0 } = pagination;

  let whereClause = [];
  let params = [];

  if (customerId) {
    whereClause.push('customer_id = ?');
    params.push(customerId);
  }
  if (status) {
    whereClause.push('checkin_status = ?');
    params.push(status);
  }
  if (airline) {
    whereClause.push('airline = ?');
    params.push(airline);
  }
  if (pnr) {
    whereClause.push('pnr LIKE ?');
    params.push(`%${pnr}%`);
  }
  if (ticketRequestId) {
    whereClause.push('ticket_request_id = ?');
    params.push(ticketRequestId);
  }
  if (scheduledDate) {
    whereClause.push('DATE(departure_date) = ?');
    params.push(scheduledDate);
  }
  if (search) {
    whereClause.push('(pnr LIKE ? OR flight_number LIKE ? OR airline LIKE ?)');
    params.push(`%${search}%`, `%${search}%`, `%${search}%`);
  }

  const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';

  // Get total count
  const countQuery = `SELECT COUNT(*) as total FROM checkin_tickets ${where}`;
  const countResult = await db.get(countQuery, params);

  // Get data
  const dataQuery = `SELECT * FROM checkin_tickets ${where} ORDER BY departure_date DESC LIMIT ? OFFSET ?`;
  const rows = await db.all(dataQuery, [...params, limit, offset]);

  return {
    data: rows.map(rowToCamelCase),
    total: countResult.total,
    limit,
    offset
  };
}

// Update check-in ticket
async function updateCheckinTicket(id, updates) {
  const timestamp = new Date().toISOString();
  const fields = [];
  const values = [];

  // Convert camelCase to snake_case for database
  const snakeCaseMap = {
    ticketRequestId: 'ticket_request_id',
    customerId: 'customer_id',
    requestedByUserId: 'requested_by_user_id',
    flightNumber: 'flight_number',
    departureAirport: 'departure_airport',
    arrivalAirport: 'arrival_airport',
    departureDate: 'departure_date',
    departureTime: 'departure_time',
    checkinStatus: 'checkin_status',
    checkinOpenTime: 'checkin_open_time',
    isAutoCheckinEnabled: 'is_auto_checkin_enabled',
    checkinCompletedAt: 'checkin_completed_at'
  };

  for (const [key, value] of Object.entries(updates)) {
    if (key !== 'id' && key !== 'createdAt') {
      const dbKey = snakeCaseMap[key] || key;
      fields.push(`${dbKey} = ?`);
      values.push(value);
    }
  }

  fields.push('updated_at = ?');
  values.push(timestamp);
  values.push(id);

  await db.run(
    `UPDATE checkin_tickets SET ${fields.join(', ')} WHERE id = ?`,
    values
  );

  return await getCheckinTicket(id);
}

// Delete check-in ticket
async function deleteCheckinTicket(id) {
  const result = await db.run('DELETE FROM checkin_tickets WHERE id = ?', [id]);
  return result.changes > 0;
}

// Update check-in status
async function updateCheckinStatus(id, status, data = {}) {
  const updates = {
    checkinStatus: status,
    ...data
  };

  if (status === 'completed') {
    updates.checkinCompletedAt = new Date().toISOString();
  }

  return await updateCheckinTicket(id, updates);
}

// Get upcoming check-ins
async function getUpcomingCheckins() {
  const now = new Date().toISOString();
  const tomorrow = new Date(Date.now() + 24 * 60 * 60 * 1000).toISOString();

  const rows = await db.all(
    `SELECT * FROM checkin_tickets
     WHERE departure_date >= ? AND departure_date <= ?
     AND checkin_status IN ('waiting', 'ready')
     AND is_auto_checkin_enabled = 1
     ORDER BY departure_date ASC`,
    [now, tomorrow]
  );

  return rows.map(rowToCamelCase);
}

// List check-in tickets with ticket request info
async function listCheckinTicketsWithTicketInfo(filters = {}, pagination = {}) {
  const { customerId, status, airline, search } = filters;
  const { limit = 50, offset = 0 } = pagination;

  let whereClause = [];
  let params = [];

  if (customerId) {
    whereClause.push('ct.customer_id = ?');
    params.push(customerId);
  }
  if (status) {
    whereClause.push('ct.checkin_status = ?');
    params.push(status);
  }
  if (airline) {
    whereClause.push('ct.airline = ?');
    params.push(airline);
  }
  if (search) {
    whereClause.push('(ct.pnr LIKE ? OR ct.flight_number LIKE ? OR c.first_name LIKE ? OR c.last_name LIKE ?)');
    params.push(`%${search}%`, `%${search}%`, `%${search}%`, `%${search}%`);
  }

  const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';

  const query = `
    SELECT
      ct.*,
      c.first_name as customer_first_name,
      c.last_name as customer_last_name,
      c.email as customer_email,
      c.phone as customer_phone,
      tr.pnr as ticket_pnr,
      tr.status as ticket_status,
      p.first_name as passenger_first_name,
      p.last_name as passenger_last_name,
      p.phone as passenger_phone,
      p.email as passenger_email
    FROM checkin_tickets ct
    LEFT JOIN customers c ON ct.customer_id = c.id
    LEFT JOIN ticket_requests tr ON ct.ticket_request_id = tr.id
    LEFT JOIN checkin_ticket_passengers ctp ON ct.id = ctp.checkin_ticket_id
    LEFT JOIN passengers p ON ctp.passenger_id = p.id
    ${where}
    ORDER BY ct.departure_date DESC
    LIMIT ? OFFSET ?
  `;

  const rows = await db.all(query, [...params, limit, offset]);

  // Get total count
  const countQuery = `SELECT COUNT(DISTINCT ct.id) as total FROM checkin_tickets ct ${where}`;
  const countResult = await db.get(countQuery, params);

  return {
    data: rows.map(rowToCamelCase),
    total: countResult.total || 0
  };
}

// Get logs (placeholder)
// Save log
async function saveLog(logData) {
  try {
    // TODO: Fix checkin_logs schema - currently incompatible
    // Schema has: pnr, airline, flight_number, status, error_message
    // Code expects: checkin_ticket_id, message, level, timestamp
    // For now, silently skip logging to avoid errors
    return { success: true };

    /*
    const timestamp = new Date().toISOString().slice(0, 19).replace('T', ' '); // MySQL datetime format

    await db.run(
      `INSERT INTO checkin_logs (
        checkin_ticket_id, message, level, timestamp, created_at
      ) VALUES (?, ?, ?, ?, ?)`,
      [
        logData.checkinTicketId || null,
        logData.message || '',
        logData.level || 'info',
        timestamp,
        timestamp
      ]
    );

    return { success: true };
    */
  } catch (error) {
    // Silently fail - schema mismatch
    return { success: false, error: error.message };
  }
}

async function getLogs(filters = {}, pagination = {}) {
  try {
    const limit = pagination.limit || 50;
    const offset = pagination.offset || 0;

    let query = 'SELECT * FROM checkin_logs WHERE 1=1';
    const params = [];

    if (filters.checkinTicketId) {
      query += ' AND checkin_ticket_id = ?';
      params.push(filters.checkinTicketId);
    }

    if (filters.level) {
      query += ' AND level = ?';
      params.push(filters.level);
    }

    query += ' ORDER BY timestamp DESC LIMIT ? OFFSET ?';
    params.push(limit, offset);

    const data = await db.all(query, params);

    const countQuery = 'SELECT COUNT(*) as total FROM checkin_logs WHERE 1=1' +
      (filters.checkinTicketId ? ' AND checkin_ticket_id = ?' : '') +
      (filters.level ? ' AND level = ?' : '');

    const countParams = [];
    if (filters.checkinTicketId) countParams.push(filters.checkinTicketId);
    if (filters.level) countParams.push(filters.level);

    const countResult = await db.get(countQuery, countParams);

    return {
      data: data.map(rowToCamelCase),
      total: countResult?.total || 0,
      limit,
      offset
    };
  } catch (error) {
    console.error('Error getting logs:', error);
    return {
      data: [],
      total: 0,
      limit: pagination.limit || 50,
      offset: pagination.offset || 0
    };
  }
}

// Helper: Convert camelCase to snake_case
function toSnakeCase(str) {
  return str.replace(/[A-Z]/g, letter => `_${letter.toLowerCase()}`);
}

// ============================================
// PASSENGER FUNCTIONS
// ============================================

// Create passenger
async function createPassenger(data) {
  const id = generateUUID();
  const timestamp = new Date().toISOString().slice(0, 19).replace('T', ' ');

  await db.run(
    `INSERT INTO passengers (
      id, first_name, last_name, phone, email, created_at
    ) VALUES (?, ?, ?, ?, ?, ?)`,
    [
      id,
      data.firstName || 'YOLCU',
      data.lastName,
      data.phone || null,
      data.email || null,
      timestamp
    ]
  );

  return {
    id,
    firstName: data.firstName || 'YOLCU',
    lastName: data.lastName,
    phone: data.phone || null,
    email: data.email || null,
    createdAt: timestamp
  };
}

// Add passenger to check-in ticket
async function addPassengerToCheckin(checkinTicketId, passengerId, data = {}) {
  const id = generateUUID();

  await db.run(
    `INSERT INTO checkin_ticket_passengers (
      id, checkin_ticket_id, passenger_id, checkin_status
    ) VALUES (?, ?, ?, ?)`,
    [
      id,
      checkinTicketId,
      passengerId,
      data.checkinStatus || 'waiting'
    ]
  );

  return { id, checkinTicketId, passengerId };
}

// ============================================
// TICKET REQUEST FUNCTIONS
// ============================================

// Create ticket request
async function createTicketRequest(data) {
  const id = generateUUID();
  const timestamp = new Date().toISOString();

  await db.run(
    `INSERT INTO ticket_requests (
      id, customer_id, requested_by_user_id, airline, flight_number,
      departure_airport, arrival_airport, departure_date, departure_time,
      status, request_date, ticket_price, currency, agent_notes, pnr,
      created_at, updated_at
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
    [
      id, data.customerId, data.requestedByUserId, data.airline, data.flightNumber,
      data.departureAirport, data.arrivalAirport, data.departureDate, data.departureTime,
      data.status || 'pending', data.requestDate || timestamp, data.ticketPrice,
      data.currency || 'TRY', data.agentNotes, data.pnr, timestamp, timestamp
    ]
  );

  return { id, ...data, createdAt: timestamp, updatedAt: timestamp };
}

// Get single ticket request
async function getTicketRequest(id) {
  const row = await db.get('SELECT * FROM ticket_requests WHERE id = ?', [id]);
  return rowToCamelCase(row);
}

// List ticket requests with filters
async function listTicketRequests(filters = {}, pagination = {}) {
  const { customerId, status, airline, dateFrom, dateTo } = filters;
  const { limit = 50, offset = 0 } = pagination;

  let whereClause = [];
  let params = [];

  if (customerId) {
    whereClause.push('customer_id = ?');
    params.push(customerId);
  }
  if (status) {
    whereClause.push('status = ?');
    params.push(status);
  }
  if (airline) {
    whereClause.push('airline = ?');
    params.push(airline);
  }
  if (dateFrom) {
    whereClause.push('departure_date >= ?');
    params.push(dateFrom);
  }
  if (dateTo) {
    whereClause.push('departure_date <= ?');
    params.push(dateTo);
  }

  const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';

  // Get total count
  const countQuery = `SELECT COUNT(*) as total FROM ticket_requests ${where}`;
  const countResult = await db.get(countQuery, params);

  // Get data
  const dataQuery = `SELECT * FROM ticket_requests ${where} ORDER BY departure_date DESC LIMIT ? OFFSET ?`;
  const rows = await db.all(dataQuery, [...params, limit, offset]);

  return {
    data: rows.map(rowToCamelCase),
    total: countResult.total,
    limit,
    offset
  };
}

// Update ticket request
async function updateTicketRequest(id, updates) {
  const timestamp = new Date().toISOString();
  const fields = [];
  const values = [];

  for (const [key, value] of Object.entries(updates)) {
    if (key !== 'id' && key !== 'createdAt') {
      fields.push(`${toSnakeCase(key)} = ?`);
      values.push(value);
    }
  }

  fields.push('updated_at = ?');
  values.push(timestamp);
  values.push(id);

  await db.run(
    `UPDATE ticket_requests SET ${fields.join(', ')} WHERE id = ?`,
    values
  );

  return await getTicketRequest(id);
}

// Delete ticket request (soft delete - set status to cancelled)
async function deleteTicketRequest(id) {
  const timestamp = new Date().toISOString();
  const result = await db.run(
    'UPDATE ticket_requests SET status = ?, updated_at = ? WHERE id = ?',
    ['cancelled', timestamp, id]
  );
  return result.affectedRows > 0;
}

// Assign PNR to ticket
async function assignPnrToTicket(ticketId, pnr) {
  const timestamp = new Date().toISOString();
  await db.run(
    'UPDATE ticket_requests SET pnr = ?, status = ?, updated_at = ? WHERE id = ?',
    [pnr, 'ticketed', timestamp, ticketId]
  );
  return await getTicketRequest(ticketId);
}

// Get ticket with passengers
async function getTicketWithPassengers(ticketId) {
  const ticketRow = await db.get('SELECT * FROM ticket_requests WHERE id = ?', [ticketId]);

  if (!ticketRow) {
    return null;
  }

  const passengerRows = await db.all(
    `SELECT p.* FROM passengers p
     INNER JOIN ticket_request_passengers trp ON trp.passenger_id = p.id
     WHERE trp.ticket_request_id = ?`,
    [ticketId]
  );

  return {
    ...rowToCamelCase(ticketRow),
    passengers: passengerRows.map(rowToCamelCase)
  };
}

// Get dashboard statistics
async function getDashboardStats() {
  try {
    // Get ticket stats
    const ticketStats = await db.get(`
      SELECT
        COUNT(*) as total_tickets,
        COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_tickets,
        COUNT(CASE WHEN status = 'approved' THEN 1 END) as approved_tickets,
        COUNT(CASE WHEN status = 'ticketed' THEN 1 END) as ticketed_tickets,
        COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN 1 END) as today_tickets
      FROM ticket_requests
    `) || { total_tickets: 0, pending_tickets: 0, approved_tickets: 0, ticketed_tickets: 0, today_tickets: 0 };

    // Get check-in stats
    const checkinStats = await db.get(`
      SELECT
        COUNT(*) as total_checkins,
        COUNT(CASE WHEN checkin_status = 'completed' THEN 1 END) as completed_checkins,
        COUNT(CASE WHEN checkin_status = 'failed' THEN 1 END) as failed_checkins,
        COUNT(CASE WHEN checkin_status = 'waiting' THEN 1 END) as waiting_checkins,
        COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN 1 END) as today_checkins
      FROM checkin_tickets
    `) || { total_checkins: 0, completed_checkins: 0, failed_checkins: 0, waiting_checkins: 0, today_checkins: 0 };

    // Get customer stats
    const customerStats = await db.get(`
      SELECT
        COUNT(*) as total_customers,
        COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN 1 END) as today_customers
      FROM customers
    `) || { total_customers: 0, today_customers: 0 };

    // Get user stats
    const userStats = await db.get(`
      SELECT
        COUNT(*) as total_users,
        COUNT(CASE WHEN is_active = 1 THEN 1 END) as active_users,
        COUNT(CASE WHEN is_online = 1 THEN 1 END) as online_users
      FROM agency_users
    `) || { total_users: 0, active_users: 0, online_users: 0 };

    return {
      tickets: {
        total: ticketStats.total_tickets || 0,
        pending: ticketStats.pending_tickets || 0,
        approved: ticketStats.approved_tickets || 0,
        ticketed: ticketStats.ticketed_tickets || 0,
        today: ticketStats.today_tickets || 0
      },
      checkins: {
        total: checkinStats.total_checkins || 0,
        completed: checkinStats.completed_checkins || 0,
        failed: checkinStats.failed_checkins || 0,
        waiting: checkinStats.waiting_checkins || 0,
        today: checkinStats.today_checkins || 0
      },
      customers: {
        total: customerStats.total_customers || 0,
        today: customerStats.today_customers || 0
      },
      users: {
        total: userStats.total_users || 0,
        active: userStats.active_users || 0,
        online: userStats.online_users || 0
      }
    };
  } catch (error) {
    console.error('Error getting dashboard stats:', error);
    // Return empty stats on error
    return {
      tickets: { total: 0, pending: 0, approved: 0, ticketed: 0, today: 0 },
      checkins: { total: 0, completed: 0, failed: 0, waiting: 0, today: 0 },
      customers: { total: 0, today: 0 },
      users: { total: 0, active: 0, online: 0 }
    };
  }
}

// Stub functions for reports route (to be implemented)
async function listCustomers(filters = {}, pagination = {}) {
  return { customers: [], total: 0 };
}

async function getAccountingStats(period) {
  return { totalRevenue: 0, totalExpenses: 0, netProfit: 0 };
}

async function listAccommodationRequests() {
  return [];
}

async function listVisaRequests() {
  return [];
}

async function listTransferRequests() {
  return [];
}

async function listInvoices() {
  return [];
}

async function listRevenueRecords() {
  return [];
}

// Export all functions including check-in and ticket requests
module.exports = {
  ...db,
  createCheckinTicket,
  getCheckinTicket,
  listCheckinTickets,
  updateCheckinTicket,
  deleteCheckinTicket,
  updateCheckinStatus,
  getUpcomingCheckins,
  listCheckinTicketsWithTicketInfo,
  saveLog,
  getLogs,
  createPassenger,
  addPassengerToCheckin,
  createTicketRequest,
  getTicketRequest,
  listTicketRequests,
  updateTicketRequest,
  deleteTicketRequest,
  assignPnrToTicket,
  getTicketWithPassengers,
  getDashboardStats,
  listCustomers,
  getAccountingStats,
  listAccommodationRequests,
  listVisaRequests,
  listTransferRequests,
  listInvoices,
  listRevenueRecords
};
