require('dotenv').config({ path: '.env.local' });

// Check if MySQL or SQLite should be used
const USE_MYSQL = process.env.DB_TYPE === 'mysql';

if (USE_MYSQL) {
  // Use MySQL
  const db = require('./database-mysql');
  module.exports = db;
  console.log('🔄 Using MySQL database');
} else {
  // Use SQLite (legacy)
  const sqlite3 = require('sqlite3').verbose();
  const path = require('path');
  const fs = require('fs');
  const crypto = require('crypto');

  // SQLite veritabanı dosya yolu - environment variable'dan veya default
  const dbPath = process.env.DB_PATH || path.join(__dirname, '../data/vtravel.db');

  // Veritabanı bağlantısı
  const db = new sqlite3.Database(dbPath, (err) => {
    if (err) {
      console.error('❌ Database connection error:', err);
    } else {
      console.log(`✅ Connected to SQLite database: ${dbPath}`);
      initializeDatabase();
    }
  });

// Veritabanı tablolarını oluştur
function initializeDatabase() {
  const schemaPath = path.join(__dirname, 'database-schema.sql');

  // Schema dosyasını oku ve execute et
  try {
    const schema = fs.readFileSync(schemaPath, 'utf8');

    // SQL statements'ları ayır (-- ile başlayan comment'ler hariç)
    const statements = schema
      .split(';')
      .map(s => s.trim())
      .filter(s => s.length > 0 && !s.startsWith('--'));

    // CREATE TABLE ve CREATE INDEX'leri ayır
    const createTables = statements.filter(s => s.includes('CREATE TABLE'));
    const createIndexes = statements.filter(s => s.includes('CREATE INDEX'));

    db.serialize(() => {
      let completedCount = 0;
      const totalCount = statements.length;

      // Önce tüm tabloları oluştur
      createTables.forEach((statement, index) => {
        db.run(statement, (err) => {
          completedCount++;

          if (err) {
            if (!err.message.includes('already exists')) {
              console.error(`❌ Error creating table ${index + 1}:`, err.message);
            }
          }

          // Tüm tablolar oluşturulduysa index'leri oluştur
          if (completedCount === createTables.length) {
            console.log(`✅ ${createTables.length} tables created`);

            // Şimdi index'leri oluştur
            createIndexes.forEach((statement, indexNum) => {
              db.run(statement, (err) => {
                completedCount++;

                if (err) {
                  if (!err.message.includes('already exists')) {
                    console.error(`❌ Error creating index ${indexNum + 1}:`, err.message);
                  }
                }

                // Her şey tamamlandıysa log göster
                if (completedCount === totalCount) {
                  console.log(`✅ ${createIndexes.length} indexes created`);
                  console.log(`✅ Database initialized: ${totalCount} statements executed`);
                  setTimeout(logTableCounts, 500); // Biraz bekle ki tüm işlemler tamamlansın
                }
              });
            });
          }
        });
      });
    });
  } catch (error) {
    console.error('❌ Error reading schema file:', error);
  }
}

// Oluşturulan tabloları ve kayıt sayılarını logla
function logTableCounts() {
  db.all("SELECT name FROM sqlite_master WHERE type='table' ORDER BY name", [], (err, tables) => {
    if (err) {
      console.error('❌ Error fetching tables:', err);
      return;
    }

    console.log(`📊 Database tables (${tables.length} total):`);
    tables.forEach(table => {
      if (table.name !== 'sqlite_sequence') {
        db.get(`SELECT COUNT(*) as count FROM ${table.name}`, [], (err, row) => {
          if (!err && row) {
            console.log(`   - ${table.name}: ${row.count} records`);
          }
        });
      }
    });
  });
}

// ============================================
// UTILITY FUNCTIONS
// ============================================

// Generate UUID
function generateUUID() {
  return crypto.randomUUID();
}

// Get current timestamp
function getCurrentTimestamp() {
  return new Date().toISOString();
}

// Convert camelCase to snake_case for database columns
function toSnakeCase(str) {
  return str.replace(/[A-Z]/g, letter => `_${letter.toLowerCase()}`);
}

// Convert snake_case to camelCase for JavaScript objects
function toCamelCase(str) {
  return str.replace(/_([a-z])/g, (g) => g[1].toUpperCase());
}

// Convert database row to camelCase object
function rowToCamelCase(row) {
  if (!row) return null;
  const result = {};
  for (const key in row) {
    result[toCamelCase(key)] = row[key];
  }
  return result;
}

// ============================================
// CUSTOMERS
// ============================================

function createCustomer(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO customers (
        id, type, first_name, last_name, company_name, tax_number, tax_office,
        contact_person, department, parent_company_id, email, phone, address,
        city, country, notes, is_vip, is_active, created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.type, data.firstName, data.lastName, data.companyName,
        data.taxNumber, data.taxOffice, data.contactPerson, data.department,
        data.parentCompanyId, data.email, data.phone, data.address,
        data.city, data.country, data.notes, data.isVip ? 1 : 0,
        data.isActive !== undefined ? (data.isActive ? 1 : 0) : 1,
        timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getCustomer(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM customers WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else resolve(rowToCamelCase(row));
    });
  });
}

function listCustomers(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { type, status, search, isVip } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (type) {
      whereClause.push('type = ?');
      params.push(type);
    }
    if (status !== undefined) {
      whereClause.push('is_active = ?');
      params.push(status ? 1 : 0);
    }
    if (isVip !== undefined) {
      whereClause.push('is_vip = ?');
      params.push(isVip ? 1 : 0);
    }
    if (search) {
      whereClause.push('(first_name LIKE ? OR last_name LIKE ? OR company_name LIKE ? OR email LIKE ? OR phone LIKE ?)');
      const searchTerm = `%${search}%`;
      params.push(searchTerm, searchTerm, searchTerm, searchTerm, searchTerm);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM customers ${where}`;
    const dataQuery = `SELECT * FROM customers ${where} ORDER BY created_at DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else resolve({
          data: rows.map(rowToCamelCase),
          total: countRow.total,
          limit,
          offset
        });
      });
    });
  });
}

function updateCustomer(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(value);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE customers SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getCustomer(id).then(resolve).catch(reject);
      }
    );
  });
}

function deleteCustomer(id) {
  return new Promise((resolve, reject) => {
    // Soft delete
    const timestamp = getCurrentTimestamp();
    db.run(
      'UPDATE customers SET is_active = 0, updated_at = ? WHERE id = ?',
      [timestamp, id],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

function searchCustomers(query) {
  return new Promise((resolve, reject) => {
    const searchTerm = `%${query}%`;
    db.all(
      `SELECT * FROM customers
       WHERE (first_name LIKE ? OR last_name LIKE ? OR company_name LIKE ? OR email LIKE ? OR phone LIKE ?)
       AND is_active = 1
       ORDER BY created_at DESC
       LIMIT 50`,
      [searchTerm, searchTerm, searchTerm, searchTerm, searchTerm],
      (err, rows) => {
        if (err) reject(err);
        else resolve(rows.map(rowToCamelCase));
      }
    );
  });
}

function getCustomerStats(customerId) {
  return new Promise((resolve, reject) => {
    db.get(
      `SELECT
        (SELECT COUNT(*) FROM ticket_requests WHERE customer_id = ?) as total_tickets,
        (SELECT COUNT(*) FROM accommodation_requests WHERE customer_id = ?) as total_accommodations,
        (SELECT COUNT(*) FROM visa_requests WHERE customer_id = ?) as total_visas,
        (SELECT COUNT(*) FROM transfer_requests WHERE customer_id = ?) as total_transfers,
        (SELECT SUM(total_amount) FROM invoices WHERE customer_id = ? AND payment_status = 'paid') as total_paid,
        (SELECT SUM(total_amount) FROM invoices WHERE customer_id = ? AND payment_status = 'unpaid') as total_unpaid`,
      [customerId, customerId, customerId, customerId, customerId, customerId],
      (err, row) => {
        if (err) reject(err);
        else resolve(rowToCamelCase(row));
      }
    );
  });
}

// ============================================
// CORPORATE USERS
// ============================================

function createCorporateUser(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO corporate_users (
        id, customer_id, first_name, last_name, email, phone, department,
        position, permissions, is_active, created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.customerId, data.firstName, data.lastName, data.email,
        data.phone, data.department, data.position,
        JSON.stringify(data.permissions || []),
        data.isActive !== undefined ? (data.isActive ? 1 : 0) : 1,
        timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getCorporateUser(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM corporate_users WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else {
        const user = rowToCamelCase(row);
        if (user && user.permissions) {
          user.permissions = JSON.parse(user.permissions);
        }
        resolve(user);
      }
    });
  });
}

function listCorporateUsers(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { customerId, isActive } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (customerId) {
      whereClause.push('customer_id = ?');
      params.push(customerId);
    }
    if (isActive !== undefined) {
      whereClause.push('is_active = ?');
      params.push(isActive ? 1 : 0);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM corporate_users ${where}`;
    const dataQuery = `SELECT * FROM corporate_users ${where} ORDER BY created_at DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else {
          const users = rows.map(row => {
            const user = rowToCamelCase(row);
            if (user.permissions) user.permissions = JSON.parse(user.permissions);
            return user;
          });
          resolve({
            data: users,
            total: countRow.total,
            limit,
            offset
          });
        }
      });
    });
  });
}

function updateCorporateUser(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(key === 'permissions' ? JSON.stringify(value) : value);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE corporate_users SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getCorporateUser(id).then(resolve).catch(reject);
      }
    );
  });
}

function deleteCorporateUser(id) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    db.run(
      'UPDATE corporate_users SET is_active = 0, updated_at = ? WHERE id = ?',
      [timestamp, id],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

// ============================================
// AGENCY USERS
// ============================================

function createAgencyUser(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO agency_users (
        id, first_name, last_name, email, phone, avatar, department_id,
        position, role, permissions, is_active, manager_id, created_at, updated_at, created_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.firstName, data.lastName, data.email, data.phone, data.avatar,
        data.departmentId, data.position, data.role,
        JSON.stringify(data.permissions || []),
        data.isActive !== undefined ? (data.isActive ? 1 : 0) : 1,
        data.managerId, timestamp, timestamp, data.createdBy
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getAgencyUser(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM agency_users WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else {
        const user = rowToCamelCase(row);
        if (user && user.permissions) {
          user.permissions = JSON.parse(user.permissions);
        }
        resolve(user);
      }
    });
  });
}

function listAgencyUsers(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { departmentId, role, isActive } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (departmentId) {
      whereClause.push('department_id = ?');
      params.push(departmentId);
    }
    if (role) {
      whereClause.push('role = ?');
      params.push(role);
    }
    if (isActive !== undefined) {
      whereClause.push('is_active = ?');
      params.push(isActive ? 1 : 0);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM agency_users ${where}`;
    const dataQuery = `SELECT * FROM agency_users ${where} ORDER BY created_at DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else {
          const users = rows.map(row => {
            const user = rowToCamelCase(row);
            if (user.permissions) user.permissions = JSON.parse(user.permissions);
            return user;
          });
          resolve({
            data: users,
            total: countRow.total,
            limit,
            offset
          });
        }
      });
    });
  });
}

function updateAgencyUser(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt' && key !== 'createdBy') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(key === 'permissions' ? JSON.stringify(value) : value);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE agency_users SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getAgencyUser(id).then(resolve).catch(reject);
      }
    );
  });
}

function deleteAgencyUser(id) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    db.run(
      'UPDATE agency_users SET is_active = 0, updated_at = ? WHERE id = ?',
      [timestamp, id],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

// ============================================
// DEPARTMENTS
// ============================================

function createDepartment(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO departments (
        id, name, description, color, parent_department_id, manager_id,
        is_active, created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.name, data.description, data.color, data.parentDepartmentId,
        data.managerId, data.isActive !== undefined ? (data.isActive ? 1 : 0) : 1,
        timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getDepartment(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM departments WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else resolve(rowToCamelCase(row));
    });
  });
}

function listDepartments(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { isActive } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (isActive !== undefined) {
      whereClause.push('is_active = ?');
      params.push(isActive ? 1 : 0);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM departments ${where}`;
    const dataQuery = `SELECT * FROM departments ${where} ORDER BY name ASC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else resolve({
          data: rows.map(rowToCamelCase),
          total: countRow.total,
          limit,
          offset
        });
      });
    });
  });
}

function updateDepartment(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(value);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE departments SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getDepartment(id).then(resolve).catch(reject);
      }
    );
  });
}

function deleteDepartment(id) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    db.run(
      'UPDATE departments SET is_active = 0, updated_at = ? WHERE id = ?',
      [timestamp, id],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

// ============================================
// PASSENGERS
// ============================================

function createPassenger(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO passengers (
        id, first_name, last_name, phone, email, employee_id, department,
        position, age, is_child, special_needs, created_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.firstName, data.lastName, data.phone, data.email,
        data.employeeId, data.department, data.position, data.age,
        data.isChild ? 1 : 0, data.specialNeeds, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp });
      }
    );
  });
}

function getPassenger(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM passengers WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else resolve(rowToCamelCase(row));
    });
  });
}

function listPassengers(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { search, firstName, lastName } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (search) {
      whereClause.push('(first_name LIKE ? OR last_name LIKE ? OR employee_id LIKE ?)');
      const searchTerm = `%${search}%`;
      params.push(searchTerm, searchTerm, searchTerm);
    }

    if (firstName) {
      whereClause.push('first_name = ?');
      params.push(firstName);
    }

    if (lastName) {
      whereClause.push('last_name = ?');
      params.push(lastName);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM passengers ${where}`;
    const dataQuery = `SELECT * FROM passengers ${where} ORDER BY created_at DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else resolve({
          data: rows.map(rowToCamelCase),
          total: countRow.total,
          limit,
          offset
        });
      });
    });
  });
}

function updatePassenger(id, updates) {
  return new Promise((resolve, reject) => {
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(value);
      }
    }

    values.push(id);

    db.run(
      `UPDATE passengers SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getPassenger(id).then(resolve).catch(reject);
      }
    );
  });
}

function deletePassenger(id) {
  return new Promise((resolve, reject) => {
    db.run('DELETE FROM passengers WHERE id = ?', [id], function(err) {
      if (err) reject(err);
      else resolve(this.changes > 0);
    });
  });
}

// ============================================
// TICKET REQUESTS
// ============================================

function createTicketRequest(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO ticket_requests (
        id, customer_id, requested_by_user_id, airline, flight_number,
        departure_airport, arrival_airport, departure_date, departure_time,
        status, request_date, ticket_price, currency, agent_notes, pnr,
        created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.customerId, data.requestedByUserId, data.airline, data.flightNumber,
        data.departureAirport, data.arrivalAirport, data.departureDate, data.departureTime,
        data.status || 'pending', data.requestDate || timestamp, data.ticketPrice,
        data.currency || 'TRY', data.agentNotes, data.pnr, timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getTicketRequest(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM ticket_requests WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else resolve(rowToCamelCase(row));
    });
  });
}

function listTicketRequests(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { customerId, status, airline, dateFrom, dateTo } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (customerId) {
      whereClause.push('customer_id = ?');
      params.push(customerId);
    }
    if (status) {
      whereClause.push('status = ?');
      params.push(status);
    }
    if (airline) {
      whereClause.push('airline = ?');
      params.push(airline);
    }
    if (dateFrom) {
      whereClause.push('departure_date >= ?');
      params.push(dateFrom);
    }
    if (dateTo) {
      whereClause.push('departure_date <= ?');
      params.push(dateTo);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM ticket_requests ${where}`;
    const dataQuery = `SELECT * FROM ticket_requests ${where} ORDER BY departure_date DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else resolve({
          data: rows.map(rowToCamelCase),
          total: countRow.total,
          limit,
          offset
        });
      });
    });
  });
}

function updateTicketRequest(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(value);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE ticket_requests SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getTicketRequest(id).then(resolve).catch(reject);
      }
    );
  });
}

function deleteTicketRequest(id) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    db.run(
      'UPDATE ticket_requests SET status = "cancelled", updated_at = ? WHERE id = ?',
      [timestamp, id],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

function assignPnrToTicket(ticketId, pnr) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    db.run(
      'UPDATE ticket_requests SET pnr = ?, status = "ticketed", updated_at = ? WHERE id = ?',
      [pnr, timestamp, ticketId],
      function(err) {
        if (err) reject(err);
        else getTicketRequest(ticketId).then(resolve).catch(reject);
      }
    );
  });
}

function getTicketWithPassengers(ticketId) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM ticket_requests WHERE id = ?', [ticketId], (err, ticketRow) => {
      if (err) {
        reject(err);
        return;
      }
      if (!ticketRow) {
        resolve(null);
        return;
      }

      db.all(
        `SELECT p.* FROM passengers p
         INNER JOIN ticket_request_passengers trp ON p.id = trp.passenger_id
         WHERE trp.ticket_request_id = ?`,
        [ticketId],
        (err, passengerRows) => {
          if (err) {
            reject(err);
            return;
          }

          const ticket = rowToCamelCase(ticketRow);
          ticket.passengers = passengerRows.map(rowToCamelCase);
          resolve(ticket);
        }
      );
    });
  });
}

function getUpcomingTickets() {
  return new Promise((resolve, reject) => {
    const now = new Date().toISOString();
    const tomorrow = new Date(Date.now() + 24 * 60 * 60 * 1000).toISOString();

    db.all(
      `SELECT * FROM ticket_requests
       WHERE departure_date >= ? AND departure_date <= ?
       AND status IN ('confirmed', 'ticketed')
       ORDER BY departure_date ASC`,
      [now, tomorrow],
      (err, rows) => {
        if (err) reject(err);
        else resolve(rows.map(rowToCamelCase));
      }
    );
  });
}

// ============================================
// TICKET REQUEST PASSENGERS
// ============================================

function addPassengerToTicket(ticketRequestId, passengerId) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    db.run(
      'INSERT INTO ticket_request_passengers (id, ticket_request_id, passenger_id) VALUES (?, ?, ?)',
      [id, ticketRequestId, passengerId],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ticketRequestId, passengerId });
      }
    );
  });
}

function removePassengerFromTicket(ticketRequestId, passengerId) {
  return new Promise((resolve, reject) => {
    db.run(
      'DELETE FROM ticket_request_passengers WHERE ticket_request_id = ? AND passenger_id = ?',
      [ticketRequestId, passengerId],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

// ============================================
// CHECKIN TICKETS
// ============================================

function createCheckinTicket(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO checkin_tickets (
        id, ticket_request_id, pnr, customer_id, requested_by_user_id, airline,
        flight_number, departure_airport, arrival_airport, departure_date,
        departure_time, checkin_status, checkin_open_time, is_auto_checkin_enabled,
        notes, created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.ticketRequestId, data.pnr, data.customerId, data.requestedByUserId,
        data.airline, data.flightNumber, data.departureAirport, data.arrivalAirport,
        data.departureDate, data.departureTime, data.checkinStatus || 'waiting',
        data.checkinOpenTime, data.isAutoCheckinEnabled !== undefined ? (data.isAutoCheckinEnabled ? 1 : 0) : 1,
        data.notes, timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getCheckinTicket(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM checkin_tickets WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else resolve(rowToCamelCase(row));
    });
  });
}

function listCheckinTickets(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { customerId, status, airline, pnr } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (customerId) {
      whereClause.push('customer_id = ?');
      params.push(customerId);
    }
    if (status) {
      whereClause.push('checkin_status = ?');
      params.push(status);
    }
    if (airline) {
      whereClause.push('airline = ?');
      params.push(airline);
    }
    if (pnr) {
      whereClause.push('pnr = ?');
      params.push(pnr);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM checkin_tickets ${where}`;
    const dataQuery = `SELECT * FROM checkin_tickets ${where} ORDER BY departure_date DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], async (err, rows) => {
        if (err) {
          reject(err);
          return;
        }

        // Her check-in ticket için yolcuları getir
        const ticketsWithPassengers = await Promise.all(
          rows.map(async (ticket) => {
            return new Promise((resolveTicket, rejectTicket) => {
              db.all(
                `SELECT p.* FROM passengers p
                 INNER JOIN checkin_ticket_passengers ctp ON p.id = ctp.passenger_id
                 WHERE ctp.checkin_ticket_id = ?`,
                [ticket.id],
                (err, passengerRows) => {
                  if (err) {
                    console.error('Error fetching passengers for ticket:', err);
                    resolveTicket({
                      ...rowToCamelCase(ticket),
                      passengers: []
                    });
                  } else {
                    resolveTicket({
                      ...rowToCamelCase(ticket),
                      passengers: passengerRows.map(rowToCamelCase)
                    });
                  }
                }
              );
            });
          })
        );

        resolve({
          data: ticketsWithPassengers,
          total: countRow.total,
          limit,
          offset
        });
      });
    });
  });
}

function updateCheckinTicket(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(value);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE checkin_tickets SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getCheckinTicket(id).then(resolve).catch(reject);
      }
    );
  });
}

function deleteCheckinTicket(id) {
  return new Promise((resolve, reject) => {
    // Hard delete - completely remove the record from database
    db.run(
      'DELETE FROM checkin_tickets WHERE id = ?',
      [id],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

function scheduleAutoCheckin(ticketId, scheduledTime) {
  return new Promise((resolve, reject) => {
    const jobId = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO auto_checkin_jobs (
        id, checkin_ticket_id, scheduled_for, status, created_at, updated_at
      ) VALUES (?, ?, ?, 'scheduled', ?, ?)`,
      [jobId, ticketId, scheduledTime, timestamp, timestamp],
      function(err) {
        if (err) reject(err);
        else resolve({ id: jobId, checkinTicketId: ticketId, scheduledFor: scheduledTime });
      }
    );
  });
}

function getUpcomingCheckins() {
  return new Promise((resolve, reject) => {
    const now = new Date().toISOString();
    const tomorrow = new Date(Date.now() + 24 * 60 * 60 * 1000).toISOString();

    db.all(
      `SELECT * FROM checkin_tickets
       WHERE departure_date >= ? AND departure_date <= ?
       AND checkin_status IN ('waiting', 'ready')
       AND is_auto_checkin_enabled = 1
       ORDER BY departure_date ASC`,
      [now, tomorrow],
      (err, rows) => {
        if (err) reject(err);
        else resolve(rows.map(rowToCamelCase));
      }
    );
  });
}

// ============================================
// CHECKIN TICKET PASSENGERS
// ============================================

function addPassengerToCheckin(checkinTicketId, passengerId, data = {}) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    db.run(
      `INSERT INTO checkin_ticket_passengers (
        id, checkin_ticket_id, passenger_id, checkin_status, seat, boarding_pass_url, error_message
      ) VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [id, checkinTicketId, passengerId, data.checkinStatus || 'waiting', data.seat, data.boardingPassUrl, data.errorMessage],
      function(err) {
        if (err) reject(err);
        else resolve({ id, checkinTicketId, passengerId, ...data });
      }
    );
  });
}

function updateCheckinPassengerResult(id, updates) {
  return new Promise((resolve, reject) => {
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(value);
      }
    }

    values.push(id);

    db.run(
      `UPDATE checkin_ticket_passengers SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

// ============================================
// AUTO CHECKIN JOBS
// ============================================

function createAutoCheckinJob(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO auto_checkin_jobs (
        id, checkin_ticket_id, scheduled_for, status, attempts, parent_job_id,
        retry_reason, created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.checkinTicketId, data.scheduledFor, data.status || 'scheduled',
        data.attempts || 0, data.parentJobId, data.retryReason, timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getAutoCheckinJob(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM auto_checkin_jobs WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else resolve(rowToCamelCase(row));
    });
  });
}

function listAutoCheckinJobs(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { status, scheduledFrom, scheduledTo } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (status) {
      whereClause.push('status = ?');
      params.push(status);
    }
    if (scheduledFrom) {
      whereClause.push('scheduled_for >= ?');
      params.push(scheduledFrom);
    }
    if (scheduledTo) {
      whereClause.push('scheduled_for <= ?');
      params.push(scheduledTo);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM auto_checkin_jobs ${where}`;
    const dataQuery = `SELECT * FROM auto_checkin_jobs ${where} ORDER BY scheduled_for ASC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else resolve({
          data: rows.map(rowToCamelCase),
          total: countRow.total,
          limit,
          offset
        });
      });
    });
  });
}

function updateAutoCheckinJob(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(value);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE auto_checkin_jobs SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getAutoCheckinJob(id).then(resolve).catch(reject);
      }
    );
  });
}

// ============================================
// FLIGHTS
// ============================================

function createFlight(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO flights (
        id, airline, flight_number, departure_airport, arrival_airport,
        departure_date, departure_time, arrival_date, arrival_time, status,
        delay_minutes, gate, terminal, created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.airline, data.flightNumber, data.departureAirport, data.arrivalAirport,
        data.departureDate, data.departureTime, data.arrivalDate, data.arrivalTime,
        data.status || 'scheduled', data.delayMinutes, data.gate, data.terminal,
        timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getFlight(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM flights WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else resolve(rowToCamelCase(row));
    });
  });
}

function listFlights(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { airline, status, departureDate } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (airline) {
      whereClause.push('airline = ?');
      params.push(airline);
    }
    if (status) {
      whereClause.push('status = ?');
      params.push(status);
    }
    if (departureDate) {
      whereClause.push('DATE(departure_date) = DATE(?)');
      params.push(departureDate);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM flights ${where}`;
    const dataQuery = `SELECT * FROM flights ${where} ORDER BY departure_date DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else resolve({
          data: rows.map(rowToCamelCase),
          total: countRow.total,
          limit,
          offset
        });
      });
    });
  });
}

function updateFlight(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(value);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE flights SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getFlight(id).then(resolve).catch(reject);
      }
    );
  });
}

// ============================================
// ACCOMMODATION REQUESTS
// ============================================

function createAccommodationRequest(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO accommodation_requests (
        id, customer_id, requested_by_user_id, destination, check_in_date,
        check_out_date, nights, total_guests, room_type, room_count, budget_min,
        budget_max, currency, special_requests, meal_plan, hotel_category,
        status, agent_notes, created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.customerId, data.requestedByUserId, data.destination, data.checkInDate,
        data.checkOutDate, data.nights, data.totalGuests, data.roomType, data.roomCount,
        data.budgetMin, data.budgetMax, data.currency || 'TRY', data.specialRequests,
        data.mealPlan, data.hotelCategory, data.status || 'pending', data.agentNotes,
        timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getAccommodationRequest(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM accommodation_requests WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else resolve(rowToCamelCase(row));
    });
  });
}

function listAccommodationRequests(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { customerId, status, destination } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (customerId) {
      whereClause.push('customer_id = ?');
      params.push(customerId);
    }
    if (status) {
      whereClause.push('status = ?');
      params.push(status);
    }
    if (destination) {
      whereClause.push('destination LIKE ?');
      params.push(`%${destination}%`);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM accommodation_requests ${where}`;
    const dataQuery = `SELECT * FROM accommodation_requests ${where} ORDER BY check_in_date DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else resolve({
          data: rows.map(rowToCamelCase),
          total: countRow.total,
          limit,
          offset
        });
      });
    });
  });
}

function updateAccommodationRequest(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(value);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE accommodation_requests SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getAccommodationRequest(id).then(resolve).catch(reject);
      }
    );
  });
}

function deleteAccommodationRequest(id) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    db.run(
      'UPDATE accommodation_requests SET status = "cancelled", updated_at = ? WHERE id = ?',
      [timestamp, id],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

function getAccommodationWithGuests(requestId) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM accommodation_requests WHERE id = ?', [requestId], (err, requestRow) => {
      if (err) {
        reject(err);
        return;
      }
      if (!requestRow) {
        resolve(null);
        return;
      }

      db.all(
        'SELECT * FROM accommodation_guests WHERE accommodation_request_id = ?',
        [requestId],
        (err, guestRows) => {
          if (err) {
            reject(err);
            return;
          }

          const request = rowToCamelCase(requestRow);
          request.guests = guestRows.map(rowToCamelCase);
          resolve(request);
        }
      );
    });
  });
}

function updateAccommodationStatus(requestId, status, details = {}) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const updates = {
      status,
      ...details,
      updatedAt: timestamp
    };
    updateAccommodationRequest(requestId, updates).then(resolve).catch(reject);
  });
}

// ============================================
// ACCOMMODATION GUESTS
// ============================================

function addGuestToAccommodation(accommodationRequestId, guestData) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    db.run(
      `INSERT INTO accommodation_guests (
        id, accommodation_request_id, first_name, last_name, age, is_child, special_needs
      ) VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [
        id, accommodationRequestId, guestData.firstName, guestData.lastName,
        guestData.age, guestData.isChild ? 1 : 0, guestData.specialNeeds
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, accommodationRequestId, ...guestData });
      }
    );
  });
}

// ============================================
// VISA REQUESTS
// ============================================

function createVisaRequest(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO visa_requests (
        id, customer_id, requested_by_user_id, applicant_data, country, visa_type,
        application_date, travel_date, return_date, status, priority, documents,
        required_documents, application_fee, service_fee, total_amount, currency,
        agent_notes, special_requests, created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.customerId, data.requestedByUserId, JSON.stringify(data.applicantData),
        data.country, data.visaType, data.applicationDate, data.travelDate, data.returnDate,
        data.status || 'pending', data.priority || 'medium',
        JSON.stringify(data.documents || []), JSON.stringify(data.requiredDocuments || []),
        data.applicationFee, data.serviceFee, data.totalAmount, data.currency || 'TRY',
        data.agentNotes, data.specialRequests, timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getVisaRequest(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM visa_requests WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else {
        const visa = rowToCamelCase(row);
        if (visa) {
          if (visa.applicantData) visa.applicantData = JSON.parse(visa.applicantData);
          if (visa.documents) visa.documents = JSON.parse(visa.documents);
          if (visa.requiredDocuments) visa.requiredDocuments = JSON.parse(visa.requiredDocuments);
        }
        resolve(visa);
      }
    });
  });
}

function listVisaRequests(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { customerId, status, country, priority } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (customerId) {
      whereClause.push('customer_id = ?');
      params.push(customerId);
    }
    if (status) {
      whereClause.push('status = ?');
      params.push(status);
    }
    if (country) {
      whereClause.push('country = ?');
      params.push(country);
    }
    if (priority) {
      whereClause.push('priority = ?');
      params.push(priority);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM visa_requests ${where}`;
    const dataQuery = `SELECT * FROM visa_requests ${where} ORDER BY travel_date DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else {
          const visas = rows.map(row => {
            const visa = rowToCamelCase(row);
            if (visa.applicantData) visa.applicantData = JSON.parse(visa.applicantData);
            if (visa.documents) visa.documents = JSON.parse(visa.documents);
            if (visa.requiredDocuments) visa.requiredDocuments = JSON.parse(visa.requiredDocuments);
            return visa;
          });
          resolve({
            data: visas,
            total: countRow.total,
            limit,
            offset
          });
        }
      });
    });
  });
}

function updateVisaRequest(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        const dbValue = ['applicantData', 'documents', 'requiredDocuments'].includes(key)
          ? JSON.stringify(value)
          : value;
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(dbValue);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE visa_requests SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getVisaRequest(id).then(resolve).catch(reject);
      }
    );
  });
}

function deleteVisaRequest(id) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    db.run(
      'UPDATE visa_requests SET status = "cancelled", updated_at = ? WHERE id = ?',
      [timestamp, id],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

function uploadVisaDocument(requestId, documentData) {
  return new Promise((resolve, reject) => {
    getVisaRequest(requestId)
      .then(visa => {
        if (!visa) {
          reject(new Error('Visa request not found'));
          return;
        }

        const documents = visa.documents || [];
        const newDoc = {
          id: generateUUID(),
          ...documentData,
          uploadedAt: getCurrentTimestamp()
        };
        documents.push(newDoc);

        updateVisaRequest(requestId, { documents })
          .then(updated => resolve(newDoc))
          .catch(reject);
      })
      .catch(reject);
  });
}

function updateVisaStatus(requestId, status, notes) {
  return new Promise((resolve, reject) => {
    const updates = { status };
    if (notes) {
      updates.agentNotes = notes;
    }
    updateVisaRequest(requestId, updates).then(resolve).catch(reject);
  });
}

// ============================================
// TRANSFER REQUESTS
// ============================================

function createTransferRequest(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO transfer_requests (
        id, customer_id, requested_by_user_id, transfer_type, pickup_location,
        dropoff_location, pickup_date, pickup_time, total_passengers, vehicle_type,
        vehicle_count, special_requests, has_luggage, needs_child_seat,
        is_wheelchair_accessible, status, assigned_driver_data, assigned_vehicle_data,
        estimated_price, final_price, currency, agent_notes, driver_notes,
        created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.customerId, data.requestedByUserId, data.transferType, data.pickupLocation,
        data.dropoffLocation, data.pickupDate, data.pickupTime, data.totalPassengers,
        data.vehicleType, data.vehicleCount || 1, data.specialRequests,
        data.hasLuggage ? 1 : 0, data.needsChildSeat ? 1 : 0, data.isWheelchairAccessible ? 1 : 0,
        data.status || 'pending',
        data.assignedDriver ? JSON.stringify(data.assignedDriver) : null,
        data.assignedVehicle ? JSON.stringify(data.assignedVehicle) : null,
        data.estimatedPrice, data.finalPrice, data.currency || 'TRY',
        data.agentNotes, data.driverNotes, timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getTransferRequest(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM transfer_requests WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else {
        const transfer = rowToCamelCase(row);
        if (transfer) {
          if (transfer.assignedDriverData) transfer.assignedDriver = JSON.parse(transfer.assignedDriverData);
          if (transfer.assignedVehicleData) transfer.assignedVehicle = JSON.parse(transfer.assignedVehicleData);
        }
        resolve(transfer);
      }
    });
  });
}

function listTransferRequests(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { customerId, status, transferType } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (customerId) {
      whereClause.push('customer_id = ?');
      params.push(customerId);
    }
    if (status) {
      whereClause.push('status = ?');
      params.push(status);
    }
    if (transferType) {
      whereClause.push('transfer_type = ?');
      params.push(transferType);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM transfer_requests ${where}`;
    const dataQuery = `SELECT * FROM transfer_requests ${where} ORDER BY pickup_date DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else {
          const transfers = rows.map(row => {
            const transfer = rowToCamelCase(row);
            if (transfer.assignedDriverData) transfer.assignedDriver = JSON.parse(transfer.assignedDriverData);
            if (transfer.assignedVehicleData) transfer.assignedVehicle = JSON.parse(transfer.assignedVehicleData);
            return transfer;
          });
          resolve({
            data: transfers,
            total: countRow.total,
            limit,
            offset
          });
        }
      });
    });
  });
}

function updateTransferRequest(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        let dbValue = value;
        if (key === 'assignedDriver') {
          dbValue = JSON.stringify(value);
          fields.push('assigned_driver_data = ?');
        } else if (key === 'assignedVehicle') {
          dbValue = JSON.stringify(value);
          fields.push('assigned_vehicle_data = ?');
        } else {
          fields.push(`${toSnakeCase(key)} = ?`);
        }
        values.push(dbValue);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE transfer_requests SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getTransferRequest(id).then(resolve).catch(reject);
      }
    );
  });
}

function deleteTransferRequest(id) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    db.run(
      'UPDATE transfer_requests SET status = "cancelled", updated_at = ? WHERE id = ?',
      [timestamp, id],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

// ============================================
// TRANSFER PASSENGERS
// ============================================

function addPassengerToTransfer(transferRequestId, passengerData) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    db.run(
      `INSERT INTO transfer_passengers (
        id, transfer_request_id, first_name, last_name, phone, is_child, age, special_needs
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, transferRequestId, passengerData.firstName, passengerData.lastName,
        passengerData.phone, passengerData.isChild ? 1 : 0, passengerData.age,
        passengerData.specialNeeds
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, transferRequestId, ...passengerData });
      }
    );
  });
}

// ============================================
// INVOICES
// ============================================

function createInvoice(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO invoices (
        id, invoice_number, customer_id, issue_date, due_date, amount, currency,
        tax_amount, total_amount, status, payment_status, paid_amount, notes,
        created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.invoiceNumber, data.customerId, data.issueDate, data.dueDate,
        data.amount, data.currency || 'TRY', data.taxAmount || 0, data.totalAmount,
        data.status || 'draft', data.paymentStatus || 'unpaid', data.paidAmount || 0,
        data.notes, timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getInvoice(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM invoices WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else resolve(rowToCamelCase(row));
    });
  });
}

function listInvoices(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { customerId, status, paymentStatus } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (customerId) {
      whereClause.push('customer_id = ?');
      params.push(customerId);
    }
    if (status) {
      whereClause.push('status = ?');
      params.push(status);
    }
    if (paymentStatus) {
      whereClause.push('payment_status = ?');
      params.push(paymentStatus);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM invoices ${where}`;
    const dataQuery = `SELECT * FROM invoices ${where} ORDER BY issue_date DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else resolve({
          data: rows.map(rowToCamelCase),
          total: countRow.total,
          limit,
          offset
        });
      });
    });
  });
}

function updateInvoice(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(value);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE invoices SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getInvoice(id).then(resolve).catch(reject);
      }
    );
  });
}

function deleteInvoice(id) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    db.run(
      'UPDATE invoices SET status = "cancelled", updated_at = ? WHERE id = ?',
      [timestamp, id],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

function addInvoiceItem(invoiceId, itemData) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    db.run(
      `INSERT INTO invoice_items (
        id, invoice_id, service_type, service_id, description, quantity,
        unit_price, total_price, tax_rate, tax_amount
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, invoiceId, itemData.serviceType, itemData.serviceId, itemData.description,
        itemData.quantity || 1, itemData.unitPrice, itemData.totalPrice,
        itemData.taxRate || 0, itemData.taxAmount || 0
      ],
      function(err) {
        if (err) reject(err);
        else {
          // Recalculate invoice total
          calculateInvoiceTotal(invoiceId)
            .then(() => resolve({ id, invoiceId, ...itemData }))
            .catch(reject);
        }
      }
    );
  });
}

function recordPayment(invoiceId, paymentData) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();

    getInvoice(invoiceId)
      .then(invoice => {
        if (!invoice) {
          reject(new Error('Invoice not found'));
          return;
        }

        const newPaidAmount = (invoice.paidAmount || 0) + paymentData.amount;
        const paymentStatus = newPaidAmount >= invoice.totalAmount ? 'paid' : 'partial';

        updateInvoice(invoiceId, {
          paidAmount: newPaidAmount,
          paidDate: paymentData.paidDate || timestamp,
          paymentMethod: paymentData.paymentMethod,
          paymentStatus
        }).then(resolve).catch(reject);
      })
      .catch(reject);
  });
}

function calculateInvoiceTotal(invoiceId) {
  return new Promise((resolve, reject) => {
    db.all(
      'SELECT SUM(total_price) as amount, SUM(tax_amount) as tax FROM invoice_items WHERE invoice_id = ?',
      [invoiceId],
      (err, rows) => {
        if (err) {
          reject(err);
          return;
        }

        const totals = rows[0];
        const amount = totals.amount || 0;
        const taxAmount = totals.tax || 0;
        const totalAmount = amount + taxAmount;

        updateInvoice(invoiceId, { amount, taxAmount, totalAmount })
          .then(resolve)
          .catch(reject);
      }
    );
  });
}

// ============================================
// INVOICE ITEMS
// ============================================

function getInvoiceItems(invoiceId) {
  return new Promise((resolve, reject) => {
    db.all(
      'SELECT * FROM invoice_items WHERE invoice_id = ? ORDER BY id',
      [invoiceId],
      (err, rows) => {
        if (err) reject(err);
        else resolve(rows.map(rowToCamelCase));
      }
    );
  });
}

// ============================================
// REVENUE RECORDS
// ============================================

function createRevenueRecord(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO revenue_records (
        id, type, customer_id, requested_by_user_id, description, amount, currency,
        commission, net_amount, reference_id, reference_type, status, payment_status,
        invoice_number, service_date, notes, created_at, updated_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.type, data.customerId, data.requestedByUserId, data.description,
        data.amount, data.currency || 'TRY', data.commission, data.netAmount,
        data.referenceId, data.referenceType, data.status || 'pending',
        data.paymentStatus || 'unpaid', data.invoiceNumber, data.serviceDate,
        data.notes, timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, updatedAt: timestamp });
      }
    );
  });
}

function getRevenueRecord(id) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM revenue_records WHERE id = ?', [id], (err, row) => {
      if (err) reject(err);
      else resolve(rowToCamelCase(row));
    });
  });
}

function listRevenueRecords(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { type, customerId, status, paymentStatus, dateFrom, dateTo } = filters;
    const { limit = 50, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (type) {
      whereClause.push('type = ?');
      params.push(type);
    }
    if (customerId) {
      whereClause.push('customer_id = ?');
      params.push(customerId);
    }
    if (status) {
      whereClause.push('status = ?');
      params.push(status);
    }
    if (paymentStatus) {
      whereClause.push('payment_status = ?');
      params.push(paymentStatus);
    }
    if (dateFrom) {
      whereClause.push('service_date >= ?');
      params.push(dateFrom);
    }
    if (dateTo) {
      whereClause.push('service_date <= ?');
      params.push(dateTo);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM revenue_records ${where}`;
    const dataQuery = `SELECT * FROM revenue_records ${where} ORDER BY service_date DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else resolve({
          data: rows.map(rowToCamelCase),
          total: countRow.total,
          limit,
          offset
        });
      });
    });
  });
}

function updateRevenueRecord(id, updates) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    const fields = [];
    const values = [];

    for (const [key, value] of Object.entries(updates)) {
      if (key !== 'id' && key !== 'createdAt') {
        fields.push(`${toSnakeCase(key)} = ?`);
        values.push(value);
      }
    }

    fields.push('updated_at = ?');
    values.push(timestamp);
    values.push(id);

    db.run(
      `UPDATE revenue_records SET ${fields.join(', ')} WHERE id = ?`,
      values,
      function(err) {
        if (err) reject(err);
        else getRevenueRecord(id).then(resolve).catch(reject);
      }
    );
  });
}

function getRevenueByPeriod(startDate, endDate, groupBy = 'day') {
  return new Promise((resolve, reject) => {
    let dateFormat;
    switch (groupBy) {
      case 'month':
        dateFormat = '%Y-%m';
        break;
      case 'year':
        dateFormat = '%Y';
        break;
      default:
        dateFormat = '%Y-%m-%d';
    }

    db.all(
      `SELECT
        strftime('${dateFormat}', service_date) as period,
        type,
        COUNT(*) as count,
        SUM(amount) as total_amount,
        SUM(net_amount) as total_net,
        currency
       FROM revenue_records
       WHERE service_date >= ? AND service_date <= ?
       AND status = 'completed'
       GROUP BY period, type, currency
       ORDER BY period DESC, type`,
      [startDate, endDate],
      (err, rows) => {
        if (err) reject(err);
        else resolve(rows.map(rowToCamelCase));
      }
    );
  });
}

function getRevenueByCustomer(customerId, period = 'all') {
  return new Promise((resolve, reject) => {
    let dateFilter = '';
    const params = [customerId];

    if (period === 'month') {
      dateFilter = "AND service_date >= date('now', 'start of month')";
    } else if (period === 'year') {
      dateFilter = "AND service_date >= date('now', 'start of year')";
    }

    db.all(
      `SELECT
        type,
        COUNT(*) as count,
        SUM(amount) as total_amount,
        SUM(net_amount) as total_net,
        currency
       FROM revenue_records
       WHERE customer_id = ? ${dateFilter}
       AND status = 'completed'
       GROUP BY type, currency
       ORDER BY type`,
      params,
      (err, rows) => {
        if (err) reject(err);
        else resolve(rows.map(rowToCamelCase));
      }
    );
  });
}

function getRevenueByService(serviceType, period = 'all') {
  return new Promise((resolve, reject) => {
    let dateFilter = '';
    const params = [serviceType];

    if (period === 'month') {
      dateFilter = "AND service_date >= date('now', 'start of month')";
    } else if (period === 'year') {
      dateFilter = "AND service_date >= date('now', 'start of year')";
    }

    db.all(
      `SELECT
        strftime('%Y-%m-%d', service_date) as date,
        COUNT(*) as count,
        SUM(amount) as total_amount,
        SUM(net_amount) as total_net,
        currency
       FROM revenue_records
       WHERE type = ? ${dateFilter}
       AND status = 'completed'
       GROUP BY date, currency
       ORDER BY date DESC`,
      params,
      (err, rows) => {
        if (err) reject(err);
        else resolve(rows.map(rowToCamelCase));
      }
    );
  });
}

// ============================================
// USER ACTIVITIES (Audit Log)
// ============================================

function createUserActivity(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO user_activities (
        id, user_id, action, description, entity_type, entity_id, ip_address, user_agent, created_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.userId, data.action, data.description, data.entityType,
        data.entityId, data.ipAddress, data.userAgent, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp });
      }
    );
  });
}

function listUserActivities(filters = {}, pagination = {}) {
  return new Promise((resolve, reject) => {
    const { userId, action, entityType, dateFrom, dateTo } = filters;
    const { limit = 100, offset = 0 } = pagination;

    let whereClause = [];
    let params = [];

    if (userId) {
      whereClause.push('user_id = ?');
      params.push(userId);
    }
    if (action) {
      whereClause.push('action = ?');
      params.push(action);
    }
    if (entityType) {
      whereClause.push('entity_type = ?');
      params.push(entityType);
    }
    if (dateFrom) {
      whereClause.push('created_at >= ?');
      params.push(dateFrom);
    }
    if (dateTo) {
      whereClause.push('created_at <= ?');
      params.push(dateTo);
    }

    const where = whereClause.length > 0 ? `WHERE ${whereClause.join(' AND ')}` : '';
    const countQuery = `SELECT COUNT(*) as total FROM user_activities ${where}`;
    const dataQuery = `SELECT * FROM user_activities ${where} ORDER BY created_at DESC LIMIT ? OFFSET ?`;

    db.get(countQuery, params, (err, countRow) => {
      if (err) {
        reject(err);
        return;
      }

      db.all(dataQuery, [...params, limit, offset], (err, rows) => {
        if (err) reject(err);
        else resolve({
          data: rows.map(rowToCamelCase),
          total: countRow.total,
          limit,
          offset
        });
      });
    });
  });
}

// ============================================
// USER SESSIONS
// ============================================

function createUserSession(data) {
  return new Promise((resolve, reject) => {
    const id = generateUUID();
    const timestamp = getCurrentTimestamp();

    db.run(
      `INSERT INTO user_sessions (
        id, user_id, token, ip_address, user_agent, is_active, expires_at, created_at, last_activity_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        id, data.userId, data.token, data.ipAddress, data.userAgent,
        1, data.expiresAt, timestamp, timestamp
      ],
      function(err) {
        if (err) reject(err);
        else resolve({ id, ...data, createdAt: timestamp, lastActivityAt: timestamp });
      }
    );
  });
}

function getUserSession(token) {
  return new Promise((resolve, reject) => {
    db.get('SELECT * FROM user_sessions WHERE token = ? AND is_active = 1', [token], (err, row) => {
      if (err) reject(err);
      else resolve(rowToCamelCase(row));
    });
  });
}

function updateSessionActivity(token) {
  return new Promise((resolve, reject) => {
    const timestamp = getCurrentTimestamp();
    db.run(
      'UPDATE user_sessions SET last_activity_at = ? WHERE token = ?',
      [timestamp, token],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

function invalidateSession(token) {
  return new Promise((resolve, reject) => {
    db.run(
      'UPDATE user_sessions SET is_active = 0 WHERE token = ?',
      [token],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes > 0);
      }
    );
  });
}

function cleanExpiredSessions() {
  return new Promise((resolve, reject) => {
    const now = getCurrentTimestamp();
    db.run(
      'UPDATE user_sessions SET is_active = 0 WHERE expires_at < ? AND is_active = 1',
      [now],
      function(err) {
        if (err) reject(err);
        else resolve(this.changes);
      }
    );
  });
}

// ============================================
// DASHBOARD AND REPORTS
// ============================================

function getDashboardStats() {
  return new Promise((resolve, reject) => {
    const today = new Date().toISOString().split('T')[0];

    db.get(
      `SELECT
        (SELECT COUNT(*) FROM checkin_tickets WHERE DATE(departure_date) = ? AND checkin_status = 'completed') as today_checkins,
        (SELECT COUNT(*) FROM checkin_tickets WHERE DATE(departure_date) = ? AND checkin_status IN ('waiting', 'ready')) as today_scheduled,
        (SELECT COUNT(*) FROM ticket_requests WHERE status = 'pending') as pending_requests,
        (SELECT COUNT(*) FROM ticket_requests WHERE DATE(created_at) = ? AND status = 'ticketed') as ticketed_today,
        (SELECT COUNT(*) FROM checkin_tickets WHERE departure_date >= ? AND checkin_status IN ('waiting', 'ready')) as upcoming_flights,
        (SELECT COUNT(*) FROM checkin_tickets WHERE DATE(last_checkin_attempt) = ? AND checkin_status = 'failed') as failed_checkins,
        (SELECT COUNT(*) FROM ticket_requests) as total_ticket_requests`,
      [today, today, today, today, today],
      (err, row) => {
        if (err) {
          reject(err);
          return;
        }

        const stats = rowToCamelCase(row);
        const totalAttempts = (stats.todayCheckins || 0) + (stats.failedCheckins || 0);
        stats.successRate = totalAttempts > 0 ? ((stats.todayCheckins || 0) / totalAttempts * 100) : 0;

        resolve(stats);
      }
    );
  });
}

function getAccountingStats(period = 'month') {
  return new Promise((resolve, reject) => {
    let dateFilter = "date('now', 'start of month')";
    if (period === 'year') {
      dateFilter = "date('now', 'start of year')";
    } else if (period === 'week') {
      dateFilter = "date('now', 'weekday 0', '-7 days')";
    }

    db.get(
      `SELECT
        (SELECT COALESCE(SUM(amount), 0) FROM revenue_records WHERE service_date >= ${dateFilter} AND status = 'completed') as total_revenue,
        (SELECT COALESCE(SUM(net_amount), 0) FROM revenue_records WHERE service_date >= ${dateFilter} AND status = 'completed') as net_revenue,
        (SELECT COALESCE(SUM(total_amount), 0) FROM invoices WHERE payment_status = 'unpaid') as pending_payments,
        (SELECT COUNT(*) FROM revenue_records WHERE service_date >= ${dateFilter} AND status = 'completed') as completed_transactions,
        (SELECT COALESCE(SUM(amount), 0) FROM revenue_records WHERE service_date >= ${dateFilter} AND type = 'ticket' AND status = 'completed') as ticket_revenue,
        (SELECT COALESCE(SUM(amount), 0) FROM revenue_records WHERE service_date >= ${dateFilter} AND type = 'accommodation' AND status = 'completed') as accommodation_revenue,
        (SELECT COALESCE(SUM(amount), 0) FROM revenue_records WHERE service_date >= ${dateFilter} AND type = 'visa' AND status = 'completed') as visa_revenue,
        (SELECT COALESCE(SUM(amount), 0) FROM revenue_records WHERE service_date >= ${dateFilter} AND type = 'transfer' AND status = 'completed') as transfer_revenue`,
      [],
      (err, row) => {
        if (err) reject(err);
        else resolve(rowToCamelCase(row));
      }
    );
  });
}

// ============================================
// CHECKIN LOGS (Existing functions)
// ============================================

function saveLog(logData) {
  return new Promise((resolve, reject) => {
    const { jobId, pnr, airline, percentage, level, message, details } = logData;

    db.run(
      `INSERT INTO checkin_logs (job_id, pnr, airline, percentage, level, message, details)
       VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [jobId, pnr, airline, percentage || 0, level || 'info', message, details || null],
      function(err) {
        if (err) {
          console.error('❌ Error saving log:', err);
          reject(err);
        } else {
          resolve(this.lastID);
        }
      }
    );
  });
}

function getLogs(jobId) {
  return new Promise((resolve, reject) => {
    db.all(
      `SELECT * FROM checkin_logs
       WHERE job_id = ?
       ORDER BY created_at ASC`,
      [jobId],
      (err, rows) => {
        if (err) {
          console.error('❌ Error fetching logs:', err);
          reject(err);
        } else {
          resolve(rows);
        }
      }
    );
  });
}

function getLogsByPnr(pnr) {
  return new Promise((resolve, reject) => {
    db.all(
      `SELECT * FROM checkin_logs
       WHERE pnr = ?
       ORDER BY created_at DESC
       LIMIT 100`,
      [pnr],
      (err, rows) => {
        if (err) {
          console.error('❌ Error fetching logs by PNR:', err);
          reject(err);
        } else {
          resolve(rows);
        }
      }
    );
  });
}

function cleanOldLogs() {
  return new Promise((resolve, reject) => {
    db.run(
      `DELETE FROM checkin_logs
       WHERE created_at < datetime('now', '-30 days')`,
      function(err) {
        if (err) {
          console.error('❌ Error cleaning old logs:', err);
          reject(err);
        } else {
          console.log(`✅ Cleaned ${this.changes} old log entries`);
          resolve(this.changes);
        }
      }
    );
  });
}

function updateJobStatus(jobId, status, details = null) {
  return new Promise((resolve, reject) => {
    const message = status === 'completed' ? 'Check-in tamamlandı' :
                    status === 'failed' ? 'Check-in başarısız' :
                    status === 'processing' ? 'Check-in işleniyor...' : 'Check-in bekleniyor';

    db.run(
      `INSERT INTO checkin_logs (job_id, pnr, airline, percentage, level, message, details)
       VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [jobId, 'JOB', 'SYSTEM', status === 'completed' ? 100 : 0,
       status === 'failed' ? 'error' : 'info', message, details],
      function(err) {
        if (err) {
          console.error('❌ Error updating job status:', err);
          reject(err);
        } else {
          resolve(this.lastID);
        }
      }
    );
  });
}

function getLastLog(pnr, jobId) {
  return new Promise((resolve, reject) => {
    db.get(
      `SELECT * FROM checkin_logs
       WHERE pnr = ? AND job_id = ?
       ORDER BY created_at DESC
       LIMIT 1`,
      [pnr, jobId],
      (err, row) => {
        if (err) {
          console.error('❌ Error fetching last log:', err);
          reject(err);
        } else {
          resolve(row);
        }
      }
    );
  });
}

function getActiveJobs() {
  return new Promise((resolve, reject) => {
    db.all(
      `SELECT DISTINCT job_id, pnr, airline,
       MAX(created_at) as last_update,
       (SELECT message FROM checkin_logs cl2
        WHERE cl2.job_id = checkin_logs.job_id
        ORDER BY created_at DESC LIMIT 1) as last_message,
       (SELECT percentage FROM checkin_logs cl3
        WHERE cl3.job_id = checkin_logs.job_id
        ORDER BY created_at DESC LIMIT 1) as progress
       FROM checkin_logs
       WHERE created_at > datetime('now', '-1 hour')
       GROUP BY job_id
       ORDER BY last_update DESC`,
      [],
      (err, rows) => {
        if (err) {
          console.error('❌ Error fetching active jobs:', err);
          reject(err);
        } else {
          resolve(rows);
        }
      }
    );
  });
}

// Günlük temizleme işlemi - her gün 03:00'de çalışır
setInterval(() => {
  const now = new Date();
  if (now.getHours() === 3 && now.getMinutes() === 0) {
    cleanOldLogs();
    cleanExpiredSessions();
  }
}, 60000); // Her dakika kontrol et

// ============================================
// EXPORTS
// ============================================

module.exports = {
  db,
  get: (sql, params) => new Promise((resolve, reject) => {
    db.get(sql, params, (err, row) => {
      if (err) reject(err);
      else resolve(row);
    });
  }),
  all: (sql, params) => new Promise((resolve, reject) => {
    db.all(sql, params, (err, rows) => {
      if (err) reject(err);
      else resolve(rows);
    });
  }),
  run: (sql, params) => new Promise((resolve, reject) => {
    db.run(sql, params, function(err) {
      if (err) reject(err);
      else resolve({ lastID: this.lastID, changes: this.changes });
    });
  }),

  // Utility
  generateUUID,
  getCurrentTimestamp,

  // Customers
  createCustomer,
  getCustomer,
  listCustomers,
  updateCustomer,
  deleteCustomer,
  searchCustomers,
  getCustomerStats,

  // Corporate Users
  createCorporateUser,
  getCorporateUser,
  listCorporateUsers,
  updateCorporateUser,
  deleteCorporateUser,

  // Agency Users
  createAgencyUser,
  getAgencyUser,
  listAgencyUsers,
  updateAgencyUser,
  deleteAgencyUser,

  // Departments
  createDepartment,
  getDepartment,
  listDepartments,
  updateDepartment,
  deleteDepartment,

  // Passengers
  createPassenger,
  getPassenger,
  listPassengers,
  updatePassenger,
  deletePassenger,

  // Ticket Requests
  createTicketRequest,
  getTicketRequest,
  listTicketRequests,
  updateTicketRequest,
  deleteTicketRequest,
  assignPnrToTicket,
  getTicketWithPassengers,
  getUpcomingTickets,
  addPassengerToTicket,
  removePassengerFromTicket,

  // Checkin Tickets
  createCheckinTicket,
  getCheckinTicket,
  listCheckinTickets,
  updateCheckinTicket,
  deleteCheckinTicket,
  scheduleAutoCheckin,
  getUpcomingCheckins,
  addPassengerToCheckin,
  updateCheckinPassengerResult,

  // Auto Checkin Jobs
  createAutoCheckinJob,
  getAutoCheckinJob,
  listAutoCheckinJobs,
  updateAutoCheckinJob,

  // Flights
  createFlight,
  getFlight,
  listFlights,
  updateFlight,

  // Accommodation Requests
  createAccommodationRequest,
  getAccommodationRequest,
  listAccommodationRequests,
  updateAccommodationRequest,
  deleteAccommodationRequest,
  getAccommodationWithGuests,
  updateAccommodationStatus,
  addGuestToAccommodation,

  // Visa Requests
  createVisaRequest,
  getVisaRequest,
  listVisaRequests,
  updateVisaRequest,
  deleteVisaRequest,
  uploadVisaDocument,
  updateVisaStatus,

  // Transfer Requests
  createTransferRequest,
  getTransferRequest,
  listTransferRequests,
  updateTransferRequest,
  deleteTransferRequest,
  addPassengerToTransfer,

  // Invoices
  createInvoice,
  getInvoice,
  listInvoices,
  updateInvoice,
  deleteInvoice,
  addInvoiceItem,
  recordPayment,
  calculateInvoiceTotal,
  getInvoiceItems,

  // Revenue Records
  createRevenueRecord,
  getRevenueRecord,
  listRevenueRecords,
  updateRevenueRecord,
  getRevenueByPeriod,
  getRevenueByCustomer,
  getRevenueByService,

  // User Activities
  createUserActivity,
  listUserActivities,

  // User Sessions
  createUserSession,
  getUserSession,
  updateSessionActivity,
  invalidateSession,
  cleanExpiredSessions,

  // Dashboard and Reports
  getDashboardStats,
  getAccountingStats,

  // Checkin Logs (existing)
  saveLog,
  getLogs,
  getLogsByPnr,
  cleanOldLogs,
  updateJobStatus,
  getLastLog,
  getActiveJobs
};
} // Close the else block from line 11
