'use client';

import React, { useState, useEffect } from 'react';
import {
  AlertCircle,
  ArrowLeft,
  Building2,
  Calendar,
  Hotel,
  Plus,
  Save,
  Trash2,
  User,
  Users
} from 'lucide-react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';
import type { Customer, AccommodationGuest, RoomType, MealPlan, HotelCategory } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

interface FormData {
  customerId: string;
  requestedBy?: string;
  destination: string;
  checkInDate: string;
  checkOutDate: string;
  guests: AccommodationGuest[];
  roomType: RoomType;
  roomCount: number;
  budgetMin?: number;
  budgetMax?: number;
  currency: 'TRY' | 'USD' | 'EUR';
  specialRequests?: string;
  mealPlan?: MealPlan;
  hotelCategory?: HotelCategory;
  agentNotes?: string;
  priority: 'low' | 'medium' | 'high' | 'urgent';
}

export default function NewAccommodationRequestPage() {
  return (
    <PermissionGuard requiredPermissions={['accommodation_view']}>
      <NewAccommodationRequestPageContent />
    </PermissionGuard>
  );
}

function NewAccommodationRequestPageContent() {
  const router = useRouter();
  const [customers, setCustomers] = useState<Customer[]>([]);
  const [corporateUsers, setCorporateUsers] = useState<Array<{ id: string; firstName: string; lastName: string; email: string }>>([]);
  const [loading, setLoading] = useState(false);
  const [errors, setErrors] = useState<Record<string, string>>({});
  
  const [formData, setFormData] = useState<FormData>({
    customerId: '',
    destination: '',
    checkInDate: '',
    checkOutDate: '',
    guests: [
      {
        id: '1',
        firstName: '',
        lastName: '',
        isChild: false
      }
    ],
    roomType: 'double',
    roomCount: 1,
    currency: 'TRY',
    priority: 'medium'
  });

  useEffect(() => {
    fetchCustomers();
  }, []);

  useEffect(() => {
    // Gece sayısını hesapla
    if (formData.checkInDate && formData.checkOutDate) {
      const checkIn = new Date(formData.checkInDate);
      const checkOut = new Date(formData.checkOutDate);
      const nights = Math.ceil((checkOut.getTime() - checkIn.getTime()) / (1000 * 60 * 60 * 24));
      console.log('Nights:', nights);
    }
  }, [formData.checkInDate, formData.checkOutDate]);

  const fetchCustomers = async () => {
    try {
      // Mock customers data
      const mockCustomers: Customer[] = [
        {
          id: 'c1',
          type: 'individual',
          firstName: 'Ahmet',
          lastName: 'Yılmaz',
          phone: '+905551234567',
          email: 'ahmet@email.com',
          isVip: false,
          isActive: true,
          createdAt: new Date(),
          updatedAt: new Date()
        },
        {
          id: 'c2',
          type: 'corporate',
          companyName: 'ABC Teknoloji A.Ş.',
          phone: '+905559876543',
          email: 'info@abcteknoloji.com',
          isVip: true,
          isActive: true,
          createdAt: new Date(),
          updatedAt: new Date()
        }
      ];
      
      setCustomers(mockCustomers);
    } catch (error) {
      console.error('Error fetching customers:', error);
    }
  };

  const fetchCorporateUsers = async (customerId: string) => {
    try {
      // Mock corporate users
      const mockUsers = [
        { id: 'u1', firstName: 'Mehmet', lastName: 'Özkan', email: 'mehmet@abcteknoloji.com' },
        { id: 'u2', firstName: 'Ayşe', lastName: 'Demir', email: 'ayse@abcteknoloji.com' }
      ];
      setCorporateUsers(mockUsers);
    } catch (error) {
      console.error('Error fetching corporate users:', error);
    }
  };

  const handleInputChange = (field: keyof FormData, value: string | number | undefined | MealPlan | HotelCategory | RoomType) => {
    setFormData(prev => ({ ...prev, [field]: value }));

    // Clear error when user starts typing
    if (errors[field]) {
      const newErrors = { ...errors };
      delete newErrors[field];
      setErrors(newErrors);
    }

    // Fetch corporate users when corporate customer is selected
    if (field === 'customerId' && typeof value === 'string') {
      const customer = customers.find(c => c.id === value);
      if (customer?.type === 'corporate') {
        fetchCorporateUsers(value);
      } else {
        setCorporateUsers([]);
        setFormData(prev => ({ ...prev, requestedBy: undefined }));
      }
    }
  };

  const handleGuestChange = (index: number, field: keyof AccommodationGuest, value: string | boolean | number) => {
    const newGuests = [...formData.guests];
    newGuests[index] = { ...newGuests[index], [field]: value };
    setFormData(prev => ({ ...prev, guests: newGuests }));
  };

  const addGuest = () => {
    const newGuest: AccommodationGuest = {
      id: Date.now().toString(),
      firstName: '',
      lastName: '',
      isChild: false
    };
    setFormData(prev => ({
      ...prev,
      guests: [...prev.guests, newGuest]
    }));
  };

  const removeGuest = (index: number) => {
    if (formData.guests.length > 1) {
      const newGuests = formData.guests.filter((_, i) => i !== index);
      setFormData(prev => ({ ...prev, guests: newGuests }));
    }
  };

  const validateForm = (): boolean => {
    const newErrors: Record<string, string> = {};

    if (!formData.customerId) {
      newErrors.customerId = 'Müşteri seçimi gereklidir';
    }

    if (!formData.destination) {
      newErrors.destination = 'Destinasyon gereklidir';
    }

    if (!formData.checkInDate) {
      newErrors.checkInDate = 'Giriş tarihi gereklidir';
    }

    if (!formData.checkOutDate) {
      newErrors.checkOutDate = 'Çıkış tarihi gereklidir';
    }

    if (formData.checkInDate && formData.checkOutDate) {
      const checkIn = new Date(formData.checkInDate);
      const checkOut = new Date(formData.checkOutDate);
      if (checkOut <= checkIn) {
        newErrors.checkOutDate = 'Çıkış tarihi giriş tarihinden sonra olmalıdır';
      }
    }

    // Misafir validasyonu
    formData.guests.forEach((guest, index) => {
      if (!guest.firstName) {
        newErrors[`guest_${index}_firstName`] = 'Misafir adı gereklidir';
      }
      if (!guest.lastName) {
        newErrors[`guest_${index}_lastName`] = 'Misafir soyadı gereklidir';
      }
    });

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!validateForm()) {
      return;
    }

    try {
      setLoading(true);
      
      // Calculate nights and total guests
      const checkIn = new Date(formData.checkInDate);
      const checkOut = new Date(formData.checkOutDate);
      const nights = Math.ceil((checkOut.getTime() - checkIn.getTime()) / (1000 * 60 * 60 * 24));
      const totalGuests = formData.guests.length;

      const requestData = {
        ...formData,
        nights,
        totalGuests
      };
      
      // Mock API call
      console.log('Creating accommodation request:', requestData);
      
      // Simulate API delay
      await new Promise(resolve => setTimeout(resolve, 1000));
      
      // Redirect to accommodation requests list
      router.push('/accommodation/requests');
    } catch (error) {
      console.error('Error creating accommodation request:', error);
    } finally {
      setLoading(false);
    }
  };

  const selectedCustomer = customers.find(c => c.id === formData.customerId);

  const destinationOptions = [
    'İstanbul', 'Antalya', 'Bodrum', 'Çeşme', 'Kapadokya', 'Pamukkale',
    'Marmaris', 'Kuşadası', 'Fethiye', 'Kas', 'Kalkan', 'Side',
    'Belek', 'Kemer', 'Alanya', 'Trabzon', 'Rize', 'Artvin'
  ];

  const roomTypeOptions = [
    { value: 'single', label: 'Tek Kişilik' },
    { value: 'double', label: 'Çift Kişilik' },
    { value: 'twin', label: 'İki Yataklı' },
    { value: 'triple', label: 'Üç Kişilik' },
    { value: 'family', label: 'Aile Odası' },
    { value: 'suite', label: 'Suit' },
    { value: 'connecting', label: 'Bağlantılı Odalar' }
  ];

  const mealPlanOptions = [
    { value: 'room_only', label: 'Sadece Oda' },
    { value: 'breakfast', label: 'Kahvaltı Dahil' },
    { value: 'half_board', label: 'Yarım Pansiyon' },
    { value: 'full_board', label: 'Tam Pansiyon' },
    { value: 'all_inclusive', label: 'Her Şey Dahil' }
  ];

  const hotelCategoryOptions = [
    { value: 'budget', label: 'Ekonomik' },
    { value: 'standard', label: 'Standart' },
    { value: 'comfort', label: 'Konforlu' },
    { value: 'luxury', label: 'Lüks' },
    { value: 'boutique', label: 'Butik' }
  ];

  const calculateNights = () => {
    if (formData.checkInDate && formData.checkOutDate) {
      const checkIn = new Date(formData.checkInDate);
      const checkOut = new Date(formData.checkOutDate);
      const nights = Math.ceil((checkOut.getTime() - checkIn.getTime()) / (1000 * 60 * 60 * 24));
      return nights > 0 ? nights : 0;
    }
    return 0;
  };

  return (
    <div className="p-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center space-x-4">
          <Link
            href="/accommodation/requests"
            className="flex items-center text-gray-600 hover:text-gray-900"
          >
            <ArrowLeft className="h-5 w-5 mr-2" />
            Konaklama Taleplerine Dön
          </Link>
          <div className="h-6 border-l border-gray-300"></div>
          <h1 className="text-2xl font-bold text-gray-900 flex items-center">
            <Hotel className="mr-3 h-8 w-8 text-purple-600" />
            Yeni Konaklama Talebi
          </h1>
        </div>
      </div>

      <form onSubmit={handleSubmit} className="space-y-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Sol Panel - Müşteri ve Misafir Bilgileri */}
          <div className="lg:col-span-2 space-y-6">
            {/* Müşteri Seçimi */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h2 className="text-lg font-medium text-gray-900 mb-4 flex items-center">
                <User className="h-5 w-5 mr-2" />
                Müşteri Bilgileri
              </h2>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Müşteri *
                  </label>
                  <select
                    value={formData.customerId}
                    onChange={(e) => handleInputChange('customerId', e.target.value)}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.customerId ? 'border-red-300' : 'border-gray-300'
                    }`}
                  >
                    <option value="">Müşteri seçiniz</option>
                    {customers.map((customer) => (
                      <option key={customer.id} value={customer.id}>
                        {customer.type === 'individual' 
                          ? `${customer.firstName} ${customer.lastName}`
                          : customer.companyName
                        } - {customer.phone}
                      </option>
                    ))}
                  </select>
                  {errors.customerId && (
                    <p className="mt-1 text-sm text-red-600 flex items-center">
                      <AlertCircle className="h-4 w-4 mr-1" />
                      {errors.customerId}
                    </p>
                  )}
                </div>

                {selectedCustomer?.type === 'corporate' && corporateUsers.length > 0 && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Talep Eden Kullanıcı
                    </label>
                    <select
                      value={formData.requestedBy || ''}
                      onChange={(e) => handleInputChange('requestedBy', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    >
                      <option value="">Kullanıcı seçiniz (isteğe bağlı)</option>
                      {corporateUsers.map((user) => (
                        <option key={user.id} value={user.id}>
                          {user.firstName} {user.lastName} - {user.email}
                        </option>
                      ))}
                    </select>
                  </div>
                )}
              </div>

              {selectedCustomer && (
                <div className="mt-4 p-3 bg-gray-50 rounded-lg">
                  <div className="flex items-center">
                    {selectedCustomer.type === 'corporate' ? (
                      <Building2 className="h-5 w-5 text-purple-600 mr-2" />
                    ) : (
                      <User className="h-5 w-5 text-blue-600 mr-2" />
                    )}
                    <div>
                      <p className="font-medium text-gray-900">
                        {selectedCustomer.type === 'individual' 
                          ? `${selectedCustomer.firstName} ${selectedCustomer.lastName}`
                          : selectedCustomer.companyName
                        }
                      </p>
                      <p className="text-sm text-gray-600">{selectedCustomer.phone}</p>
                      {selectedCustomer.isVip && (
                        <span className="inline-flex px-2 py-0.5 text-xs bg-yellow-100 text-yellow-800 rounded-full">
                          VIP Müşteri
                        </span>
                      )}
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Misafir Bilgileri */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center justify-between mb-4">
                <h2 className="text-lg font-medium text-gray-900 flex items-center">
                  <Users className="h-5 w-5 mr-2" />
                  Misafir Bilgileri ({formData.guests.length})
                </h2>
                <button
                  type="button"
                  onClick={addGuest}
                  className="flex items-center px-3 py-2 text-sm bg-purple-600 text-white rounded-lg hover:bg-purple-700"
                >
                  <Plus className="h-4 w-4 mr-1" />
                  Misafir Ekle
                </button>
              </div>

              <div className="space-y-4">
                {formData.guests.map((guest, index) => (
                  <div key={guest.id} className="border border-gray-200 rounded-lg p-4">
                    <div className="flex items-center justify-between mb-3">
                      <h3 className="font-medium text-gray-900">Misafir {index + 1}</h3>
                      {formData.guests.length > 1 && (
                        <button
                          type="button"
                          onClick={() => removeGuest(index)}
                          className="text-red-600 hover:text-red-800"
                        >
                          <Trash2 className="h-4 w-4" />
                        </button>
                      )}
                    </div>
                    
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          Ad *
                        </label>
                        <input
                          type="text"
                          value={guest.firstName}
                          onChange={(e) => handleGuestChange(index, 'firstName', e.target.value)}
                          className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                            errors[`guest_${index}_firstName`] ? 'border-red-300' : 'border-gray-300'
                          }`}
                          placeholder="Misafir adı"
                        />
                        {errors[`guest_${index}_firstName`] && (
                          <p className="mt-1 text-sm text-red-600">{errors[`guest_${index}_firstName`]}</p>
                        )}
                      </div>
                      
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          Soyad *
                        </label>
                        <input
                          type="text"
                          value={guest.lastName}
                          onChange={(e) => handleGuestChange(index, 'lastName', e.target.value)}
                          className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                            errors[`guest_${index}_lastName`] ? 'border-red-300' : 'border-gray-300'
                          }`}
                          placeholder="Misafir soyadı"
                        />
                        {errors[`guest_${index}_lastName`] && (
                          <p className="mt-1 text-sm text-red-600">{errors[`guest_${index}_lastName`]}</p>
                        )}
                      </div>
                      
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          Yaş & Durum
                        </label>
                        <div className="flex space-x-2">
                          <input
                            type="number"
                            value={guest.age || ''}
                            onChange={(e) => handleGuestChange(index, 'age', parseInt(e.target.value))}
                            className="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            placeholder="Yaş"
                            min="0"
                            max="120"
                          />
                          <label className="flex items-center">
                            <input
                              type="checkbox"
                              checked={guest.isChild}
                              onChange={(e) => handleGuestChange(index, 'isChild', e.target.checked)}
                              className="h-4 w-4 text-purple-600 focus:ring-purple-500 border-gray-300 rounded"
                            />
                            <span className="ml-1 text-xs text-gray-600">Çocuk</span>
                          </label>
                        </div>
                      </div>
                    </div>

                    {/* Özel İhtiyaçlar */}
                    <div className="mt-3">
                      <label className="block text-sm font-medium text-gray-700 mb-1">
                        Özel İhtiyaçlar
                      </label>
                      <input
                        type="text"
                        value={guest.specialNeeds || ''}
                        onChange={(e) => handleGuestChange(index, 'specialNeeds', e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="Diyet, engellilik durumu vb."
                      />
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>

          {/* Sağ Panel - Konaklama Bilgileri */}
          <div className="space-y-6">
            {/* Konaklama Detayları */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h2 className="text-lg font-medium text-gray-900 mb-4 flex items-center">
                <Hotel className="h-5 w-5 mr-2" />
                Konaklama Bilgileri
              </h2>
              
              <div className="space-y-4">
                {/* Destinasyon */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Destinasyon *
                  </label>
                  <select
                    value={formData.destination}
                    onChange={(e) => handleInputChange('destination', e.target.value)}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.destination ? 'border-red-300' : 'border-gray-300'
                    }`}
                  >
                    <option value="">Destinasyon seçiniz</option>
                    {destinationOptions.map((destination) => (
                      <option key={destination} value={destination}>
                        {destination}
                      </option>
                    ))}
                  </select>
                  {errors.destination && (
                    <p className="mt-1 text-sm text-red-600">{errors.destination}</p>
                  )}
                </div>

                {/* Giriş Tarihi */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Giriş Tarihi *
                  </label>
                  <input
                    type="date"
                    value={formData.checkInDate}
                    onChange={(e) => handleInputChange('checkInDate', e.target.value)}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.checkInDate ? 'border-red-300' : 'border-gray-300'
                    }`}
                  />
                  {errors.checkInDate && (
                    <p className="mt-1 text-sm text-red-600">{errors.checkInDate}</p>
                  )}
                </div>

                {/* Çıkış Tarihi */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Çıkış Tarihi *
                  </label>
                  <input
                    type="date"
                    value={formData.checkOutDate}
                    onChange={(e) => handleInputChange('checkOutDate', e.target.value)}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.checkOutDate ? 'border-red-300' : 'border-gray-300'
                    }`}
                  />
                  {errors.checkOutDate && (
                    <p className="mt-1 text-sm text-red-600">{errors.checkOutDate}</p>
                  )}
                </div>

                {/* Gece Sayısı Gösterimi */}
                {calculateNights() > 0 && (
                  <div className="bg-blue-50 p-3 rounded-lg">
                    <p className="text-sm text-blue-800">
                      <Calendar className="h-4 w-4 inline mr-1" />
                      {calculateNights()} gece konaklama
                    </p>
                  </div>
                )}

                {/* Oda Türü */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Oda Türü
                  </label>
                  <select
                    value={formData.roomType}
                    onChange={(e) => handleInputChange('roomType', e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    {roomTypeOptions.map((option) => (
                      <option key={option.value} value={option.value}>
                        {option.label}
                      </option>
                    ))}
                  </select>
                </div>

                {/* Oda Sayısı */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Oda Sayısı
                  </label>
                  <input
                    type="number"
                    value={formData.roomCount}
                    onChange={(e) => handleInputChange('roomCount', parseInt(e.target.value))}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    min="1"
                    max="10"
                  />
                </div>

                {/* Yemek Planı */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Yemek Planı
                  </label>
                  <select
                    value={formData.mealPlan || ''}
                    onChange={(e) => handleInputChange('mealPlan', e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option value="">Seçiniz</option>
                    {mealPlanOptions.map((option) => (
                      <option key={option.value} value={option.value}>
                        {option.label}
                      </option>
                    ))}
                  </select>
                </div>

                {/* Otel Kategorisi */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Otel Kategorisi
                  </label>
                  <select
                    value={formData.hotelCategory || ''}
                    onChange={(e) => handleInputChange('hotelCategory', e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option value="">Seçiniz</option>
                    {hotelCategoryOptions.map((option) => (
                      <option key={option.value} value={option.value}>
                        {option.label}
                      </option>
                    ))}
                  </select>
                </div>
              </div>
            </div>

            {/* Bütçe ve Ek Bilgiler */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h2 className="text-lg font-medium text-gray-900 mb-4">
                Bütçe ve Ek Bilgiler
              </h2>
              
              <div className="space-y-4">
                {/* Bütçe Aralığı */}
                <div className="grid grid-cols-2 gap-3">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Min. Bütçe
                    </label>
                    <input
                      type="number"
                      value={formData.budgetMin || ''}
                      onChange={(e) => handleInputChange('budgetMin', parseFloat(e.target.value))}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      placeholder="0"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Max. Bütçe
                    </label>
                    <input
                      type="number"
                      value={formData.budgetMax || ''}
                      onChange={(e) => handleInputChange('budgetMax', parseFloat(e.target.value))}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      placeholder="0"
                    />
                  </div>
                </div>

                {/* Para Birimi */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Para Birimi
                  </label>
                  <select
                    value={formData.currency}
                    onChange={(e) => handleInputChange('currency', e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option value="TRY">TRY</option>
                    <option value="USD">USD</option>
                    <option value="EUR">EUR</option>
                  </select>
                </div>

                {/* Öncelik */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Öncelik
                  </label>
                  <select
                    value={formData.priority}
                    onChange={(e) => handleInputChange('priority', e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option value="low">Düşük</option>
                    <option value="medium">Orta</option>
                    <option value="high">Yüksek</option>
                    <option value="urgent">Acil</option>
                  </select>
                </div>

                {/* Özel İstekler */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Özel İstekler
                  </label>
                  <textarea
                    value={formData.specialRequests || ''}
                    onChange={(e) => handleInputChange('specialRequests', e.target.value)}
                    rows={3}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    placeholder="Deniz manzarası, balkonlu oda, havuz yakını vb."
                  />
                </div>

                {/* Acente Notları */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Acente Notları
                  </label>
                  <textarea
                    value={formData.agentNotes || ''}
                    onChange={(e) => handleInputChange('agentNotes', e.target.value)}
                    rows={3}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    placeholder="İç notlar, özel durumlar..."
                  />
                </div>
              </div>

              {/* Submit Button */}
              <div className="mt-6 pt-6 border-t border-gray-200">
                <button
                  type="submit"
                  disabled={loading}
                  className="w-full flex items-center justify-center px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {loading ? (
                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                  ) : (
                    <Save className="h-4 w-4 mr-2" />
                  )}
                  {loading ? 'Kaydediliyor...' : 'Konaklama Talebi Oluştur'}
                </button>
              </div>
            </div>
          </div>
        </div>
      </form>
    </div>
  );
}