'use client';

import React, { useState, useEffect } from 'react';
import {
  ArrowLeft,
  Save,
  Building2,
  User,
  Palette,
  AlertCircle
} from 'lucide-react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';
import type { AgencyUser, AgencyPermission } from '@/types';
import { departmentApi, agencyUserApi } from '@/services/api';
import PermissionSelector from '@/components/permissions/PermissionSelector';

interface FormData {
  name: string;
  description: string;
  color: string;
  managerId: string;
  permissions: AgencyPermission[];
  isActive: boolean;
}

export default function NewDepartmentPage() {
  const router = useRouter();
  const [users, setUsers] = useState<AgencyUser[]>([]);
  const [loading, setLoading] = useState(false);
  const [errors, setErrors] = useState<Partial<FormData>>({});
  
  const [formData, setFormData] = useState<FormData>({
    name: '',
    description: '',
    color: 'bg-blue-500',
    managerId: '',
    permissions: [],
    isActive: true
  });

  useEffect(() => {
    fetchUsers();
  }, []);

  const fetchUsers = async () => {
    try {
      // Fetch users from API
      const data = await agencyUserApi.getAgencyUsers({ is_active: 'true' });
      setUsers(data);
    } catch (error) {
      console.error('Error fetching users:', error);
    }
  };

  const handleInputChange = (field: keyof FormData, value: string | boolean | AgencyPermission[]) => {
    setFormData(prev => ({ ...prev, [field]: value }));

    // Clear error when user starts typing
    if (errors[field]) {
      setErrors(prev => ({ ...prev, [field]: undefined }));
    }
  };

  const validateForm = (): boolean => {
    const newErrors: Partial<FormData> = {};

    if (!formData.name.trim()) {
      newErrors.name = 'Departman adı gereklidir';
    }

    if (!formData.description.trim()) {
      newErrors.description = 'Açıklama gereklidir';
    }

    if (!formData.color) {
      newErrors.color = 'Renk seçimi gereklidir';
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!validateForm()) {
      return;
    }

    try {
      setLoading(true);

      // Create department via API
      await departmentApi.createDepartment({
        name: formData.name,
        description: formData.description,
        color: formData.color,
        manager_id: formData.managerId || null,
        permissions: formData.permissions
      });

      // Redirect to departments list
      router.push('/departments');
    } catch (error) {
      console.error('Error creating department:', error);
      alert('Departman oluşturulurken bir hata oluştu. Lütfen tekrar deneyin.');
    } finally {
      setLoading(false);
    }
  };

  const colorOptions = [
    { value: 'bg-blue-500', label: 'Mavi', preview: 'bg-blue-500' },
    { value: 'bg-purple-500', label: 'Mor', preview: 'bg-purple-500' },
    { value: 'bg-indigo-500', label: 'İndigo', preview: 'bg-indigo-500' },
    { value: 'bg-green-500', label: 'Yeşil', preview: 'bg-green-500' },
    { value: 'bg-orange-500', label: 'Turuncu', preview: 'bg-orange-500' },
    { value: 'bg-red-500', label: 'Kırmızı', preview: 'bg-red-500' },
    { value: 'bg-pink-500', label: 'Pembe', preview: 'bg-pink-500' },
    { value: 'bg-yellow-500', label: 'Sarı', preview: 'bg-yellow-500' },
    { value: 'bg-teal-500', label: 'Teal', preview: 'bg-teal-500' },
    { value: 'bg-cyan-500', label: 'Cyan', preview: 'bg-cyan-500' }
  ];

  // Yönetici olabilecek kullanıcıları filtrele
  const managerCandidates = users.filter(user => 
    ['admin', 'manager', 'senior_agent'].includes(user.role) && user.isActive
  );

  return (
    <div className="p-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center space-x-4">
          <Link
            href="/departments"
            className="flex items-center text-gray-600 hover:text-gray-900"
          >
            <ArrowLeft className="h-5 w-5 mr-2" />
            Departmanlara Dön
          </Link>
          <div className="h-6 border-l border-gray-300"></div>
          <h1 className="text-2xl font-bold text-gray-900">Yeni Departman Ekle</h1>
        </div>
      </div>

      <form onSubmit={handleSubmit} className="max-w-2xl">
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 space-y-6">
          <h2 className="text-lg font-medium text-gray-900 flex items-center">
            <Building2 className="h-5 w-5 mr-2" />
            Departman Bilgileri
          </h2>
          
          {/* Departman Adı */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Departman Adı *
            </label>
            <input
              type="text"
              value={formData.name}
              onChange={(e) => handleInputChange('name', e.target.value)}
              className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                errors.name ? 'border-red-300' : 'border-gray-300'
              }`}
              placeholder="Örn: Biletleme, Konaklama, Vize İşlemleri"
            />
            {errors.name && (
              <p className="mt-1 text-sm text-red-600 flex items-center">
                <AlertCircle className="h-4 w-4 mr-1" />
                {errors.name}
              </p>
            )}
          </div>

          {/* Açıklama */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Açıklama *
            </label>
            <textarea
              value={formData.description}
              onChange={(e) => handleInputChange('description', e.target.value)}
              rows={3}
              className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                errors.description ? 'border-red-300' : 'border-gray-300'
              }`}
              placeholder="Departmanın görev ve sorumluluklarını açıklayın..."
            />
            {errors.description && (
              <p className="mt-1 text-sm text-red-600 flex items-center">
                <AlertCircle className="h-4 w-4 mr-1" />
                {errors.description}
              </p>
            )}
          </div>

          {/* Renk Seçimi */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-3">
              Departman Rengi *
            </label>
            <div className="grid grid-cols-5 gap-3">
              {colorOptions.map((color) => (
                <button
                  key={color.value}
                  type="button"
                  onClick={() => handleInputChange('color', color.value)}
                  className={`relative flex items-center justify-center p-3 rounded-lg border-2 transition-all ${
                    formData.color === color.value
                      ? 'border-gray-900 ring-2 ring-gray-900 ring-offset-2'
                      : 'border-gray-200 hover:border-gray-300'
                  }`}
                >
                  <div className={`w-6 h-6 rounded-full ${color.preview}`}></div>
                  <span className="ml-2 text-xs text-gray-700">{color.label}</span>
                  {formData.color === color.value && (
                    <div className="absolute -top-1 -right-1 w-4 h-4 bg-gray-900 rounded-full flex items-center justify-center">
                      <div className="w-2 h-2 bg-white rounded-full"></div>
                    </div>
                  )}
                </button>
              ))}
            </div>
            {errors.color && (
              <p className="mt-1 text-sm text-red-600 flex items-center">
                <AlertCircle className="h-4 w-4 mr-1" />
                {errors.color}
              </p>
            )}
          </div>

          {/* Departman Yöneticisi */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Departman Yöneticisi
            </label>
            <select
              value={formData.managerId}
              onChange={(e) => handleInputChange('managerId', e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="">Yönetici seçiniz (isteğe bağlı)</option>
              {managerCandidates.map((user) => (
                <option key={user.id} value={user.id}>
                  {user.firstName} {user.lastName} - {user.position}
                </option>
              ))}
            </select>
            <p className="mt-1 text-xs text-gray-500">
              Sadece yönetici, kıdemli acente veya admin rolündeki kullanıcılar seçilebilir.
            </p>
          </div>

          {/* Durum */}
          <div>
            <label className="flex items-center">
              <input
                type="checkbox"
                checked={formData.isActive}
                onChange={(e) => handleInputChange('isActive', e.target.checked)}
                className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
              />
              <span className="ml-2 text-sm text-gray-700">Departman aktif</span>
            </label>
            <p className="mt-1 text-xs text-gray-500">
              Pasif departmanlar kullanıcı atamalarında görünmez.
            </p>
          </div>

          {/* Önizleme */}
          <div className="border-t border-gray-200 pt-6">
            <h3 className="text-sm font-medium text-gray-700 mb-3">Önizleme</h3>
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="flex items-center">
                <div className={`w-4 h-4 rounded-full ${formData.color} mr-3`}></div>
                <div>
                  <h4 className="font-medium text-gray-900">
                    {formData.name || 'Departman Adı'}
                  </h4>
                  <p className="text-sm text-gray-600">
                    {formData.description || 'Departman açıklaması...'}
                  </p>
                  {formData.managerId && (
                    <p className="text-xs text-gray-500 mt-1">
                      Yönetici: {managerCandidates.find(u => u.id === formData.managerId)?.firstName} {managerCandidates.find(u => u.id === formData.managerId)?.lastName}
                    </p>
                  )}
                </div>
              </div>
            </div>
          </div>

          {/* Yetkiler */}
          <div className="border-t border-gray-200 pt-6">
            <h3 className="text-sm font-medium text-gray-700 mb-3">Departman Yetkileri</h3>
            <PermissionSelector
              selectedPermissions={formData.permissions}
              onChange={(perms) => handleInputChange('permissions', perms)}
              departmentName={formData.name}
            />
          </div>

          {/* Submit Button */}
          <div className="flex justify-end space-x-3 pt-6 border-t border-gray-200">
            <Link
              href="/departments"
              className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50"
            >
              İptal
            </Link>
            <button
              type="submit"
              disabled={loading}
              className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {loading ? (
                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
              ) : (
                <Save className="h-4 w-4 mr-2" />
              )}
              {loading ? 'Kaydediliyor...' : 'Departman Oluştur'}
            </button>
          </div>
        </div>
      </form>
    </div>
  );
}