'use client';

import React, { useState, useEffect } from 'react';
import {
  Plane,
  Hotel,
  Globe,
  Car,
  Users,
  TrendingUp,
  Clock,
  CheckCircle,
  AlertTriangle,
  Building2,
  User,
  Crown,
  Calendar,
  ArrowRight,
  Plus
} from 'lucide-react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';
import type { DashboardStats } from '@/types';
import { systemApi } from '@/services/api';

interface RecentActivity {
  id: string;
  type: 'ticket_request' | 'checkin_completed' | 'accommodation_request' | 'visa_request' | 'transfer_request' | 'customer_added';
  title: string;
  description: string;
  customer: string;
  customerType: 'individual' | 'corporate';
  time: string;
  status: 'pending' | 'completed' | 'confirmed' | 'failed';
  icon: any;
  color: string;
}

export default function DashboardPage() {
  const router = useRouter();
  const [stats, setStats] = useState<DashboardStats>({
    todayCheckIns: 0,
    todayScheduled: 0,
    successRate: 0,
    totalTicketRequests: 0,
    pendingRequests: 0,
    ticketedToday: 0,
    upcomingFlights: 0,
    failedCheckIns: 0
  });

  const [recentActivity, setRecentActivity] = useState<RecentActivity[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    // Check authentication
    const token = localStorage.getItem('token');
    if (!token) {
      router.push('/login');
      return;
    }
    fetchDashboardData();
  }, [router]);

  const fetchDashboardData = async () => {
    try {
      setLoading(true);

      // Fetch real data from API
      const dashboardData = await systemApi.getDashboardStats();

      // Map API response to stats
      const apiStats: DashboardStats = {
        todayCheckIns: dashboardData.todayCheckins || 0,
        todayScheduled: dashboardData.todayScheduled || 0,
        successRate: Math.round(dashboardData.successRate || 0),
        totalTicketRequests: dashboardData.totalTicketRequests || 0,
        pendingRequests: dashboardData.pendingRequests || 0,
        ticketedToday: dashboardData.ticketedToday || 0,
        upcomingFlights: dashboardData.upcomingFlights || 0,
        failedCheckIns: dashboardData.failedCheckins || 0
      };

      // Mock recent activity - can be fetched from API later
      const mockActivity: RecentActivity[] = [
        {
          id: '1',
          type: 'ticket_request',
          title: 'Yeni Bilet Talebi',
          description: 'İstanbul → Antalya, 15 Ocak',
          customer: 'Ahmet Yılmaz',
          customerType: 'individual',
          time: '5 dakika önce',
          status: 'pending',
          icon: Plane,
          color: 'text-blue-600'
        },
        {
          id: '2',
          type: 'checkin_completed',
          title: 'Check-in Tamamlandı',
          description: 'TK1234 - Koltuk 12A',
          customer: 'ABC Teknoloji A.Ş.',
          customerType: 'corporate',
          time: '15 dakika önce',
          status: 'completed',
          icon: CheckCircle,
          color: 'text-green-600'
        },
        {
          id: '3',
          type: 'accommodation_request',
          title: 'Konaklama Talebi',
          description: 'Antalya, 3 gece',
          customer: 'Fatma Kaya',
          customerType: 'individual',
          time: '30 dakika önce',
          status: 'pending',
          icon: Hotel,
          color: 'text-purple-600'
        },
        {
          id: '4',
          type: 'visa_request',
          title: 'Vize Başvurusu',
          description: 'ABD Turist Vizesi',
          customer: 'XYZ İnşaat Ltd.',
          customerType: 'corporate',
          time: '1 saat önce',
          status: 'confirmed',
          icon: Globe,
          color: 'text-indigo-600'
        },
        {
          id: '5',
          type: 'customer_added',
          title: 'Yeni Müşteri',
          description: 'Kurumsal müşteri kaydı',
          customer: 'DEF Yazılım A.Ş.',
          customerType: 'corporate',
          time: '2 saat önce',
          status: 'completed',
          icon: Building2,
          color: 'text-gray-600'
        }
      ];

      setStats(apiStats);
      setRecentActivity(mockActivity);

    } catch (error) {
      console.error('Error fetching dashboard data:', error);
      // Set empty stats on error
      setStats({
        todayCheckIns: 0,
        todayScheduled: 0,
        successRate: 0,
        totalTicketRequests: 0,
        pendingRequests: 0,
        ticketedToday: 0,
        upcomingFlights: 0,
        failedCheckIns: 0
      });
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed': return 'bg-green-100 text-green-800';
      case 'confirmed': return 'bg-blue-100 text-blue-800';
      case 'pending': return 'bg-yellow-100 text-yellow-800';
      case 'failed': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusText = (status: string) => {
    switch (status) {
      case 'completed': return 'Tamamlandı';
      case 'confirmed': return 'Onaylandı';
      case 'pending': return 'Bekliyor';
      case 'failed': return 'Başarısız';
      default: return status;
    }
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse">
          <div className="h-8 bg-gray-200 rounded w-1/3 mb-8"></div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            {[1, 2, 3, 4].map(i => (
              <div key={i} className="h-32 bg-gray-200 rounded"></div>
            ))}
          </div>
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <div className="h-64 bg-gray-200 rounded"></div>
            <div className="h-64 bg-gray-200 rounded"></div>
            <div className="h-64 bg-gray-200 rounded"></div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      {/* Welcome Header */}
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900">
          vTravel Yönetim Paneli 🌍
        </h1>
        <p className="text-gray-600 mt-2">
          Seyahat hizmetlerinizi tek yerden yönetin
        </p>
      </div>

      {/* Main Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-500">Bekleyen Talepler</p>
              <p className="text-3xl font-bold text-gray-900">{stats.pendingRequests}</p>
              <p className="text-sm text-gray-600 mt-1">Toplam {stats.totalTicketRequests} talep</p>
            </div>
            <div className="p-3 bg-yellow-100 rounded-full">
              <Clock className="h-8 w-8 text-yellow-600" />
            </div>
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-500">Bugün Biletlenen</p>
              <p className="text-3xl font-bold text-green-600">{stats.ticketedToday}</p>
              <p className="text-sm text-gray-600 mt-1">{stats.todayScheduled} check-in planlandı</p>
            </div>
            <div className="p-3 bg-green-100 rounded-full">
              <CheckCircle className="h-8 w-8 text-green-600" />
            </div>
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-500">Check-in Başarı</p>
              <p className="text-3xl font-bold text-blue-600">{stats.successRate}%</p>
              <p className="text-sm text-gray-600 mt-1">{stats.todayCheckIns} bugün tamamlandı</p>
            </div>
            <div className="p-3 bg-blue-100 rounded-full">
              <TrendingUp className="h-8 w-8 text-blue-600" />
            </div>
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-500">Yaklaşan Uçuşlar</p>
              <p className="text-3xl font-bold text-purple-600">{stats.upcomingFlights}</p>
              <p className="text-sm text-gray-600 mt-1">{stats.failedCheckIns} başarısız</p>
            </div>
            <div className="p-3 bg-purple-100 rounded-full">
              <Plane className="h-8 w-8 text-purple-600" />
            </div>
          </div>
        </div>
      </div>

      {/* Service Overview */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <Link href="/tickets" className="group">
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-blue-100 rounded-full">
                <Plane className="h-6 w-6 text-blue-600" />
              </div>
              <ArrowRight className="h-5 w-5 text-gray-400 group-hover:text-blue-600 transition-colors" />
            </div>
            <h3 className="text-lg font-semibold text-gray-900 mb-2">Biletleme</h3>
            <p className="text-sm text-gray-600">Uçak bileti talepleri ve check-in işlemleri</p>
            <div className="mt-4 flex items-center space-x-4 text-sm">
              <span className="text-blue-600 font-medium">{stats.pendingRequests} bekliyor</span>
              <span className="text-green-600">{stats.ticketedToday} biletlendi</span>
            </div>
          </div>
        </Link>

        <Link href="/accommodation/requests" className="group">
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-purple-100 rounded-full">
                <Hotel className="h-6 w-6 text-purple-600" />
              </div>
              <ArrowRight className="h-5 w-5 text-gray-400 group-hover:text-purple-600 transition-colors" />
            </div>
            <h3 className="text-lg font-semibold text-gray-900 mb-2">Konaklama</h3>
            <p className="text-sm text-gray-600">Otel rezervasyonları ve konaklama hizmetleri</p>
            <div className="mt-4 flex items-center space-x-4 text-sm">
              <span className="text-yellow-600 font-medium">8 bekliyor</span>
              <span className="text-green-600">12 onaylandı</span>
            </div>
          </div>
        </Link>

        <Link href="/visa/requests" className="group">
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-indigo-100 rounded-full">
                <Globe className="h-6 w-6 text-indigo-600" />
              </div>
              <ArrowRight className="h-5 w-5 text-gray-400 group-hover:text-indigo-600 transition-colors" />
            </div>
            <h3 className="text-lg font-semibold text-gray-900 mb-2">Vize İşlemleri</h3>
            <p className="text-sm text-gray-600">Vize başvuruları ve takip işlemleri</p>
            <div className="mt-4 flex items-center space-x-4 text-sm">
              <span className="text-blue-600 font-medium">5 işlemde</span>
              <span className="text-green-600">18 onaylandı</span>
            </div>
          </div>
        </Link>

        <Link href="/transfer/requests" className="group">
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-green-100 rounded-full">
                <Car className="h-6 w-6 text-green-600" />
              </div>
              <ArrowRight className="h-5 w-5 text-gray-400 group-hover:text-green-600 transition-colors" />
            </div>
            <h3 className="text-lg font-semibold text-gray-900 mb-2">Transfer</h3>
            <p className="text-sm text-gray-600">Havalimanı ve şehir içi transfer hizmetleri</p>
            <div className="mt-4 flex items-center space-x-4 text-sm">
              <span className="text-purple-600 font-medium">3 atandı</span>
              <span className="text-green-600">7 tamamlandı</span>
            </div>
          </div>
        </Link>
      </div>

      {/* Main Content Grid */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Quick Actions */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Hızlı İşlemler</h3>
          <div className="space-y-3">
            <Link href="/tickets/new" className="flex items-center p-3 rounded-lg border border-gray-200 hover:bg-gray-50 transition-colors">
              <div className="p-2 bg-blue-100 rounded-lg mr-3">
                <Plus className="h-4 w-4 text-blue-600" />
              </div>
              <div>
                <p className="font-medium text-gray-900">Yeni Bilet Talebi</p>
                <p className="text-sm text-gray-500">Müşteri için bilet talebi oluştur</p>
              </div>
            </Link>
            
            <Link href="/customers/new" className="flex items-center p-3 rounded-lg border border-gray-200 hover:bg-gray-50 transition-colors">
              <div className="p-2 bg-green-100 rounded-lg mr-3">
                <Users className="h-4 w-4 text-green-600" />
              </div>
              <div>
                <p className="font-medium text-gray-900">Yeni Müşteri</p>
                <p className="text-sm text-gray-500">Bireysel veya kurumsal müşteri ekle</p>
              </div>
            </Link>
            
            <Link href="/checkin" className="flex items-center p-3 rounded-lg border border-gray-200 hover:bg-gray-50 transition-colors">
              <div className="p-2 bg-purple-100 rounded-lg mr-3">
                <CheckCircle className="h-4 w-4 text-purple-600" />
              </div>
              <div>
                <p className="font-medium text-gray-900">Check-in Takibi</p>
                <p className="text-sm text-gray-500">Otomatik check-in durumunu kontrol et</p>
              </div>
            </Link>
          </div>
        </div>

        {/* Recent Activity */}
        <div className="lg:col-span-2 bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <h3 className="text-lg font-semibold text-gray-900">Son Aktiviteler</h3>
            <Link href="/tickets" className="text-sm text-blue-600 hover:text-blue-800">
              Tümünü Gör
            </Link>
          </div>
          
          <div className="space-y-4">
            {recentActivity.map((activity) => (
              <div key={activity.id} className="flex items-start space-x-4 p-4 rounded-lg border border-gray-100 hover:bg-gray-50 transition-colors">
                <div className={`p-2 rounded-full bg-gray-100`}>
                  <activity.icon className={`h-4 w-4 ${activity.color}`} />
                </div>
                <div className="flex-1 min-w-0">
                  <div className="flex items-center justify-between">
                    <p className="font-medium text-gray-900">{activity.title}</p>
                    <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(activity.status)}`}>
                      {getStatusText(activity.status)}
                    </span>
                  </div>
                  <p className="text-sm text-gray-600 mt-1">{activity.description}</p>
                  <div className="flex items-center space-x-2 mt-2">
                    <div className="flex items-center space-x-1">
                      {activity.customerType === 'corporate' ? (
                        <Building2 className="h-3 w-3 text-purple-500" />
                      ) : (
                        <User className="h-3 w-3 text-blue-500" />
                      )}
                      <span className="text-sm text-gray-500">{activity.customer}</span>
                    </div>
                    <span className="text-gray-300">•</span>
                    <span className="text-sm text-gray-500">{activity.time}</span>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* System Info */}
      <div className="mt-8 grid grid-cols-1 md:grid-cols-2 gap-6">
        <div className="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-lg p-6">
          <h3 className="text-lg font-semibold text-blue-900 mb-3 flex items-center">
            <CheckCircle className="mr-2 h-5 w-5" />
            Sistem Özellikleri
          </h3>
          <div className="space-y-2 text-sm text-blue-800">
            <p>• Bireysel ve kurumsal müşteri yönetimi</p>
            <p>• Otomatik check-in ve koltuk seçimi</p>
            <p>• Çoklu hizmet yönetimi (bilet, otel, vize, transfer)</p>
            <p>• Real-time durum takibi ve bildirimler</p>
          </div>
        </div>

        <div className="bg-gradient-to-r from-green-50 to-emerald-50 border border-green-200 rounded-lg p-6">
          <h3 className="text-lg font-semibold text-green-900 mb-3 flex items-center">
            <TrendingUp className="mr-2 h-5 w-5" />
            İş Avantajları
          </h3>
          <div className="space-y-2 text-sm text-green-800">
            <p>• Müşteri memnuniyeti artışı</p>
            <p>• Operasyonel verimlilik</p>
            <p>• 7/24 otomatik hizmet</p>
            <p>• Merkezi yönetim ve raporlama</p>
          </div>
        </div>
      </div>
    </div>
  );
}
