'use client';

import React, { useState, useEffect, useCallback } from 'react';
import {
  ArrowLeft,
  Save,
  Plane,
  User,
  Calendar,
  MapPin,
  Plus,
  Trash2,
  AlertCircle,
  Building2,
  Users,
  Search,
  Loader2,
  X
} from 'lucide-react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';
import type { Customer, PassengerInfo } from '@/types';
import { ticketApi } from '@/services/api';
import PermissionGuard from '@/components/guards/PermissionGuard';

interface FormData {
  customerId: string;
  requestedBy?: string;
  passengers: PassengerInfo[];
  departureAirport: string;
  arrivalAirport: string;
  departureDate: string;
  departureTime?: string;
  arrivalTime?: string;
  returnDate?: string;
  returnTime?: string;
  tripType: 'one-way' | 'round-trip';
  transferPreference: 'any' | 'direct' | 'one-stop' | 'multi-stop';
  ticketPrice?: number;
  currency: 'TRY' | 'USD' | 'EUR';
  agentNotes?: string;
  priority: 'low' | 'medium' | 'high' | 'urgent';
  airline?: string;
  flightNumber?: string;
}

interface Airport {
  iataCode: string;
  name: string;
  city: string;
  country: string;
}

export default function NewTicketPage() {
  return (
    <PermissionGuard requiredPermissions={['tickets_view', 'tickets_view_own', 'tickets_view_all']}>
      <NewTicketPageContent />
    </PermissionGuard>
  );
}

function NewTicketPageContent() {
  const router = useRouter();
  const [customers, setCustomers] = useState<Customer[]>([]);
  const [corporateUsers, setCorporateUsers] = useState<any[]>([]);
  const [loading, setLoading] = useState(false);
  const [errors, setErrors] = useState<any>({});

  // Havalimanı arama state'leri
  const [airportInputs, setAirportInputs] = useState({
    departure: '',
    arrival: ''
  });
  const [airportSuggestions, setAirportSuggestions] = useState<{[key: string]: Airport[]}>({
    departure: [],
    arrival: []
  });
  const [isSearching, setIsSearching] = useState({
    departure: false,
    arrival: false
  });
  const [showDropdown, setShowDropdown] = useState({
    departure: false,
    arrival: false
  });

  // Uçuş arama state'leri
  const [showFlightResults, setShowFlightResults] = useState(false);
  const [flightSearchLoading, setFlightSearchLoading] = useState(false);
  const [availableFlights, setAvailableFlights] = useState<any[]>([]);
  const [selectedFlight, setSelectedFlight] = useState<any>(null);
  const [selectedTravelClass, setSelectedTravelClass] = useState<string>('ECONOMY');

  // Modal state'leri
  const [showUserSelectionModal, setShowUserSelectionModal] = useState(false);
  const [selectedPassengerIndex, setSelectedPassengerIndex] = useState<number>(0);
  const [userSearchQuery, setUserSearchQuery] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const usersPerPage = 5;

  // Özet modalı state'leri
  const [showSummaryModal, setShowSummaryModal] = useState(false);

  const [formData, setFormData] = useState<FormData>({
    customerId: '',
    passengers: [
      {
        id: '1',
        firstName: '',
        lastName: '',
        phone: '',
        email: ''
      }
    ],
    departureAirport: '',
    arrivalAirport: '',
    departureDate: '',
    tripType: 'one-way',
    transferPreference: 'any',
    currency: 'TRY',
    priority: 'medium'
  });

  useEffect(() => {
    fetchCustomers();
  }, []);

  const fetchCustomers = async () => {
    try {
      // Mock customers data
      const mockCustomers: Customer[] = [
        {
          id: 'c1',
          type: 'individual',
          firstName: 'Ahmet',
          lastName: 'Yılmaz',
          phone: '+905551234567',
          email: 'ahmet@email.com',
          isVip: false,
          isActive: true,
          createdAt: new Date(),
          updatedAt: new Date()
        },
        {
          id: 'c2',
          type: 'corporate',
          companyName: 'ABC Teknoloji A.Ş.',
          phone: '+905559876543',
          email: 'info@abcteknoloji.com',
          isVip: true,
          isActive: true,
          createdAt: new Date(),
          updatedAt: new Date()
        }
      ];

      setCustomers(mockCustomers);
    } catch (error) {
      console.error('Error fetching customers:', error);
    }
  };

  const fetchCorporateUsers = async (customerId: string) => {
    try {
      // Mock corporate users - büyük liste için test
      const mockUsers = [
        { id: 'u1', firstName: 'Mehmet', lastName: 'Özkan', email: 'mehmet@abcteknoloji.com', phone: '+905551234567', department: 'IT' },
        { id: 'u2', firstName: 'Ayşe', lastName: 'Demir', email: 'ayse@abcteknoloji.com', phone: '+905559876543', department: 'İnsan Kaynakları' },
        { id: 'u3', firstName: 'Ali', lastName: 'Kaya', email: 'ali@abcteknoloji.com', phone: '+905557654321', department: 'Pazarlama' },
        { id: 'u4', firstName: 'Fatma', lastName: 'Yıldız', email: 'fatma@abcteknoloji.com', phone: '+905558765432', department: 'Finans' },
        { id: 'u5', firstName: 'Ahmet', lastName: 'Çelik', email: 'ahmet@abcteknoloji.com', phone: '+905556543210', department: 'Operasyon' },
        { id: 'u6', firstName: 'Zeynep', lastName: 'Aksoy', email: 'zeynep@abcteknoloji.com', phone: '+905555432109', department: 'IT' },
        { id: 'u7', firstName: 'Mustafa', lastName: 'Bozkurt', email: 'mustafa@abcteknoloji.com', phone: '+905554321098', department: 'Satış' },
        { id: 'u8', firstName: 'Elif', lastName: 'Karaca', email: 'elif@abcteknoloji.com', phone: '+905553210987', department: 'Pazarlama' },
        { id: 'u9', firstName: 'Okan', lastName: 'Şahin', email: 'okan@abcteknoloji.com', phone: '+905552109876', department: 'Operasyon' },
        { id: 'u10', firstName: 'Selin', lastName: 'Arslan', email: 'selin@abcteknoloji.com', phone: '+905551098765', department: 'İnsan Kaynakları' },
        { id: 'u11', firstName: 'Burak', lastName: 'Koç', email: 'burak@abcteknoloji.com', phone: '+905550987654', department: 'IT' },
        { id: 'u12', firstName: 'Deniz', lastName: 'Avcı', email: 'deniz@abcteknoloji.com', phone: '+905559876543', department: 'Finans' }
      ];
      setCorporateUsers(mockUsers);
    } catch (error) {
      console.error('Error fetching corporate users:', error);
    }
  };


  // Havalimanı arama fonksiyonu
  const searchAirports = async (query: string, type: 'departure' | 'arrival') => {
    if (query.length < 2) {
      setAirportSuggestions(prev => ({ ...prev, [type]: [] }));
      setShowDropdown(prev => ({ ...prev, [type]: false }));
      return;
    }

    setIsSearching(prev => ({ ...prev, [type]: true }));

    try {
      const response = await ticketApi.searchAirports(query);

      if (response.success && response.airports) {
        setAirportSuggestions(prev => ({ ...prev, [type]: response.airports }));
        setShowDropdown(prev => ({ ...prev, [type]: true }));
      } else {
        setAirportSuggestions(prev => ({ ...prev, [type]: [] }));
        setShowDropdown(prev => ({ ...prev, [type]: false }));
      }
    } catch (error) {
      console.error('Airport search error:', error);
      setAirportSuggestions(prev => ({ ...prev, [type]: [] }));
      setShowDropdown(prev => ({ ...prev, [type]: false }));
    } finally {
      setIsSearching(prev => ({ ...prev, [type]: false }));
    }
  };

  // Basit input handler - sadece yazma
  const handleAirportInputChange = (value: string, type: 'departure' | 'arrival') => {
    // Input'u anında güncelle - başka hiçbir şey yapma
    setAirportInputs(prev => ({ ...prev, [type]: value }));
  };

  // Manuel arama fonksiyonu - ref kullanarak değeri al
  const handleSearchClick = (type: 'departure' | 'arrival', inputRef: React.RefObject<HTMLInputElement>) => {
    const query = inputRef.current?.value || '';

    if (query.length >= 2) {
      searchAirports(query, type);
    }
  };

  // Havalimanı seçimi
  const handleAirportSelect = (airport: Airport, type: 'departure' | 'arrival') => {
    const field = type === 'departure' ? 'departureAirport' : 'arrivalAirport';

    // Form data'yı güncelle
    setFormData(prev => ({ ...prev, [field]: airport.iataCode }));

    // Input'u güncelle
    setAirportInputs(prev => ({ ...prev, [type]: `${airport.iataCode} - ${airport.name}` }));

    // Dropdown'u kapat
    setShowDropdown(prev => ({ ...prev, [type]: false }));

    // Error'ı temizle
    if (errors[field]) {
      setErrors((prev: any) => ({ ...prev, [field]: undefined }));
    }
  };

  // Uçuş arama fonksiyonu
  const searchFlights = async () => {
    if (!formData.departureAirport || !formData.arrivalAirport || !formData.departureDate) {
      alert('Lütfen önce havalimanları ve kalkış tarihini seçin');
      return;
    }

    setFlightSearchLoading(true);
    setShowFlightResults(true);

    try {
      const searchParams = {
        origin: formData.departureAirport,
        destination: formData.arrivalAirport,
        departureDate: formData.departureDate,
        returnDate: formData.tripType === 'round-trip' ? formData.returnDate : undefined,
        passengers: {
          adults: formData.passengers.length,
          children: 0,
          infants: 0
        },
        travelClass: selectedTravelClass,
        nonStop: formData.transferPreference === 'direct',
        currency: formData.currency
      };

      console.log('Uçuş aranıyor:', searchParams);

      const response = await ticketApi.searchFlights(searchParams);

      if (response.success && response.offers) {
        // Aktarma tercihine göre filtrele
        let filteredFlights = response.offers;

        if (formData.transferPreference === 'direct') {
          filteredFlights = response.offers.filter(flight =>
            flight.outbound?.segments?.length === 1
          );
        } else if (formData.transferPreference === 'one-stop') {
          filteredFlights = response.offers.filter(flight =>
            flight.outbound?.segments?.length === 2
          );
        } else if (formData.transferPreference === 'multi-stop') {
          filteredFlights = response.offers.filter(flight =>
            flight.outbound?.segments?.length >= 3
          );
        }
        // 'any' için filtreleme yok, tüm sonuçlar gösterilir

        setAvailableFlights(filteredFlights);
        console.log('Bulunan uçuşlar (filtrelenmiş):', filteredFlights);
      } else {
        console.error('Uçuş bulunamadı:', response.message);
        setAvailableFlights([]);
        alert('Uçuş bulunamadı: ' + (response.message || 'Bilinmeyen hata'));
      }
    } catch (error: any) {
      console.error('Uçuş arama hatası:', error);
      setAvailableFlights([]);
      alert('Uçuş arama hatası: ' + error.message);
    } finally {
      setFlightSearchLoading(false);
    }
  };

  // Uçuş seçimi
  const selectFlight = (flight: any) => {
    setSelectedFlight(flight);

    // Form verilerini güncelle
    if (flight.outbound?.segments?.[0]) {
      const firstSegment = flight.outbound.segments[0];
      const lastSegment = flight.outbound.segments[flight.outbound.segments.length - 1];

      setFormData(prev => ({
        ...prev,
        ticketPrice: parseFloat(flight.price?.total || '0'),
        currency: flight.price?.currency || 'TRY',
        airline: firstSegment.carrierCode,
        flightNumber: `${firstSegment.carrierCode}${firstSegment.number}`,
        departureTime: new Date(firstSegment.departure.at).toLocaleTimeString('tr-TR', {
          hour: '2-digit',
          minute: '2-digit'
        }),
        arrivalTime: new Date(lastSegment.arrival.at).toLocaleTimeString('tr-TR', {
          hour: '2-digit',
          minute: '2-digit'
        })
      }));
    }

    // Uçuş sonuçları panelini kapat
    setShowFlightResults(false);
  };


  // Uçuş listesi (filtreleme kaldırıldı)
  const filteredFlights = availableFlights;

  const handleInputChange = (field: keyof FormData, value: any) => {
    setFormData(prev => ({ ...prev, [field]: value }));

    // Clear error when user starts typing
    if (errors[field]) {
      setErrors((prev: any) => ({ ...prev, [field]: undefined }));
    }

    // Fetch corporate users when corporate customer is selected
    if (field === 'customerId') {
      const customer = customers.find(c => c.id === value);
      if (customer?.type === 'corporate') {
        fetchCorporateUsers(value);
      } else {
        setCorporateUsers([]);
        setFormData(prev => ({ ...prev, requestedBy: undefined }));
      }
    }
  };

  const handlePassengerChange = (index: number, field: keyof PassengerInfo, value: string) => {
    const newPassengers = [...formData.passengers];
    newPassengers[index] = { ...newPassengers[index], [field]: value };
    setFormData(prev => ({ ...prev, passengers: newPassengers }));
  };

  const addPassenger = () => {
    const newPassenger: PassengerInfo = {
      id: Date.now().toString(),
      firstName: '',
      lastName: '',
      phone: '',
      email: ''
    };
    setFormData(prev => ({
      ...prev,
      passengers: [...prev.passengers, newPassenger]
    }));
  };

  const removePassenger = (index: number) => {
    if (formData.passengers.length > 1) {
      const newPassengers = formData.passengers.filter((_, i) => i !== index);
      setFormData(prev => ({ ...prev, passengers: newPassengers }));
    }
  };

  const addPassengerFromCurrentAccount = (passengerIndex: number) => {
    if (!selectedCustomer) return;

    if (selectedCustomer.type === 'individual') {
      // Bireysel müşteri - doğrudan bilgilerini aktar
      const passengerFromAccount: PassengerInfo = {
        id: Date.now().toString(),
        firstName: selectedCustomer.firstName,
        lastName: selectedCustomer.lastName,
        phone: selectedCustomer.phone,
        email: selectedCustomer.email
      };

      // Belirtilen yolcunun bilgilerini güncelle
      setFormData(prev => ({
        ...prev,
        passengers: prev.passengers.map((passenger, index) =>
          index === passengerIndex ? passengerFromAccount : passenger
        )
      }));
    } else if (selectedCustomer.type === 'corporate') {
      // Kurumsal müşteri - kullanıcı listesinden seçim yapılacak
      if (corporateUsers.length === 0) {
        alert('Bu kurumsal carinin kullanıcıları bulunamadı');
        return;
      }

      // Modal açma ve hangi yolcu için seçim yapıldığını kaydetme
      setSelectedPassengerIndex(passengerIndex);
      setUserSearchQuery(''); // Arama kutusunu temizle
      setCurrentPage(1); // İlk sayfaya dön
      setShowUserSelectionModal(true);
    }
  };

  const selectCorporateUser = (selectedUser: any) => {
    const passengerFromUser: PassengerInfo = {
      id: Date.now().toString(),
      firstName: selectedUser.firstName,
      lastName: selectedUser.lastName,
      phone: selectedUser.phone || selectedCustomer?.phone || '',
      email: selectedUser.email
    };

    // Seçilen yolcunun bilgilerini güncelle
    setFormData(prev => ({
      ...prev,
      passengers: prev.passengers.map((passenger, index) =>
        index === selectedPassengerIndex ? passengerFromUser : passenger
      )
    }));

    // Modal'ı kapat
    setShowUserSelectionModal(false);
  };

  const validateForm = (): boolean => {
    const newErrors: any = {};

    if (!formData.customerId) {
      newErrors.customerId = 'Müşteri seçimi gereklidir';
    }

    if (!formData.departureAirport) {
      newErrors.departureAirport = 'Kalkış havalimanı gereklidir';
    }

    if (!formData.arrivalAirport) {
      newErrors.arrivalAirport = 'Varış havalimanı gereklidir';
    }

    if (!formData.departureDate) {
      newErrors.departureDate = 'Kalkış tarihi gereklidir';
    }

    // Yolcu validasyonu
    formData.passengers.forEach((passenger, index) => {
      if (!passenger.firstName) {
        newErrors[`passenger_${index}_firstName`] = 'Yolcu adı gereklidir';
      }
      if (!passenger.lastName) {
        newErrors[`passenger_${index}_lastName`] = 'Yolcu soyadı gereklidir';
      }
    });

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!validateForm()) {
      return;
    }

    // Özet modalını aç - gerçek submit işlemi modal'da yapılacak
    setShowSummaryModal(true);
  };

  const confirmSubmit = async () => {
    try {
      setLoading(true);
      setShowSummaryModal(false);

      const selectedCustomer = customers.find(c => c.id === formData.customerId);

      // API formatına uygun data hazırla
      const ticketData = {
        pnr: `AUTO${Date.now()}`, // Otomatik PNR oluştur
        customerId: formData.customerId,
        passengers: formData.passengers,
        customerInfo: {
          firstName: formData.passengers[0].firstName,
          lastName: formData.passengers[0].lastName,
          phone: formData.passengers[0].phone || selectedCustomer?.phone || '',
          email: formData.passengers[0].email || selectedCustomer?.email || ''
        },
        // Seçilen uçuş bilgileri
        airline: formData.airline || 'N/A',
        flightNumber: formData.flightNumber || 'N/A',
        departureDate: formData.departureDate,
        departureTime: formData.departureTime || '00:00',
        arrivalTime: formData.arrivalTime || '00:00',
        departureAirport: formData.departureAirport,
        arrivalAirport: formData.arrivalAirport,
        tripType: formData.tripType,
        returnDate: formData.returnDate,
        ticketPrice: formData.ticketPrice,
        currency: formData.currency,
        priority: formData.priority,
        selectedFlight: selectedFlight,
        flightType: 'domestic', // Türkiye içi uçuşlar için
        notes: formData.agentNotes || ''
      };

      console.log('Bilet oluşturuluyor:', ticketData);

      // Şimdilik mock success response - gerçek API çağrısı eklenecek
      // const response = await ticketApi.createTicket(ticketData);

      // Mock başarılı cevap
      const mockResponse = {
        success: true,
        ticket: {
          id: Date.now().toString(),
          pnr: ticketData.pnr,
          ...ticketData
        }
      };

      // 1 saniye bekle (API simülasyonu)
      await new Promise(resolve => setTimeout(resolve, 1000));

      if (mockResponse.success) {
        console.log('✅ Bilet başarıyla oluşturuldu:', mockResponse.ticket);

        // LocalStorage'a kaydet (geçici çözüm)
        try {
          const existingTickets = JSON.parse(localStorage.getItem('vtravel-tickets') || '[]');
          const newTicket = {
            ...mockResponse.ticket,
            id: Date.now().toString(),
            createdAt: new Date().toISOString(),
            status: 'pending', // pending, confirmed, cancelled
            updatedAt: new Date().toISOString(),
            customer: selectedCustomer // Customer bilgisini ekle
          };
          existingTickets.unshift(newTicket); // En başa ekle
          localStorage.setItem('vtravel-tickets', JSON.stringify(existingTickets));
          console.log('💾 Bilet localStorage\'a kaydedildi');
        } catch (error) {
          console.error('LocalStorage kayıt hatası:', error);
        }

        alert('Bilet talebi başarıyla oluşturuldu! PNR: ' + ticketData.pnr);
        router.push('/tickets');
      } else {
        console.error('❌ Bilet oluşturma hatası:', mockResponse.message);
        setErrors({ submit: mockResponse.message || 'Bilet oluşturulurken bir hata oluştu' });
      }
    } catch (error: any) {
      console.error('Error creating ticket request:', error);
      setErrors({ submit: error.message || 'Bilet oluşturulurken bir hata oluştu' });
    } finally {
      setLoading(false);
    }
  };

  // Helper functions for display names
  const getAirlineName = (carrierCode: string) => {
    const airlines: { [key: string]: string } = {
      // Türk Havayolları
      'TK': 'Turkish Airlines',
      'PC': 'Pegasus',
      'VF': 'SunExpress',
      'AJT': 'AJet',
      'XQ': 'SunExpress Deutschland',

      // Avrupa Havayolları
      'LH': 'Lufthansa',
      'AF': 'Air France',
      'KL': 'KLM',
      'BA': 'British Airways',
      'IB': 'Iberia',
      'AZ': 'Alitalia',
      'LX': 'Swiss International',
      'OS': 'Austrian Airlines',
      'SN': 'Brussels Airlines',
      'TP': 'TAP Portugal',
      'SK': 'SAS',
      'AY': 'Finnair',
      'DL': 'Delta Air Lines',
      'UA': 'United Airlines',
      'AA': 'American Airlines',
      'AC': 'Air Canada',
      'VS': 'Virgin Atlantic',
      'EI': 'Aer Lingus',
      'FR': 'Ryanair',
      'U2': 'easyJet',
      'W6': 'Wizz Air',
      'VY': 'Vueling',
      'EW': 'Eurowings',

      // Ortadoğu ve Afrika
      'EK': 'Emirates',
      'QR': 'Qatar Airways',
      'EY': 'Etihad Airways',
      'FZ': 'flydubai',
      'WY': 'Oman Air',
      'MS': 'EgyptAir',
      'RJ': 'Royal Jordanian',
      'ME': 'Middle East Airlines',
      'GF': 'Gulf Air',
      'KU': 'Kuwait Airways',
      'SV': 'Saudi Arabian Airlines',
      'F3': 'Flynas',
      'XY': 'Flynas',

      // Asya-Pasifik
      'SQ': 'Singapore Airlines',
      'CX': 'Cathay Pacific',
      'TG': 'Thai Airways',
      'NH': 'ANA',
      'JL': 'Japan Airlines',
      'KE': 'Korean Air',
      'OZ': 'Asiana Airlines',
      'MH': 'Malaysia Airlines',
      'GA': 'Garuda Indonesia',
      'AI': 'Air India',
      'VN': 'Vietnam Airlines',
      'PR': 'Philippine Airlines',
      'CI': 'China Airlines',
      'BR': 'EVA Air',
      'CZ': 'China Southern',
      'CA': 'Air China',
      'MU': 'China Eastern',
      '3K': 'Jetstar Asia',
      'QF': 'Qantas',
      'JQ': 'Jetstar',
      'VA': 'Virgin Australia',

      // Rusya ve CIS
      'SU': 'Aeroflot',
      'S7': 'S7 Airlines',
      'UT': 'UTair',
      'FV': 'Rossiya Airlines',
      'UN': 'Transaero',
      'KC': 'Air Astana',
      'HY': 'Uzbekistan Airways',
      'B2': 'Belavia',

      // Balkans & Eastern Europe
      'JU': 'Air Serbia',
      'OU': 'Croatia Airlines',
      'JP': 'Adria Airways',
      'YM': 'Montenegro Airlines',
      'FB': 'Bulgaria Air',
      'RO': 'Tarom',
      'LO': 'LOT Polish Airlines',
      'OK': 'Czech Airlines',
      'PS': 'Ukraine International',

      // Latin America
      'LA': 'LATAM',
      'AR': 'Aerolíneas Argentinas',
      'AM': 'Aeroméxico',
      'CM': 'Copa Airlines',
      'AV': 'Avianca',
      'G3': 'Gol',
      'JJ': 'TAM',

      // Low-cost carriers
      'W4': 'Wizz Air Malta',
      'G9': 'Air Arabia',
      '6E': 'IndiGo',
      'SG': 'SpiceJet',
      'I5': 'AirAsia India',
      'FD': 'Thai AirAsia',
      'AK': 'AirAsia',
      'D7': 'AirAsia X',
      'TR': 'Scoot',
      'GK': 'Jetstar Japan',
      'MM': 'Peach',
      'BC': 'Skymark',
      'ZE': 'Eastar Jet',
      'LJ': 'Jin Air',
      'BX': 'Air Busan',
      'TW': 'T\u0027way Air',

      // African airlines
      'SA': 'South African Airways',
      'ET': 'Ethiopian Airlines',
      'KQ': 'Kenya Airways',
      'AT': 'Royal Air Maroc',
      'UU': 'Air Austral',
      'MD': 'Air Madagascar',
      'HC': 'Avianca Honduras'
    };
    return airlines[carrierCode] || carrierCode;
  };

  const getAirportName = (iataCode: string) => {
    const airports: { [key: string]: string } = {
      'IST': 'İstanbul Havalimanı',
      'SAW': 'Sabiha Gökçen Havalimanı',
      'ESB': 'Esenboğa Havalimanı',
      'AYT': 'Antalya Havalimanı',
      'ADB': 'Adnan Menderes Havalimanı',
      'TZX': 'Trabzon Havalimanı',
      'GZT': 'Gaziantep Havalimanı'
    };
    return airports[iataCode] || iataCode;
  };

  const selectedCustomer = customers.find(c => c.id === formData.customerId);

  // Yeni Havalimanı Input Komponenti
  const AirportInput = ({
    type,
    label,
    required = false
  }: {
    type: 'departure' | 'arrival';
    label: string;
    required?: boolean;
  }) => {
    const inputRef = React.useRef<HTMLInputElement>(null);
    const dropdownRef = React.useRef<HTMLDivElement>(null);

    // Uncontrolled input - hiç state güncelleme yapmayız
    const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
      // Hiçbir şey yapma - sadece kullanıcı yazsın
    };

    const handleInputFocus = () => {
      // Sadece önceden arama yapılmışsa dropdown'u aç, otomatik arama yapma
      if (airportSuggestions[type].length > 0) {
        setShowDropdown(prev => ({ ...prev, [type]: true }));
      }
    };

    const selectAirport = (airport: Airport) => {
      const field = type === 'departure' ? 'departureAirport' : 'arrivalAirport';

      // Form data'yı güncelle
      setFormData(prev => ({ ...prev, [field]: airport.iataCode }));

      // Input'u güncelle
      setAirportInputs(prev => ({ ...prev, [type]: `${airport.iataCode} - ${airport.name}` }));

      // Dropdown'u kapat
      setShowDropdown(prev => ({ ...prev, [type]: false }));

      // Error'ı temizle
      if (errors[field]) {
        setErrors((prev: any) => ({ ...prev, [field]: undefined }));
      }

      // Input'a focus'u geri ver
      setTimeout(() => {
        if (inputRef.current) {
          inputRef.current.focus();
        }
      }, 10);
    };

    const clearAirport = () => {
      const field = type === 'departure' ? 'departureAirport' : 'arrivalAirport';

      // Form data'yı temizle
      setFormData(prev => ({ ...prev, [field]: '' }));

      // Input'u temizle
      setAirportInputs(prev => ({ ...prev, [type]: '' }));
      if (inputRef.current) {
        inputRef.current.value = '';
      }

      // Dropdown'u kapat
      setShowDropdown(prev => ({ ...prev, [type]: false }));

      // Suggestions'ı temizle
      setAirportSuggestions(prev => ({ ...prev, [type]: [] }));

      // Error'ı temizle
      if (errors[field]) {
        setErrors((prev: any) => ({ ...prev, [field]: undefined }));
      }

      // Input'a focus ver
      if (inputRef.current) {
        inputRef.current.focus();
      }
    };

    // Document click listener to close dropdown
    React.useEffect(() => {
      const handleClickOutside = (event: MouseEvent) => {
        if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
          setShowDropdown(prev => ({ ...prev, [type]: false }));
        }
      };

      if (showDropdown[type]) {
        document.addEventListener('mousedown', handleClickOutside);
        return () => document.removeEventListener('mousedown', handleClickOutside);
      }
    }, [showDropdown, type]);

    return (
      <div className="relative" ref={dropdownRef}>
        <label className="block text-sm font-medium text-gray-700 mb-1">
          {label} {required && '*'}
        </label>
        <div className="flex gap-2">
          <div className="relative flex-1">
            <input
              ref={inputRef}
              type="text"
              data-type={type}
              defaultValue={airportInputs[type]}
              onChange={handleChange}
              onFocus={handleInputFocus}
              className={`w-full px-3 py-2 pl-10 pr-10 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                errors[type === 'departure' ? 'departureAirport' : 'arrivalAirport']
                  ? 'border-red-300'
                  : 'border-gray-300'
              }`}
              placeholder="Şehir veya havalimanı yaz (örn: İstanbul)"
              autoComplete="off"
            />
            <div className="absolute left-3 top-1/2 transform -translate-y-1/2 pointer-events-none">
              <MapPin className="h-4 w-4 text-gray-400" />
            </div>
            {/* Clear Button - only show when there's a selected airport */}
            {airportInputs[type] && (
              <button
                type="button"
                onClick={clearAirport}
                className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600 transition-colors"
                title="Temizle"
              >
                <X className="h-4 w-4" />
              </button>
            )}
          </div>
          <button
            type="button"
            onClick={() => handleSearchClick(type, inputRef)}
            disabled={isSearching[type]}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed flex items-center gap-2"
          >
            {isSearching[type] ? (
              <Loader2 className="h-4 w-4 animate-spin" />
            ) : (
              <Search className="h-4 w-4" />
            )}
            Ara
          </button>
        </div>

        {/* Dropdown */}
        {showDropdown[type] && airportSuggestions[type].length > 0 && (
          <div className="absolute z-50 w-full mt-1 bg-white border border-gray-300 rounded-lg shadow-lg max-h-60 overflow-y-auto">
            {airportSuggestions[type].map((airport, index) => (
              <div
                key={`${airport.iataCode}-${index}`}
                onClick={() => selectAirport(airport)}
                className="w-full text-left px-4 py-3 hover:bg-blue-50 cursor-pointer border-b border-gray-100 last:border-b-0"
              >
                <div className="font-medium text-gray-900">
                  <span className="bg-blue-100 text-blue-800 px-2 py-0.5 rounded text-xs font-mono mr-2">
                    {airport.iataCode}
                  </span>
                  {airport.name}
                </div>
                <div className="text-sm text-gray-600">
                  {airport.city}, {airport.country}
                </div>
              </div>
            ))}
          </div>
        )}

        {errors[type === 'departure' ? 'departureAirport' : 'arrivalAirport'] && (
          <p className="mt-1 text-sm text-red-600 flex items-center">
            <AlertCircle className="h-4 w-4 mr-1" />
            {errors[type === 'departure' ? 'departureAirport' : 'arrivalAirport']}
          </p>
        )}
      </div>
    );
  };

  return (
    <div className="p-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center space-x-4">
          <Link
            href="/tickets"
            className="flex items-center text-gray-600 hover:text-gray-900"
          >
            <ArrowLeft className="h-5 w-5 mr-2" />
            Bilet Taleplerine Dön
          </Link>
          <div className="h-6 border-l border-gray-300"></div>
          <h1 className="text-2xl font-bold text-gray-900 flex items-center">
            <Plane className="mr-3 h-8 w-8 text-blue-600" />
            Yeni Bilet Talebi
          </h1>
        </div>
      </div>

      <form onSubmit={handleSubmit} className="space-y-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* SOL PANEL - UÇUŞ BİLGİLERİ VE EK BİLGİLER */}
          <div className="lg:col-span-2 space-y-6">
            {/* Uçuş Bilgileri */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h2 className="text-lg font-medium text-gray-900 mb-4 flex items-center">
                <Plane className="h-5 w-5 mr-2 text-blue-600" />
                Uçuş Bilgileri
              </h2>

              <div className="space-y-4">
                {/* Seyahat Türü */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Seyahat Türü
                  </label>
                  <div className="flex space-x-4">
                    <label className="flex items-center">
                      <input
                        type="radio"
                        value="one-way"
                        checked={formData.tripType === 'one-way'}
                        onChange={(e) => handleInputChange('tripType', e.target.value)}
                        className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300"
                      />
                      <span className="ml-2 text-sm text-gray-700">Tek Yön</span>
                    </label>
                    <label className="flex items-center">
                      <input
                        type="radio"
                        value="round-trip"
                        checked={formData.tripType === 'round-trip'}
                        onChange={(e) => handleInputChange('tripType', e.target.value)}
                        className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300"
                      />
                      <span className="ml-2 text-sm text-gray-700">Gidiş-Dönüş</span>
                    </label>
                  </div>
                </div>


                {/* Havalimanları */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <AirportInput
                    type="departure"
                    label="Kalkış Havalimanı"
                    required={true}
                  />

                  <AirportInput
                    type="arrival"
                    label="Varış Havalimanı"
                    required={true}
                  />
                </div>

                {/* Tarih Seçimi */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1 flex items-center">
                      <Calendar className="h-4 w-4 mr-1" />
                      Gidiş Tarihi *
                    </label>
                    <input
                      type="date"
                      value={formData.departureDate}
                      onChange={(e) => handleInputChange('departureDate', e.target.value)}
                      className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                        errors.departureDate ? 'border-red-300' : 'border-gray-300'
                      }`}
                    />
                    {errors.departureDate && (
                      <p className="mt-1 text-sm text-red-600 flex items-center">
                        <AlertCircle className="h-4 w-4 mr-1" />
                        {errors.departureDate}
                      </p>
                    )}
                  </div>

                  {/* Dönüş Tarihi (Gidiş-dönüş ise) */}
                  {formData.tripType === 'round-trip' && (
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1 flex items-center">
                        <Calendar className="h-4 w-4 mr-1" />
                        Dönüş Tarihi *
                      </label>
                      <input
                        type="date"
                        value={formData.returnDate || ''}
                        onChange={(e) => handleInputChange('returnDate', e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                  )}
                </div>

                {/* Yolcu Sayısı, Seyahat Sınıfı, Aktarma Tercihi ve Uçuş Arama */}
                <div className="grid grid-cols-1 md:grid-cols-5 gap-3 items-end">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Yolcu Sayısı *
                    </label>
                    <select
                      value={formData.passengers.length}
                      onChange={(e) => {
                        const newCount = parseInt(e.target.value);
                        const currentCount = formData.passengers.length;

                        if (newCount > currentCount) {
                          // Yolcu ekle
                          const newPassengers = [...formData.passengers];
                          for (let i = currentCount; i < newCount; i++) {
                            newPassengers.push({
                              id: Date.now().toString() + i,
                              firstName: '',
                              lastName: '',
                              phone: '',
                              email: ''
                            });
                          }
                          setFormData(prev => ({ ...prev, passengers: newPassengers }));
                        } else if (newCount < currentCount) {
                          // Yolcu çıkar
                          const newPassengers = formData.passengers.slice(0, newCount);
                          setFormData(prev => ({ ...prev, passengers: newPassengers }));
                        }
                      }}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    >
                      {[1,2,3,4,5,6,7,8,9].map(num => (
                        <option key={num} value={num}>{num} Kişi</option>
                      ))}
                    </select>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Seyahat Sınıfı *
                    </label>
                    <select
                      value={selectedTravelClass}
                      onChange={(e) => setSelectedTravelClass(e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    >
                      <option value="ECONOMY">Ekonomi</option>
                      <option value="BUSINESS">Business</option>
                    </select>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Aktarma Tercihi
                    </label>
                    <select
                      value={formData.transferPreference}
                      onChange={(e) => handleInputChange('transferPreference', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    >
                      <option value="any">Tümü</option>
                      <option value="direct">Sadece Direkt</option>
                      <option value="one-stop">1 Aktarma</option>
                      <option value="multi-stop">Çoklu Aktarma</option>
                    </select>
                  </div>

                  <div>
                    <button
                      type="button"
                      onClick={searchFlights}
                      disabled={!formData.departureAirport || !formData.arrivalAirport || !formData.departureDate}
                      className="w-full px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2"
                    >
                      <Plane className="h-4 w-4" />
                      Uçuş Ara
                    </button>
                  </div>
                </div>

                {/* Inline Flight Results */}
                {showFlightResults && (
                  <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
                    <div className="flex items-center justify-between mb-4">
                      <h3 className="text-lg font-medium text-gray-900 flex items-center">
                        <Plane className="h-5 w-5 mr-2 text-blue-600" />
                        Uygun Uçuşlar
                      </h3>
                      <button
                        onClick={() => setShowFlightResults(false)}
                        className="text-gray-400 hover:text-gray-600"
                      >
                        <X className="h-5 w-5" />
                      </button>
                    </div>


                    {/* Flight Results */}
                    <div className="max-h-96 overflow-y-auto">
                      {flightSearchLoading ? (
                        <div className="flex items-center justify-center py-8">
                          <Loader2 className="h-6 w-6 animate-spin text-blue-600" />
                          <span className="ml-3 text-gray-600">Uçuşlar aranıyor...</span>
                        </div>
                      ) : availableFlights.length === 0 ? (
                        <div className="text-center py-8">
                          <Plane className="h-8 w-8 text-gray-400 mx-auto mb-3" />
                          <p className="text-gray-600">Bu rotada uçuş bulunamadı.</p>
                        </div>
                      ) : (
                        <div className="space-y-3">
                          {availableFlights
                            .sort((a, b) => {
                              const aSegments = a.outbound?.segments?.length || 0;
                              const bSegments = b.outbound?.segments?.length || 0;

                              // Önce direkt uçuşlar (segments = 1), sonra aktarmalılar
                              if (aSegments !== bSegments) {
                                return aSegments - bSegments;
                              }

                              // Aynı aktarma sayısında saate göre sırala
                              const aTime = a.outbound?.segments?.[0]?.departure?.at || '';
                              const bTime = b.outbound?.segments?.[0]?.departure?.at || '';
                              return aTime.localeCompare(bTime);
                            })
                            .map((flight, index) => {
                            const outbound = flight.outbound;
                            const firstSegment = outbound?.segments?.[0];
                            const lastSegment = outbound?.segments?.[outbound.segments.length - 1];

                            if (!firstSegment || !lastSegment) return null;


                            return (
                              <div
                                key={index}
                                className="bg-white border border-gray-200 rounded-lg p-4 hover:border-blue-300 hover:shadow-md transition-all cursor-pointer"
                                onClick={() => selectFlight(flight)}
                              >
                                <div className="flex items-center justify-between">
                                  <div className="flex-1">
                                    {/* Flight Info With Airline and Flight Number */}
                                    <div className="flex items-center space-x-4 mb-3">
                                      <div className="text-sm font-medium text-gray-900">
                                        {getAirlineName(firstSegment.carrierCode)} {firstSegment.carrierCode}{firstSegment.number}
                                      </div>
                                      <div className="text-sm text-gray-500">
                                        {outbound.duration?.replace('PT', '').replace('H', 'sa ').replace('M', 'dk')}
                                      </div>
                                      <div className="text-sm text-gray-500">
                                        {outbound.segments.length === 1 ? 'Direkt' : `${outbound.segments.length - 1} aktarma`}
                                      </div>
                                    </div>

                                    {/* Route Information */}
                                    <div className="flex items-center space-x-6">
                                      <div className="text-center">
                                        <div className="text-sm font-medium text-gray-900">
                                          {firstSegment.departure.iataCode}
                                        </div>
                                        <div className="text-xs text-gray-500 mb-1">
                                          {getAirportName(firstSegment.departure.iataCode)}
                                        </div>
                                        <div className="text-xs text-gray-500">
                                          {new Date(firstSegment.departure.at).toLocaleTimeString('tr-TR', {
                                            hour: '2-digit',
                                            minute: '2-digit'
                                          })}
                                        </div>
                                      </div>

                                      <div className="flex-1 flex items-center">
                                        <div className="w-full border-t border-gray-300 relative">
                                          <Plane className="h-4 w-4 text-blue-600 absolute left-1/2 top-1/2 transform -translate-x-1/2 -translate-y-1/2 bg-white" />
                                        </div>
                                      </div>

                                      <div className="text-center">
                                        <div className="text-sm font-medium text-gray-900">
                                          {lastSegment.arrival.iataCode}
                                        </div>
                                        <div className="text-xs text-gray-500 mb-1">
                                          {getAirportName(lastSegment.arrival.iataCode)}
                                        </div>
                                        <div className="text-xs text-gray-500">
                                          {new Date(lastSegment.arrival.at).toLocaleTimeString('tr-TR', {
                                            hour: '2-digit',
                                            minute: '2-digit'
                                          })}
                                        </div>
                                      </div>
                                    </div>

                                    {/* Transfer Information */}
                                    {outbound.segments.length > 1 && (
                                      <div className="mt-2 space-y-1">
                                        <div className="text-xs text-orange-600 font-medium">
                                          🔄 {outbound.segments.length - 1} Aktarma
                                        </div>
                                        <div className="text-xs text-gray-500 space-y-1">
                                          {outbound.segments.slice(0, -1).map((segment, idx) => {
                                            const nextSegment = outbound.segments[idx + 1];
                                            const layoverTime = nextSegment ?
                                              Math.round((new Date(nextSegment.departure.at).getTime() - new Date(segment.arrival.at).getTime()) / (1000 * 60)) : 0;
                                            return (
                                              <div key={idx} className="flex items-center text-xs">
                                                <span className="text-orange-600">✈</span>
                                                <span className="ml-1">
                                                  {getAirportName(segment.arrival.iataCode)} ({segment.arrival.iataCode})
                                                </span>
                                                {layoverTime > 0 && (
                                                  <span className="ml-2 text-gray-400">
                                                    • {Math.floor(layoverTime / 60)}sa {layoverTime % 60}dk bekleme
                                                  </span>
                                                )}
                                              </div>
                                            );
                                          })}
                                        </div>
                                      </div>
                                    )}
                                  </div>

                                  {/* Price */}
                                  <div className="text-right ml-6">
                                    <div className="text-xl font-bold text-green-600">
                                      {flight.price?.total} {flight.price?.currency}
                                    </div>
                                    <div className="text-sm text-gray-500">
                                      Kişi başı
                                    </div>
                                    <button className="mt-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 text-sm">
                                      Seç
                                    </button>
                                  </div>
                                </div>
                              </div>
                            );
                          })}
                        </div>
                      )}
                    </div>
                  </div>
                )}

                {/* Selected Flight Information */}
                {selectedFlight && (
                  <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                    <div className="flex items-center justify-between">
                      <div className="flex-1">
                        <h4 className="font-medium text-green-900 flex items-center">
                          <Plane className="h-4 w-4 mr-2" />
                          ✅ Uçuş Seçildi
                        </h4>
                        <div className="mt-2 space-y-1">
                          <div className="text-sm text-green-700">
                            <strong>{getAirlineName(selectedFlight.outbound?.segments?.[0]?.carrierCode)} {selectedFlight.outbound?.segments?.[0]?.carrierCode}{selectedFlight.outbound?.segments?.[0]?.number}</strong>
                          </div>
                          <div className="text-sm text-green-700">
                            {selectedFlight.outbound?.segments?.[0]?.departure?.iataCode} → {selectedFlight.outbound?.segments?.[selectedFlight.outbound.segments.length - 1]?.arrival?.iataCode}
                          </div>
                          <div className="text-sm text-green-700">
                            🕐 {formData.departureTime} - {formData.arrivalTime}
                          </div>
                          <div className="text-sm font-medium text-green-800">
                            💰 {selectedFlight.price?.total} {selectedFlight.price?.currency}
                          </div>
                        </div>
                      </div>
                      <button
                        type="button"
                        onClick={() => {
                          setSelectedFlight(null);
                          setFormData(prev => ({
                            ...prev,
                            ticketPrice: undefined,
                            airline: undefined,
                            flightNumber: undefined,
                            departureTime: undefined,
                            arrivalTime: undefined
                          }));
                        }}
                        className="text-green-600 hover:text-green-800 ml-4"
                        title="Seçimi İptal Et"
                      >
                        <X className="h-4 w-4" />
                      </button>
                    </div>
                  </div>
                )}

              </div>
            </div>

            {/* Ek Bilgiler */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h2 className="text-lg font-medium text-gray-900 mb-4 flex items-center">
                <MapPin className="h-5 w-5 mr-2 text-green-600" />
                Ek Bilgiler
              </h2>

              <div className="space-y-4">
                {/* Acente Notları */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Acente Notları
                  </label>
                  <textarea
                    value={formData.agentNotes || ''}
                    onChange={(e) => handleInputChange('agentNotes', e.target.value)}
                    rows={3}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    placeholder="Özel istekler, notlar..."
                  />
                </div>
              </div>
            </div>
          </div>

          {/* SAĞ PANEL - MÜŞTERİ VE YOLCU BİLGİLERİ */}
          <div className="space-y-6">
            {/* Öncelik Seviyesi */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h2 className="text-lg font-medium text-gray-900 mb-4 flex items-center">
                <AlertCircle className="h-5 w-5 mr-2 text-orange-600" />
                Öncelik Seviyesi
              </h2>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Öncelik Seviyesi
                </label>
                <select
                  value={formData.priority}
                  onChange={(e) => handleInputChange('priority', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="low">Düşük</option>
                  <option value="medium">Orta</option>
                  <option value="high">Yüksek</option>
                  <option value="urgent">Acil</option>
                </select>
              </div>
            </div>

            {/* Müşteri Seçimi */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h2 className="text-lg font-medium text-gray-900 mb-4 flex items-center">
                <User className="h-5 w-5 mr-2 text-purple-600" />
                Cari Bilgileri
              </h2>

              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Cari *
                  </label>
                  <select
                    value={formData.customerId}
                    onChange={(e) => handleInputChange('customerId', e.target.value)}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.customerId ? 'border-red-300' : 'border-gray-300'
                    }`}
                  >
                    <option value="">Cari seçiniz</option>
                    {customers.map((customer) => (
                      <option key={customer.id} value={customer.id}>
                        {customer.type === 'individual'
                          ? `${customer.firstName} ${customer.lastName}`
                          : customer.companyName
                        } - {customer.phone}
                      </option>
                    ))}
                  </select>
                  {errors.customerId && (
                    <p className="mt-1 text-sm text-red-600 flex items-center">
                      <AlertCircle className="h-4 w-4 mr-1" />
                      {errors.customerId}
                    </p>
                  )}
                </div>


                {selectedCustomer && (
                  <div className="p-3 bg-gray-50 rounded-lg">
                    <div className="flex items-center">
                      {selectedCustomer.type === 'corporate' ? (
                        <Building2 className="h-5 w-5 text-purple-600 mr-2" />
                      ) : (
                        <User className="h-5 w-5 text-blue-600 mr-2" />
                      )}
                      <div>
                        <p className="font-medium text-gray-900">
                          {selectedCustomer.type === 'individual'
                            ? `${selectedCustomer.firstName} ${selectedCustomer.lastName}`
                            : selectedCustomer.companyName
                          }
                        </p>
                        <p className="text-sm text-gray-600">{selectedCustomer.phone}</p>
                        {selectedCustomer.isVip && (
                          <span className="inline-flex px-2 py-0.5 text-xs bg-yellow-100 text-yellow-800 rounded-full">
                            VIP Müşteri
                          </span>
                        )}
                      </div>
                    </div>
                  </div>
                )}
              </div>
            </div>

            {/* Yolcu Bilgileri */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center justify-between mb-4">
                <h2 className="text-lg font-medium text-gray-900 flex items-center">
                  <Users className="h-5 w-5 mr-2 text-blue-600" />
                  Yolcu Bilgileri ({formData.passengers.length})
                </h2>
                <div className="text-sm text-gray-600">
                  Yolcu sayısını uçuş arama bölümünden değiştirebilirsiniz
                </div>
              </div>

              <div className="space-y-4">
                {formData.passengers.map((passenger, index) => (
                  <div key={passenger.id} className="border border-gray-200 rounded-lg p-4">
                    <div className="flex items-center justify-between mb-3">
                      <h3 className="font-medium text-gray-900">Yolcu {index + 1}</h3>
                      <div className="flex items-center gap-2">
                        {selectedCustomer && (
                          <button
                            type="button"
                            onClick={() => addPassengerFromCurrentAccount(index)}
                            className="text-sm px-3 py-1 bg-purple-100 text-purple-700 rounded-lg hover:bg-purple-200 transition-colors"
                          >
                            Cariden Ekle
                          </button>
                        )}
                        {formData.passengers.length > 1 && (
                          <button
                            type="button"
                            onClick={() => removePassenger(index)}
                            className="text-red-600 hover:text-red-800"
                          >
                            <Trash2 className="h-4 w-4" />
                          </button>
                        )}
                      </div>
                    </div>

                    <div className="space-y-3">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          Ad *
                        </label>
                        <input
                          type="text"
                          value={passenger.firstName}
                          onChange={(e) => handlePassengerChange(index, 'firstName', e.target.value)}
                          className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                            errors[`passenger_${index}_firstName`] ? 'border-red-300' : 'border-gray-300'
                          }`}
                          placeholder="Yolcu adı"
                        />
                        {errors[`passenger_${index}_firstName`] && (
                          <p className="mt-1 text-sm text-red-600">{errors[`passenger_${index}_firstName`]}</p>
                        )}
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          Soyad *
                        </label>
                        <input
                          type="text"
                          value={passenger.lastName}
                          onChange={(e) => handlePassengerChange(index, 'lastName', e.target.value)}
                          className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                            errors[`passenger_${index}_lastName`] ? 'border-red-300' : 'border-gray-300'
                          }`}
                          placeholder="Yolcu soyadı"
                        />
                        {errors[`passenger_${index}_lastName`] && (
                          <p className="mt-1 text-sm text-red-600">{errors[`passenger_${index}_lastName`]}</p>
                        )}
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          Telefon
                        </label>
                        <input
                          type="tel"
                          value={passenger.phone || ''}
                          onChange={(e) => handlePassengerChange(index, 'phone', e.target.value)}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="+90 555 123 4567"
                        />
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          E-posta
                        </label>
                        <input
                          type="email"
                          value={passenger.email || ''}
                          onChange={(e) => handlePassengerChange(index, 'email', e.target.value)}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="ornek@email.com"
                        />
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>

            {/* Submit Button */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              {errors.submit && (
                <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg">
                  <p className="text-red-600 text-sm flex items-center">
                    <AlertCircle className="h-4 w-4 mr-2" />
                    {errors.submit}
                  </p>
                </div>
              )}

              <button
                type="submit"
                disabled={loading}
                className="w-full flex items-center justify-center px-4 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {loading ? (
                  <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                ) : (
                  <Save className="h-4 w-4 mr-2" />
                )}
                {loading ? 'Kaydediliyor...' : 'Bilet Talebi Oluştur'}
              </button>
            </div>
          </div>
        </div>
      </form>

      {/* Kurumsal Kullanıcı Seçim Modalı */}
      {showUserSelectionModal && (() => {
        // Arama filtresi
        const filteredUsers = corporateUsers.filter(user =>
          `${user.firstName} ${user.lastName}`.toLowerCase().includes(userSearchQuery.toLowerCase()) ||
          user.email.toLowerCase().includes(userSearchQuery.toLowerCase()) ||
          user.department.toLowerCase().includes(userSearchQuery.toLowerCase())
        );

        // Sayfalama hesaplaması
        const totalPages = Math.ceil(filteredUsers.length / usersPerPage);
        const startIndex = (currentPage - 1) * usersPerPage;
        const paginatedUsers = filteredUsers.slice(startIndex, startIndex + usersPerPage);

        return (
          <div className="fixed inset-0 backdrop-blur-sm bg-white/10 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-lg mx-4 max-h-[80vh] flex flex-col">
              {/* Modal Header */}
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-medium text-gray-900">
                  Kurumsal Kullanıcı Seçin
                </h3>
                <button
                  onClick={() => setShowUserSelectionModal(false)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <X className="h-5 w-5" />
                </button>
              </div>

              <p className="text-sm text-gray-600 mb-4">
                {selectedCustomer?.companyName} şirketindeki kullanıcılardan birini seçin:
              </p>

              {/* Arama Kutusu */}
              <div className="relative mb-4">
                <input
                  type="text"
                  value={userSearchQuery}
                  onChange={(e) => {
                    setUserSearchQuery(e.target.value);
                    setCurrentPage(1); // Arama yapınca ilk sayfaya dön
                  }}
                  placeholder="Ad, soyad, e-posta veya departman ara..."
                  className="w-full px-3 py-2 pl-10 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
                <Search className="h-4 w-4 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
              </div>

              {/* Kullanıcı Listesi */}
              <div className="flex-1 overflow-y-auto space-y-2 min-h-0">
                {paginatedUsers.length === 0 ? (
                  <div className="text-center py-8 text-gray-500">
                    <User className="h-8 w-8 mx-auto mb-2 text-gray-300" />
                    <p>Arama kriterlerine uygun kullanıcı bulunamadı</p>
                  </div>
                ) : (
                  paginatedUsers.map((user) => (
                    <div
                      key={user.id}
                      onClick={() => selectCorporateUser(user)}
                      className="flex items-center p-3 rounded-lg border border-gray-200 hover:border-blue-300 hover:bg-blue-50 cursor-pointer transition-colors"
                    >
                      <User className="h-5 w-5 text-gray-400 mr-3 flex-shrink-0" />
                      <div className="flex-1 min-w-0">
                        <p className="font-medium text-gray-900 truncate">
                          {user.firstName} {user.lastName}
                        </p>
                        <p className="text-sm text-gray-600 truncate">{user.email}</p>
                        <p className="text-xs text-gray-500">{user.department}</p>
                      </div>
                    </div>
                  ))
                )}
              </div>

              {/* Sayfalama */}
              {totalPages > 1 && (
                <div className="flex items-center justify-between mt-4 pt-4 border-t border-gray-200">
                  <div className="text-sm text-gray-600">
                    {filteredUsers.length} kullanıcı bulundu
                  </div>
                  <div className="flex items-center space-x-2">
                    <button
                      onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
                      disabled={currentPage === 1}
                      className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      Önceki
                    </button>
                    <span className="text-sm text-gray-600">
                      {currentPage} / {totalPages}
                    </span>
                    <button
                      onClick={() => setCurrentPage(Math.min(totalPages, currentPage + 1))}
                      disabled={currentPage === totalPages}
                      className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      Sonraki
                    </button>
                  </div>
                </div>
              )}

              {/* Alt Butonlar */}
              <div className="mt-4 flex justify-end">
                <button
                  onClick={() => setShowUserSelectionModal(false)}
                  className="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50"
                >
                  İptal
                </button>
              </div>
            </div>
          </div>
        );
      })()}

      {/* Bilet Talebi Özet Modalı */}
      {showSummaryModal && (
        <div className="fixed inset-0 backdrop-blur-sm bg-white/10 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 w-full max-w-2xl mx-4 max-h-[90vh] overflow-y-auto">
            {/* Modal Header */}
            <div className="flex items-center justify-between mb-6">
              <h3 className="text-xl font-bold text-gray-900 flex items-center">
                <Plane className="h-6 w-6 mr-2 text-blue-600" />
                Bilet Talebi Özeti
              </h3>
              <button
                onClick={() => setShowSummaryModal(false)}
                className="text-gray-400 hover:text-gray-600"
              >
                <X className="h-5 w-5" />
              </button>
            </div>

            <div className="space-y-6">
              {/* Öncelik Seviyesi */}
              <div className="bg-orange-50 border border-orange-200 rounded-lg p-4">
                <h4 className="font-medium text-orange-900 flex items-center mb-2">
                  <AlertCircle className="h-4 w-4 mr-2" />
                  Öncelik Seviyesi
                </h4>
                <p className="text-orange-800 capitalize">{
                  formData.priority === 'low' ? 'Düşük' :
                  formData.priority === 'medium' ? 'Orta' :
                  formData.priority === 'high' ? 'Yüksek' : 'Acil'
                }</p>
              </div>

              {/* Cari Bilgileri */}
              <div className="bg-purple-50 border border-purple-200 rounded-lg p-4">
                <h4 className="font-medium text-purple-900 flex items-center mb-3">
                  <User className="h-4 w-4 mr-2" />
                  Cari Bilgileri
                </h4>
                {selectedCustomer && (
                  <div className="text-purple-800">
                    <p className="font-medium">
                      {selectedCustomer.type === 'individual'
                        ? `${selectedCustomer.firstName} ${selectedCustomer.lastName}`
                        : selectedCustomer.companyName
                      }
                    </p>
                    <p className="text-sm">{selectedCustomer.phone}</p>
                    <p className="text-sm">{selectedCustomer.email}</p>
                    {selectedCustomer.isVip && (
                      <span className="inline-flex px-2 py-0.5 text-xs bg-yellow-100 text-yellow-800 rounded-full mt-1">
                        VIP Müşteri
                      </span>
                    )}
                  </div>
                )}
              </div>

              {/* Uçuş Bilgileri */}
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <h4 className="font-medium text-blue-900 flex items-center mb-3">
                  <Plane className="h-4 w-4 mr-2" />
                  Uçuş Bilgileri
                </h4>
                <div className="space-y-2 text-blue-800">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-sm font-medium">Seyahat Türü:</p>
                      <p>{formData.tripType === 'one-way' ? 'Tek Yön' : 'Gidiş-Dönüş'}</p>
                    </div>
                    <div>
                      <p className="text-sm font-medium">Yolcu Sayısı:</p>
                      <p>{formData.passengers.length} Kişi</p>
                    </div>
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-sm font-medium">Kalkış:</p>
                      <p>{getAirportName(formData.departureAirport)} ({formData.departureAirport})</p>
                    </div>
                    <div>
                      <p className="text-sm font-medium">Varış:</p>
                      <p>{getAirportName(formData.arrivalAirport)} ({formData.arrivalAirport})</p>
                    </div>
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-sm font-medium">Gidiş Tarihi:</p>
                      <p>{new Date(formData.departureDate).toLocaleDateString('tr-TR')}</p>
                    </div>
                    {formData.tripType === 'round-trip' && formData.returnDate && (
                      <div>
                        <p className="text-sm font-medium">Dönüş Tarihi:</p>
                        <p>{new Date(formData.returnDate).toLocaleDateString('tr-TR')}</p>
                      </div>
                    )}
                  </div>
                </div>
              </div>

              {/* Seçilen Uçuş */}
              {selectedFlight && (
                <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                  <h4 className="font-medium text-green-900 flex items-center mb-3">
                    ✅ Seçilen Uçuş
                  </h4>
                  <div className="text-green-800 space-y-1">
                    <p className="font-medium">
                      {getAirlineName(selectedFlight.outbound?.segments?.[0]?.carrierCode)} {formData.flightNumber}
                    </p>
                    <p>🕐 {formData.departureTime} - {formData.arrivalTime}</p>
                    <p className="font-medium">💰 {formData.ticketPrice} {formData.currency}</p>
                    <p className="text-sm">
                      🔄 Aktarma: {(() => {
                        const segments = selectedFlight.outbound?.segments?.length || 1;
                        if (segments === 1) return 'Direkt uçuş';
                        return `${segments - 1} aktarma`;
                      })()}
                    </p>
                  </div>
                </div>
              )}

              {/* Yolcu Bilgileri */}
              <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
                <h4 className="font-medium text-gray-900 flex items-center mb-3">
                  <Users className="h-4 w-4 mr-2" />
                  Yolcu Bilgileri ({formData.passengers.length} Kişi)
                </h4>
                <div className="space-y-3">
                  {formData.passengers.map((passenger, index) => (
                    <div key={passenger.id} className="bg-white p-3 rounded border">
                      <div className="flex items-center justify-between mb-2">
                        <h5 className="font-medium text-gray-900">Yolcu {index + 1}</h5>
                      </div>
                      <div className="grid grid-cols-2 gap-3 text-sm">
                        <div>
                          <span className="font-medium">Ad Soyad:</span> {passenger.firstName} {passenger.lastName}
                        </div>
                        {passenger.phone && (
                          <div>
                            <span className="font-medium">Telefon:</span> {passenger.phone}
                          </div>
                        )}
                        {passenger.email && (
                          <div className="col-span-2">
                            <span className="font-medium">E-posta:</span> {passenger.email}
                          </div>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              </div>

              {/* Acente Notları */}
              {formData.agentNotes && (
                <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                  <h4 className="font-medium text-yellow-900 flex items-center mb-2">
                    <MapPin className="h-4 w-4 mr-2" />
                    Acente Notları
                  </h4>
                  <p className="text-yellow-800">{formData.agentNotes}</p>
                </div>
              )}
            </div>

            {/* Modal Footer */}
            <div className="flex items-center justify-end space-x-3 mt-6 pt-6 border-t border-gray-200">
              <button
                onClick={() => setShowSummaryModal(false)}
                className="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50"
              >
                İptal
              </button>
              <button
                onClick={confirmSubmit}
                disabled={loading}
                className="flex items-center px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {loading ? (
                  <>
                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                    Oluşturuluyor...
                  </>
                ) : (
                  <>
                    <Save className="h-4 w-4 mr-2" />
                    Bilet Talebi Oluştur
                  </>
                )}
              </button>
            </div>
          </div>
        </div>
      )}

    </div>
  );
}