'use client';

import React, { useState, useEffect } from 'react';
import { Plus, Eye, Edit, CheckCircle, XCircle, Plane, LayoutGrid, List, RefreshCw } from 'lucide-react';
import Link from 'next/link';
import DataTable, { type TableColumn, type TableAction } from '@/components/ui/DataTable';
import KanbanBoard from '@/components/ui/KanbanBoard';
import {
  renderCustomer,
  renderStatusBadge,
  renderCurrency,
  renderDateTime,
  ticketStatusConfig
} from '@/components/ui/TableRenderers';
import type { TicketRequest } from '@/types';
import { ticketApi } from '@/services/api';
import PermissionGuard from '@/components/guards/PermissionGuard';

export default function TicketsPage() {
  return (
    <PermissionGuard requiredPermissions={['tickets_view', 'tickets_view_own', 'tickets_view_all']}>
      <TicketsPageContent />
    </PermissionGuard>
  );
}

function TicketsPageContent() {
  const [tickets, setTickets] = useState<TicketRequest[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState<string>('all');
  const [viewMode, setViewMode] = useState<'table' | 'kanban'>('table');

  useEffect(() => {
    fetchTickets();
  }, []);

  // Refetch tickets when search or filter changes
  useEffect(() => {
    const timeoutId = setTimeout(() => {
      fetchTickets();
    }, 500); // Debounce search

    return () => clearTimeout(timeoutId);
  }, [searchTerm, statusFilter]);

  const fetchTickets = async () => {
    try {
      setLoading(true);

      // Önce localStorage'dan biletleri al (geçici çözüm)
      let localTickets = [];
      try {
        const storedTickets = localStorage.getItem('vtravel-tickets');
        if (storedTickets) {
          localTickets = JSON.parse(storedTickets);
          console.log('📋 LocalStorage\'dan', localTickets.length, 'bilet yüklendi');
        }
      } catch (error) {
        console.error('LocalStorage okuma hatası:', error);
      }

      // Gerçek API çağrısı (şimdilik hata verse bile devam et)
      let apiTickets = [];
      try {
        const response = await ticketApi.getTickets({
          enrichWithFlightInfo: true,
          search: searchTerm || undefined,
          status: statusFilter !== 'all' ? statusFilter : undefined
        });

        if (response.success) {
          apiTickets = response.data || [];
          console.log('🌐 API\'dan', apiTickets.length, 'bilet yüklendi');
        }
      } catch (error: any) {
        console.error('API çağrısı başarısız, localStorage verisi kullanılıyor:', error.message);
      }

      // localStorage ve API verilerini birleştir (localStorage öncelikli)
      const allTickets = [...localTickets, ...apiTickets];

      // Filtreleme uygula
      let filteredTickets = allTickets;

      if (searchTerm) {
        filteredTickets = allTickets.filter(ticket =>
          ticket.pnr?.toLowerCase().includes(searchTerm.toLowerCase()) ||
          ticket.customerInfo?.firstName?.toLowerCase().includes(searchTerm.toLowerCase()) ||
          ticket.customerInfo?.lastName?.toLowerCase().includes(searchTerm.toLowerCase())
        );
      }

      if (statusFilter !== 'all') {
        filteredTickets = filteredTickets.filter(ticket => ticket.status === statusFilter);
      }

      setTickets(filteredTickets);
    } catch (error: any) {
      console.error('Error fetching tickets:', error);
      setTickets([]);
    } finally {
      setLoading(false);
    }
  };

  const handleStatusChange = async (ticketId: string, newStatus: string) => {
    try {
      console.log('Status change:', ticketId, newStatus);

      // Update ticket status via API
      await ticketApi.updateTicket(ticketId, { status: newStatus });

      // Update local state optimistically
      setTickets(prev => prev.map(ticket =>
        ticket.id === ticketId
          ? { ...ticket, status: newStatus as any, updatedAt: new Date() }
          : ticket
      ));
    } catch (error) {
      console.error('Error updating status:', error);
      // Could add toast notification here for user feedback
    }
  };

  // Filtrelenmiş biletler
  const filteredTickets = tickets.filter(ticket => {
    const customerName = ticket.customer?.type === 'individual'
      ? `${ticket.customer?.firstName || ''} ${ticket.customer?.lastName || ''}`
      : ticket.customer?.companyName || '';

    const passengerNames = ticket.passengers?.map(p => `${p.firstName || ''} ${p.lastName || ''}`).join(' ') || '';

    const requestedByName = ticket.requestedBy
      ? `${ticket.requestedBy.firstName || ''} ${ticket.requestedBy.lastName || ''}`
      : '';

    const matchesSearch =
      customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      passengerNames.toLowerCase().includes(searchTerm.toLowerCase()) ||
      requestedByName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      ticket.flightNumber?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      ticket.pnr?.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesStatus = statusFilter === 'all' || ticket.status === statusFilter;
    
    return matchesSearch && matchesStatus;
  });

  // Kanban için sütunlar
  const kanbanColumns = [
    { id: 'pending', title: 'Bekliyor', color: 'bg-yellow-500', count: 0 },
    { id: 'confirmed', title: 'Onaylandı', color: 'bg-blue-500', count: 0 },
    { id: 'ticketed', title: 'Biletlendi', color: 'bg-green-500', count: 0 },
    { id: 'cancelled', title: 'İptal', color: 'bg-red-500', count: 0 },
    { id: 'rejected', title: 'Reddedildi', color: 'bg-gray-500', count: 0 }
  ];

  // Kanban kartları
  const kanbanCards = filteredTickets.map(ticket => ({
    id: ticket.id,
    title: `${ticket.airline} ${ticket.flightNumber}`,
    subtitle: `${ticket.departureAirport} → ${ticket.arrivalAirport}`,
    customer: ticket.customer,
    status: ticket.status,
    amount: ticket.ticketPrice,
    currency: ticket.currency,
    date: ticket.departureDate,
    tags: [
      `${ticket.passengers?.length || 0} yolcu`,
      ticket.departureTime || ''
    ].filter(Boolean),
    onClick: () => window.location.href = `/tickets/${ticket.id}`,
    onEdit: () => window.location.href = `/tickets/${ticket.id}/edit`
  }));

  // Render yolcu bilgileri
  const renderPassengers = (passengers: any[]) => (
    <div className="space-y-1">
      {passengers.map((passenger, index) => (
        <div key={passenger.id} className="text-sm">
          <div className="font-medium text-gray-900">
            {passenger.firstName} {passenger.lastName}
            {passengers.length > 1 && (
              <span className="ml-2 px-1 py-0.5 text-xs bg-blue-100 text-blue-800 rounded">
                {index + 1}
              </span>
            )}
          </div>
          {passenger.department && (
            <div className="text-xs text-gray-500">
              {passenger.department} • {passenger.position}
            </div>
          )}
        </div>
      ))}
      {passengers.length > 1 && (
        <div className="text-xs text-blue-600 font-medium">
          Toplam {passengers.length} yolcu
        </div>
      )}
    </div>
  );

  // Render enriched flight info with airport details
  const renderFlightInfo = (record: TicketRequest & { departureAirportInfo?: any; arrivalAirportInfo?: any }) => (
    <div>
      <div className="text-sm text-gray-900 font-medium">
        {record.airline} {record.flightNumber}
      </div>
      <div className="text-sm text-gray-500 space-y-1">
        <div className="flex items-center">
          <span className="font-mono text-xs bg-gray-100 px-1 rounded mr-2">
            {record.departureAirport}
          </span>
          →
          <span className="font-mono text-xs bg-gray-100 px-1 rounded ml-2">
            {record.arrivalAirport}
          </span>
        </div>
        {(record.departureAirportInfo || record.arrivalAirportInfo) && (
          <div className="text-xs text-gray-600">
            {record.departureAirportInfo && (
              <div>{record.departureAirportInfo.name}</div>
            )}
            {record.arrivalAirportInfo && (
              <div>→ {record.arrivalAirportInfo.name}</div>
            )}
          </div>
        )}
      </div>
      {record.pnr && (
        <div className="text-xs text-blue-600 font-mono">
          PNR: {record.pnr}
        </div>
      )}
    </div>
  );

  // Tablo sütunları
  const columns: TableColumn<TicketRequest>[] = [
    {
      key: 'customer',
      title: 'Müşteri/Talep Eden',
      render: (_, record) => renderCustomer(record.customer)
    },
    {
      key: 'passengers',
      title: 'Yolcu(lar)',
      render: (_, record) => renderPassengers(record.passengers || [])
    },
    {
      key: 'flight',
      title: 'Uçuş Bilgileri',
      render: (_, record) => renderFlightInfo(record)
    },
    {
      key: 'date',
      title: 'Tarih/Saat',
      render: (_, record) => renderDateTime(record.departureDate, record.departureTime)
    },
    {
      key: 'price',
      title: 'Fiyat',
      render: (_, record) => record.ticketPrice ? renderCurrency(record.ticketPrice, record.currency) : null
    },
    {
      key: 'status',
      title: 'Durum',
      render: (_, record) => renderStatusBadge(record.status, ticketStatusConfig)
    }
  ];

  // Tablo aksiyonları
  const tableActions: TableAction<TicketRequest>[] = [
    {
      key: 'view',
      label: 'Görüntüle',
      icon: Eye,
      onClick: (record) => window.location.href = `/tickets/${record.id}`,
      variant: 'primary'
    },
    {
      key: 'edit',
      label: 'Düzenle',
      icon: Edit,
      onClick: (record) => console.log('Edit:', record.id),
      variant: 'secondary'
    },
    {
      key: 'approve',
      label: 'Onayla',
      icon: CheckCircle,
      onClick: (record) => handleStatusChange(record.id, 'confirmed'),
      variant: 'success',
      condition: (record) => record.status === 'pending'
    },
    {
      key: 'reject',
      label: 'Reddet',
      icon: XCircle,
      onClick: (record) => handleStatusChange(record.id, 'rejected'),
      variant: 'danger',
      condition: (record) => record.status === 'pending'
    },
    {
      key: 'ticket',
      label: 'Biletlendir',
      onClick: (record) => handleStatusChange(record.id, 'ticketed'),
      variant: 'success',
      condition: (record) => record.status === 'confirmed'
    }
  ];

  // Filtreler
  const filters = [
    {
      key: 'status',
      label: 'Durum',
      value: statusFilter,
      onChange: setStatusFilter,
      options: [
        { value: 'all', label: 'Tüm Durumlar' },
        { value: 'pending', label: 'Bekliyor' },
        { value: 'confirmed', label: 'Onaylandı' },
        { value: 'ticketed', label: 'Biletlendi' },
        { value: 'cancelled', label: 'İptal' },
        { value: 'rejected', label: 'Reddedildi' }
      ]
    }
  ];

  // View mode toggle
  const viewModeToggle = (
    <div className="flex items-center bg-gray-100 rounded-lg p-1">
      <button
        onClick={() => setViewMode('table')}
        className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
          viewMode === 'table'
            ? 'bg-white text-gray-900 shadow-sm'
            : 'text-gray-600 hover:text-gray-900'
        }`}
      >
        <List className="h-4 w-4 mr-2" />
        Tablo
      </button>
      <button
        onClick={() => setViewMode('kanban')}
        className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
          viewMode === 'kanban'
            ? 'bg-white text-gray-900 shadow-sm'
            : 'text-gray-600 hover:text-gray-900'
        }`}
      >
        <LayoutGrid className="h-4 w-4 mr-2" />
        Kanban
      </button>
    </div>
  );

  if (viewMode === 'kanban') {

    return (
      <div className="p-6">
        {/* Header */}
        <div className="flex justify-between items-center mb-6">
          <div>
            <h1 className="text-2xl font-bold text-gray-900 flex items-center">
              <Plane className="mr-3 h-8 w-8 text-blue-600" />
              Bilet Talepleri
            </h1>
            <p className="text-gray-600 mt-1">
              Müşterilerden gelen bilet taleplerini yönetin
            </p>
          </div>
          <div className="flex items-center space-x-4">
            {viewModeToggle}
            <button
              onClick={() => fetchTickets()}
              disabled={loading}
              className="flex items-center gap-2 px-3 py-2 text-gray-600 hover:text-gray-900 hover:bg-gray-100 rounded-lg transition-colors disabled:opacity-50"
              title="Yenile"
            >
              <RefreshCw className={`h-4 w-4 ${loading ? 'animate-spin' : ''}`} />
              Yenile
            </button>
            <Link
              href="/tickets/new"
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
            >
              <Plus className="h-4 w-4" />
              Yeni Talep
            </Link>
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <input
                  type="text"
                  placeholder="Müşteri adı, yolcu adı, uçuş numarası veya PNR ara..."
                  className="w-full pl-4 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </div>
            </div>
            <div className="sm:w-48">
              <select
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
              >
                {filters[0].options.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
          </div>
        </div>

        {/* Kanban Board */}
        <KanbanBoard
          columns={kanbanColumns}
          cards={kanbanCards}
          statusConfig={ticketStatusConfig}
          onCardClick={(card) => window.location.href = `/tickets/${card.id}`}
          onCardEdit={(card) => window.location.href = `/tickets/${card.id}/edit`}
          onStatusChange={handleStatusChange}
          loading={loading}
        />
      </div>
    );
  }



  return (
    <div className="p-6">
      <DataTable
        title="Bilet Talepleri"
        description="Müşterilerden gelen bilet taleplerini yönetin"
        data={filteredTickets}
        columns={columns}
        actions={tableActions}
        loading={loading}
        searchable={true}
        searchPlaceholder="Müşteri adı, yolcu adı, uçuş numarası veya PNR ara..."
        searchValue={searchTerm}
        onSearchChange={setSearchTerm}
        filters={filters}
        headerActions={
          <div className="flex items-center space-x-4">
            {viewModeToggle}
            <button
              onClick={() => fetchTickets()}
              disabled={loading}
              className="flex items-center gap-2 px-3 py-2 text-gray-600 hover:text-gray-900 hover:bg-gray-100 rounded-lg transition-colors disabled:opacity-50"
              title="Yenile"
            >
              <RefreshCw className={`h-4 w-4 ${loading ? 'animate-spin' : ''}`} />
              Yenile
            </button>
            <Link
              href="/tickets/new"
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
            >
              <Plus className="h-4 w-4" />
              Yeni Talep
            </Link>
          </div>
        }
        emptyTitle="Bilet talebi bulunamadı"
        emptyDescription="Henüz bilet talebi bulunmuyor."
        emptyIcon={Plane}
      />
    </div>
  );
}