'use client';

import React, { useState, useEffect } from 'react';
import { Car, Plus, Eye, Edit, CheckCircle, XCircle, UserCheck, LayoutGrid, List } from 'lucide-react';
import Link from 'next/link';
import DataTable, { type TableColumn, type TableAction } from '@/components/ui/DataTable';
import KanbanBoard from '@/components/ui/KanbanBoard';
import { 
  renderCustomer, 
  renderStatusBadge, 
  renderTransferInfo,
  renderDateTime,
  renderVehicleInfo,
  renderDriverInfo,
  renderCurrency,
  transferStatusConfig 
} from '@/components/ui/TableRenderers';
import type { TransferRequest } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

export default function TransferRequestsPage() {
  return (
    <PermissionGuard requiredPermissions={['transfer_view']}>
      <TransferRequestsPageContent />
    </PermissionGuard>
  );
}

function TransferRequestsPageContent() {
  const [requests, setRequests] = useState<TransferRequest[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState<string>('all');
  const [typeFilter, setTypeFilter] = useState<string>('all');
  const [viewMode, setViewMode] = useState<'table' | 'kanban'>('table');

  useEffect(() => {
    fetchTransferRequests();
  }, []);

  const fetchTransferRequests = async () => {
    try {
      setLoading(true);
      
      // Mock data for now
      const mockRequests: TransferRequest[] = [
        {
          id: '1',
          customerId: 'c1',
          customer: {
            id: 'c1',
            type: 'individual',
            firstName: 'Ahmet',
            lastName: 'Yılmaz',
            phone: '+905551234567',
            email: 'ahmet@email.com',
            isVip: false,
            isActive: true,
            createdAt: new Date(),
            updatedAt: new Date()
          },
          transferType: 'airport_pickup',
          pickupLocation: 'İstanbul Havalimanı',
          dropoffLocation: 'Sultanahmet Oteli',
          pickupDate: new Date('2025-01-15'),
          pickupTime: '14:30',
          passengers: [
            {
              id: 'tp1',
              firstName: 'Ahmet',
              lastName: 'Yılmaz',
              phone: '+905551234567',
              isChild: false
            },
            {
              id: 'tp2',
              firstName: 'Ayşe',
              lastName: 'Yılmaz',
              phone: '+905551234568',
              isChild: false
            }
          ],
          totalPassengers: 2,
          vehicleType: 'comfort',
          vehicleCount: 1,
          hasLuggage: true,
          needsChildSeat: false,
          isWheelchairAccessible: false,
          status: 'assigned',
          assignedDriver: {
            id: 'd1',
            firstName: 'Mehmet',
            lastName: 'Şoför',
            phone: '+905559876543',
            licenseNumber: 'B123456',
            rating: 4.8,
            isActive: true
          },
          assignedVehicle: {
            id: 'v1',
            brand: 'Mercedes',
            model: 'E-Class',
            year: 2022,
            plateNumber: '34 ABC 123',
            capacity: 4,
            type: 'comfort',
            features: ['Klima', 'GPS', 'Wi-Fi'],
            isActive: true
          },
          estimatedPrice: 250,
          finalPrice: 250,
          currency: 'TRY',
          agentNotes: 'Uçak gecikmesi olabilir, şoföre bildirildi',
          specialRequests: 'Havalimanı çıkışında bekleyecek',
          createdAt: new Date('2025-01-10'),
          updatedAt: new Date('2025-01-12')
        },
        {
          id: '2',
          customerId: 'c2',
          customer: {
            id: 'c2',
            type: 'corporate',
            companyName: 'ABC Teknoloji A.Ş.',
            contactPerson: 'Fatma Kaya',
            phone: '+905559876543',
            email: 'info@abcteknoloji.com',
            isVip: true,
            isActive: true,
            createdAt: new Date(),
            updatedAt: new Date()
          },
          requestedBy: {
            id: 'u1',
            customerId: 'c2',
            firstName: 'Fatma',
            lastName: 'Kaya',
            email: 'fatma.kaya@abcteknoloji.com',
            department: 'İnsan Kaynakları',
            position: 'İK Müdürü',
            permissions: ['create_transfer_request'],
            isActive: true,
            createdAt: new Date(),
            updatedAt: new Date()
          },
          transferType: 'city_tour',
          pickupLocation: 'Hilton Oteli',
          dropoffLocation: 'Hilton Oteli',
          pickupDate: new Date('2025-01-16'),
          pickupTime: '10:00',
          passengers: [
            {
              id: 'tp3',
              firstName: 'Fatma',
              lastName: 'Kaya',
              isChild: false
            },
            {
              id: 'tp4',
              firstName: 'Mehmet',
              lastName: 'Özkan',
              isChild: false
            },
            {
              id: 'tp5',
              firstName: 'Ali',
              lastName: 'Demir',
              age: 8,
              isChild: true
            },
            {
              id: 'tp6',
              firstName: 'Elif',
              lastName: 'Demir',
              age: 5,
              isChild: true
            }
          ],
          totalPassengers: 4,
          vehicleType: 'minibus',
          vehicleCount: 1,
          hasLuggage: false,
          needsChildSeat: true,
          isWheelchairAccessible: false,
          status: 'confirmed',
          estimatedPrice: 800,
          currency: 'TRY',
          agentNotes: 'Kurumsal müşteri - şehir turu',
          specialRequests: 'Çocuk koltuğu gerekli, 8 saatlik tur',
          createdAt: new Date('2025-01-12'),
          updatedAt: new Date('2025-01-13')
        }
      ];
      
      setRequests(mockRequests);
    } catch (error) {
      console.error('Error fetching transfer requests:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleStatusChange = async (requestId: string, newStatus: string) => {
    try {
      console.log('Status change:', requestId, newStatus);
      
      setRequests(prev => prev.map(request => 
        request.id === requestId 
          ? { ...request, status: newStatus as any, updatedAt: new Date() }
          : request
      ));
    } catch (error) {
      console.error('Error updating status:', error);
    }
  };

  // Tablo sütunları
  const columns: TableColumn<TransferRequest>[] = [
    {
      key: 'customer',
      title: 'Müşteri/Talep Eden',
      render: (_, record) => renderCustomer(record.customer)
    },
    {
      key: 'transfer',
      title: 'Transfer Bilgileri',
      render: (_, record) => renderTransferInfo(
        record.transferType,
        record.pickupLocation,
        record.dropoffLocation
      )
    },
    {
      key: 'datetime',
      title: 'Tarih & Saat',
      render: (_, record) => renderDateTime(record.pickupDate, record.pickupTime)
    },
    {
      key: 'vehicle',
      title: 'Araç & Yolcu',
      render: (_, record) => renderVehicleInfo(
        record.vehicleType,
        record.totalPassengers,
        record.vehicleCount
      )
    },
    {
      key: 'driver',
      title: 'Şoför',
      render: (_, record) => renderDriverInfo(record.assignedDriver, record.assignedVehicle)
    },
    {
      key: 'price',
      title: 'Fiyat',
      render: (_, record) => {
        const price = record.finalPrice || record.estimatedPrice;
        return price ? renderCurrency(price, record.currency) : null;
      }
    },
    {
      key: 'status',
      title: 'Durum',
      render: (_, record) => renderStatusBadge(record.status, transferStatusConfig)
    }
  ];

  // Tablo aksiyonları
  const actions: TableAction<TransferRequest>[] = [
    {
      key: 'view',
      label: 'Görüntüle',
      icon: Eye,
      onClick: (record) => window.location.href = `/transfer/requests/${record.id}`,
      variant: 'primary'
    },
    {
      key: 'edit',
      label: 'Düzenle',
      icon: Edit,
      onClick: (record) => console.log('Edit:', record.id),
      variant: 'secondary'
    },
    {
      key: 'quote',
      label: 'Fiyat Ver',
      onClick: (record) => handleStatusChange(record.id, 'quoted'),
      variant: 'success',
      condition: (record) => record.status === 'pending'
    },
    {
      key: 'confirm',
      label: 'Onayla',
      icon: CheckCircle,
      onClick: (record) => handleStatusChange(record.id, 'confirmed'),
      variant: 'success',
      condition: (record) => record.status === 'quoted'
    },
    {
      key: 'assign',
      label: 'Şoför Ata',
      icon: UserCheck,
      onClick: (record) => handleStatusChange(record.id, 'assigned'),
      variant: 'success',
      condition: (record) => record.status === 'confirmed'
    },
    {
      key: 'start',
      label: 'Başlat',
      onClick: (record) => handleStatusChange(record.id, 'in_progress'),
      variant: 'primary',
      condition: (record) => record.status === 'assigned'
    },
    {
      key: 'complete',
      label: 'Tamamla',
      onClick: (record) => handleStatusChange(record.id, 'completed'),
      variant: 'success',
      condition: (record) => record.status === 'in_progress'
    },
    {
      key: 'cancel',
      label: 'İptal Et',
      icon: XCircle,
      onClick: (record) => handleStatusChange(record.id, 'cancelled'),
      variant: 'danger',
      condition: (record) => ['pending', 'quoted', 'confirmed', 'assigned'].includes(record.status)
    }
  ];

  // Filtreler
  const filters = [
    {
      key: 'status',
      label: 'Durum',
      value: statusFilter,
      onChange: setStatusFilter,
      options: [
        { value: 'all', label: 'Tüm Durumlar' },
        { value: 'pending', label: 'Beklemede' },
        { value: 'quoted', label: 'Fiyat Verildi' },
        { value: 'confirmed', label: 'Onaylandı' },
        { value: 'assigned', label: 'Şoför Atandı' },
        { value: 'in_progress', label: 'Devam Ediyor' },
        { value: 'completed', label: 'Tamamlandı' },
        { value: 'cancelled', label: 'İptal Edildi' }
      ]
    },
    {
      key: 'type',
      label: 'Transfer Türü',
      value: typeFilter,
      onChange: setTypeFilter,
      options: [
        { value: 'all', label: 'Tüm Türler' },
        { value: 'airport_pickup', label: 'Havalimanı Karşılama' },
        { value: 'airport_dropoff', label: 'Havalimanı Uğurlama' },
        { value: 'hotel_transfer', label: 'Otel Transferi' },
        { value: 'city_tour', label: 'Şehir Turu' },
        { value: 'point_to_point', label: 'Nokta-Nokta' },
        { value: 'hourly', label: 'Saatlik' },
        { value: 'daily', label: 'Günlük' }
      ]
    }
  ];

  // Filtrelenmiş talepler
  const filteredRequests = requests.filter(request => {
    const customerName = request.customer.type === 'individual' 
      ? `${request.customer.firstName} ${request.customer.lastName}`
      : request.customer.companyName || '';
    
    const requestedByName = request.requestedBy 
      ? `${request.requestedBy.firstName} ${request.requestedBy.lastName}`
      : '';
    
    const driverName = request.assignedDriver 
      ? `${request.assignedDriver.firstName} ${request.assignedDriver.lastName}`
      : '';
    
    const matchesSearch = 
      customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      requestedByName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      request.pickupLocation.toLowerCase().includes(searchTerm.toLowerCase()) ||
      request.dropoffLocation.toLowerCase().includes(searchTerm.toLowerCase()) ||
      driverName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      request.assignedVehicle?.plateNumber.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesStatus = statusFilter === 'all' || request.status === statusFilter;
    const matchesType = typeFilter === 'all' || request.transferType === typeFilter;
    
    return matchesSearch && matchesStatus && matchesType;
  });

  // Kanban için sütunlar
  const kanbanColumns = [
    { id: 'pending', title: 'Beklemede', color: 'bg-yellow-500', count: 0 },
    { id: 'quoted', title: 'Fiyat Verildi', color: 'bg-blue-500', count: 0 },
    { id: 'confirmed', title: 'Onaylandı', color: 'bg-purple-500', count: 0 },
    { id: 'assigned', title: 'Şoför Atandı', color: 'bg-indigo-500', count: 0 },
    { id: 'in_progress', title: 'Devam Ediyor', color: 'bg-orange-500', count: 0 },
    { id: 'completed', title: 'Tamamlandı', color: 'bg-green-500', count: 0 },
    { id: 'cancelled', title: 'İptal Edildi', color: 'bg-red-500', count: 0 }
  ];

  // Kanban kartları
  const kanbanCards = filteredRequests.map(request => ({
    id: request.id,
    title: request.transferType === 'airport_pickup' ? 'Havalimanı Karşılama' :
           request.transferType === 'airport_dropoff' ? 'Havalimanı Uğurlama' :
           request.transferType === 'hotel_transfer' ? 'Otel Transferi' :
           request.transferType === 'city_tour' ? 'Şehir Turu' : 'Transfer',
    subtitle: `${request.pickupLocation} → ${request.dropoffLocation}`,
    customer: request.customer,
    status: request.status,
    amount: request.finalPrice || request.estimatedPrice,
    currency: request.currency,
    date: request.pickupDate,
    tags: [
      `${request.totalPassengers} yolcu`,
      request.vehicleType === 'economy' ? 'Ekonomi' :
      request.vehicleType === 'comfort' ? 'Konfor' :
      request.vehicleType === 'luxury' ? 'Lüks' :
      request.vehicleType === 'minibus' ? 'Minibüs' : request.vehicleType,
      request.assignedDriver ? `${request.assignedDriver.firstName} ${request.assignedDriver.lastName}` : '',
      request.assignedVehicle?.plateNumber || ''
    ].filter(Boolean),
    onClick: () => window.location.href = `/transfer/requests/${request.id}`,
    onEdit: () => window.location.href = `/transfer/requests/${request.id}/edit`
  }));

  // View mode toggle
  const viewModeToggle = (
    <div className="flex items-center bg-gray-100 rounded-lg p-1">
      <button
        onClick={() => setViewMode('table')}
        className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
          viewMode === 'table'
            ? 'bg-white text-gray-900 shadow-sm'
            : 'text-gray-600 hover:text-gray-900'
        }`}
      >
        <List className="h-4 w-4 mr-2" />
        Tablo
      </button>
      <button
        onClick={() => setViewMode('kanban')}
        className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
          viewMode === 'kanban'
            ? 'bg-white text-gray-900 shadow-sm'
            : 'text-gray-600 hover:text-gray-900'
        }`}
      >
        <LayoutGrid className="h-4 w-4 mr-2" />
        Kanban
      </button>
    </div>
  );

  if (viewMode === 'kanban') {
    return (
      <div className="p-6">
        {/* Header */}
        <div className="flex justify-between items-center mb-6">
          <div>
            <h1 className="text-2xl font-bold text-gray-900 flex items-center">
              <Car className="mr-3 h-8 w-8 text-green-600" />
              Transfer Talepleri
            </h1>
            <p className="text-gray-600 mt-1">
              Müşterilerden gelen transfer taleplerini yönetin
            </p>
          </div>
          <div className="flex items-center space-x-4">
            {viewModeToggle}
            <Link
              href="/transfer/requests/new"
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
            >
              <Plus className="h-4 w-4" />
              Yeni Transfer
            </Link>
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <input
                  type="text"
                  placeholder="Müşteri adı, lokasyon, şoför adı veya plaka ara..."
                  className="w-full pl-4 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </div>
            </div>
            <div className="sm:w-48">
              <select
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
              >
                {filters[0].options.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
            <div className="sm:w-48">
              <select
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                value={typeFilter}
                onChange={(e) => setTypeFilter(e.target.value)}
              >
                {filters[1].options.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
          </div>
        </div>

        {/* Kanban Board */}
        <KanbanBoard
          columns={kanbanColumns}
          cards={kanbanCards}
          statusConfig={transferStatusConfig}
          onCardClick={(card) => window.location.href = `/transfer/requests/${card.id}`}
          onCardEdit={(card) => window.location.href = `/transfer/requests/${card.id}/edit`}
          onStatusChange={handleStatusChange}
          loading={loading}
        />
      </div>
    );
  }

  return (
    <div className="p-6">
      <DataTable
        title="Transfer Talepleri"
        description="Müşterilerden gelen transfer taleplerini yönetin"
        data={filteredRequests}
        columns={columns}
        actions={actions}
        loading={loading}
        searchable={true}
        searchPlaceholder="Müşteri adı, lokasyon, şoför adı veya plaka ara..."
        searchValue={searchTerm}
        onSearchChange={setSearchTerm}
        filters={filters}
        headerActions={
          <div className="flex items-center space-x-4">
            {viewModeToggle}
            <Link
              href="/transfer/requests/new"
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
            >
              <Plus className="h-4 w-4" />
              Yeni Transfer
            </Link>
          </div>
        }
        emptyTitle="Transfer talebi bulunamadı"
        emptyDescription="Henüz transfer talebi bulunmuyor."
        emptyIcon={Car}
      />
    </div>
  );
}