'use client';

import React, { useState, useEffect } from 'react';
import { 
  ArrowLeft,
  Save,
  User,
  Mail,
  Phone,
  Building2,
  Shield,
  Eye,
  EyeOff,
  AlertCircle
} from 'lucide-react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';
import type { Department, UserRole, AgencyPermission } from '@/types';

interface FormData {
  firstName: string;
  lastName: string;
  email: string;
  phone: string;
  password: string;
  confirmPassword: string;
  departmentId: string;
  role: UserRole;
  permissions: AgencyPermission[];
  isActive: boolean;
}

export default function NewUserPage() {
  const router = useRouter();
  const [departments, setDepartments] = useState<Department[]>([]);
  const [loading, setLoading] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const [errors, setErrors] = useState<Partial<FormData>>({});
  
  const [formData, setFormData] = useState<FormData>({
    firstName: '',
    lastName: '',
    email: '',
    phone: '',
    password: '',
    confirmPassword: '',
    departmentId: '',
    role: 'specialist',
    permissions: [],
    isActive: true
  });

  useEffect(() => {
    fetchDepartments();
  }, []);

  const fetchDepartments = async () => {
    try {
      // Fetch real departments from API
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/departments`);
      const data = await response.json();

      const departmentsWithDates = data.map((dept: Record<string, unknown>) => ({
        id: dept.id,
        name: dept.name,
        description: dept.description || '',
        color: dept.color,
        parentDepartmentId: dept.parent_department_id,
        managerId: dept.manager_id,
        isActive: Boolean(dept.is_active),
        userCount: dept.user_count || 0,
        permissions: dept.permissions ? (typeof dept.permissions === 'string' ? JSON.parse(dept.permissions) : dept.permissions) : [],
        createdAt: new Date(dept.created_at),
        updatedAt: new Date(dept.updated_at)
      }));

      setDepartments(departmentsWithDates);
    } catch (error) {
      console.error('Error fetching departments:', error);
    }
  };

  const handleInputChange = (field: keyof FormData, value: string | boolean | AgencyPermission[]) => {
    setFormData(prev => ({ ...prev, [field]: value }));

    // Departman değiştiğinde, o departmanın yetkilerini otomatik olarak ata
    if (field === 'departmentId' && typeof value === 'string' && value) {
      const selectedDept = departments.find(d => d.id === value);
      if (selectedDept && selectedDept.permissions) {
        // Departman yetkilerini kullanıcıya ata
        setFormData(prev => ({
          ...prev,
          departmentId: value,
          permissions: selectedDept.permissions as AgencyPermission[]
        }));
      }
    }

    // Clear error when user starts typing
    if (errors[field]) {
      setErrors(prev => ({ ...prev, [field]: undefined }));
    }
  };

  const handlePermissionChange = (permission: AgencyPermission, checked: boolean) => {
    const newPermissions = checked
      ? [...formData.permissions, permission]
      : formData.permissions.filter(p => p !== permission);
    
    handleInputChange('permissions', newPermissions);
  };

  const validateForm = (): boolean => {
    const newErrors: Partial<FormData> = {};

    if (!formData.firstName.trim()) {
      newErrors.firstName = 'Ad gereklidir';
    }

    if (!formData.lastName.trim()) {
      newErrors.lastName = 'Soyad gereklidir';
    }

    if (!formData.email.trim()) {
      newErrors.email = 'E-posta gereklidir';
    } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) {
      newErrors.email = 'Geçerli bir e-posta adresi giriniz';
    }

    if (!formData.password) {
      newErrors.password = 'Şifre gereklidir';
    } else if (formData.password.length < 6) {
      newErrors.password = 'Şifre en az 6 karakter olmalıdır';
    }

    if (formData.password !== formData.confirmPassword) {
      newErrors.confirmPassword = 'Şifreler eşleşmiyor';
    }

    if (!formData.departmentId) {
      newErrors.departmentId = 'Departman seçimi gereklidir';
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!validateForm()) {
      return;
    }

    try {
      setLoading(true);

      // Create user via API
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/agency-users`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          first_name: formData.firstName,
          last_name: formData.lastName,
          email: formData.email,
          phone: formData.phone,
          department_id: formData.departmentId,
          role: formData.role,
          permissions: formData.permissions,
          is_active: formData.isActive
        })
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to create user');
      }

      // Redirect to users list
      router.push('/users');
    } catch (error) {
      console.error('Error creating user:', error);
      alert((error as Error).message || 'Kullanıcı oluşturulurken bir hata oluştu. Lütfen tekrar deneyin.');
    } finally {
      setLoading(false);
    }
  };

  const roleOptions = [
    { value: 'intern', label: 'Stajyer' },
    { value: 'assistant', label: 'Uzman Yardımcısı' },
    { value: 'specialist', label: 'Uzman' },
    { value: 'senior_specialist', label: 'Kıdemli Uzman' },
    { value: 'department_manager', label: 'Departman Yöneticisi' },
    { value: 'agency_manager', label: 'Acente Yöneticisi' },
    { value: 'system_admin', label: 'Sistem Yöneticisi' }
  ];

  const permissionGroups = {
    'Bilet İşlemleri': [
      { key: 'tickets_view_all', label: 'Tüm Biletleri Görüntüle' },
      { key: 'tickets_view_own', label: 'Kendi Biletlerini Görüntüle' },
      { key: 'tickets_create', label: 'Bilet Oluştur' },
      { key: 'tickets_edit', label: 'Bilet Düzenle' },
      { key: 'tickets_delete', label: 'Bilet Sil' },
      { key: 'tickets_approve', label: 'Bilet Onayla' },
      { key: 'tickets_issue', label: 'Bilet Kesme' }
    ],
    'Check-in İşlemleri': [
      { key: 'checkin_view_all', label: 'Tüm Check-in\'leri Görüntüle' },
      { key: 'checkin_view_own', label: 'Kendi Check-in\'lerini Görüntüle' },
      { key: 'checkin_manage', label: 'Check-in Yönet' },
      { key: 'checkin_retry', label: 'Check-in Tekrar Dene' }
    ],
    'Müşteri Yönetimi': [
      { key: 'customers_view_all', label: 'Tüm Müşterileri Görüntüle' },
      { key: 'customers_view_own', label: 'Kendi Müşterilerini Görüntüle' },
      { key: 'customers_create', label: 'Müşteri Oluştur' },
      { key: 'customers_edit', label: 'Müşteri Düzenle' },
      { key: 'customers_delete', label: 'Müşteri Sil' },
      { key: 'customers_export', label: 'Müşteri Dışa Aktar' }
    ],
    'Muhasebe': [
      { key: 'accounting_view_all', label: 'Tüm Muhasebe Kayıtlarını Görüntüle' },
      { key: 'accounting_create', label: 'Muhasebe Kaydı Oluştur' },
      { key: 'accounting_edit', label: 'Muhasebe Kaydı Düzenle' },
      { key: 'accounting_invoice', label: 'Faturalama' },
      { key: 'accounting_payment', label: 'Ödeme İşlemleri' },
      { key: 'accounting_reports', label: 'Mali Raporlar' }
    ],
    'Kullanıcı Yönetimi': [
      { key: 'users_view_all', label: 'Tüm Kullanıcıları Görüntüle' },
      { key: 'users_create', label: 'Kullanıcı Oluştur' },
      { key: 'users_edit', label: 'Kullanıcı Düzenle' },
      { key: 'users_delete', label: 'Kullanıcı Sil' },
      { key: 'users_permissions', label: 'Yetki Yönetimi' }
    ]
  };

  return (
    <div className="p-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center space-x-4">
          <Link
            href="/users"
            className="flex items-center text-gray-600 hover:text-gray-900"
          >
            <ArrowLeft className="h-5 w-5 mr-2" />
            Kullanıcılara Dön
          </Link>
          <div className="h-6 border-l border-gray-300"></div>
          <h1 className="text-2xl font-bold text-gray-900">Yeni Kullanıcı Ekle</h1>
        </div>
      </div>

      <form onSubmit={handleSubmit} className="space-y-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Kişisel Bilgiler */}
          <div className="lg:col-span-2">
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h2 className="text-lg font-medium text-gray-900 mb-4 flex items-center">
                <User className="h-5 w-5 mr-2" />
                Kişisel Bilgiler
              </h2>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Ad *
                  </label>
                  <input
                    type="text"
                    value={formData.firstName}
                    onChange={(e) => handleInputChange('firstName', e.target.value)}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.firstName ? 'border-red-300' : 'border-gray-300'
                    }`}
                    placeholder="Adınızı giriniz"
                  />
                  {errors.firstName && (
                    <p className="mt-1 text-sm text-red-600 flex items-center">
                      <AlertCircle className="h-4 w-4 mr-1" />
                      {errors.firstName}
                    </p>
                  )}
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Soyad *
                  </label>
                  <input
                    type="text"
                    value={formData.lastName}
                    onChange={(e) => handleInputChange('lastName', e.target.value)}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.lastName ? 'border-red-300' : 'border-gray-300'
                    }`}
                    placeholder="Soyadınızı giriniz"
                  />
                  {errors.lastName && (
                    <p className="mt-1 text-sm text-red-600 flex items-center">
                      <AlertCircle className="h-4 w-4 mr-1" />
                      {errors.lastName}
                    </p>
                  )}
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    E-posta *
                  </label>
                  <div className="relative">
                    <Mail className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                    <input
                      type="email"
                      value={formData.email}
                      onChange={(e) => handleInputChange('email', e.target.value)}
                      className={`w-full pl-10 pr-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                        errors.email ? 'border-red-300' : 'border-gray-300'
                      }`}
                      placeholder="ornek@milaturizm.com"
                    />
                  </div>
                  {errors.email && (
                    <p className="mt-1 text-sm text-red-600 flex items-center">
                      <AlertCircle className="h-4 w-4 mr-1" />
                      {errors.email}
                    </p>
                  )}
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Telefon
                  </label>
                  <div className="relative">
                    <Phone className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                    <input
                      type="tel"
                      value={formData.phone}
                      onChange={(e) => handleInputChange('phone', e.target.value)}
                      className="w-full pl-10 pr-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      placeholder="+90 555 123 4567"
                    />
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Şifre *
                  </label>
                  <div className="relative">
                    <input
                      type={showPassword ? 'text' : 'password'}
                      value={formData.password}
                      onChange={(e) => handleInputChange('password', e.target.value)}
                      className={`w-full px-3 py-2 pr-10 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                        errors.password ? 'border-red-300' : 'border-gray-300'
                      }`}
                      placeholder="En az 6 karakter"
                    />
                    <button
                      type="button"
                      onClick={() => setShowPassword(!showPassword)}
                      className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600"
                    >
                      {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                    </button>
                  </div>
                  {errors.password && (
                    <p className="mt-1 text-sm text-red-600 flex items-center">
                      <AlertCircle className="h-4 w-4 mr-1" />
                      {errors.password}
                    </p>
                  )}
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Şifre Tekrar *
                  </label>
                  <div className="relative">
                    <input
                      type={showConfirmPassword ? 'text' : 'password'}
                      value={formData.confirmPassword}
                      onChange={(e) => handleInputChange('confirmPassword', e.target.value)}
                      className={`w-full px-3 py-2 pr-10 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                        errors.confirmPassword ? 'border-red-300' : 'border-gray-300'
                      }`}
                      placeholder="Şifrenizi tekrar giriniz"
                    />
                    <button
                      type="button"
                      onClick={() => setShowConfirmPassword(!showConfirmPassword)}
                      className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600"
                    >
                      {showConfirmPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                    </button>
                  </div>
                  {errors.confirmPassword && (
                    <p className="mt-1 text-sm text-red-600 flex items-center">
                      <AlertCircle className="h-4 w-4 mr-1" />
                      {errors.confirmPassword}
                    </p>
                  )}
                </div>
              </div>
            </div>

            {/* Yetkiler */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h2 className="text-lg font-medium text-gray-900 mb-4 flex items-center">
                <Shield className="h-5 w-5 mr-2" />
                Kullanıcı Yetkileri
              </h2>
              
              <div className="space-y-6">
                {Object.entries(permissionGroups).map(([groupName, permissions]) => (
                  <div key={groupName}>
                    <h3 className="font-medium text-gray-900 mb-3">{groupName}</h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-2">
                      {permissions.map((permission) => (
                        <label key={permission.key} className="flex items-center">
                          <input
                            type="checkbox"
                            checked={formData.permissions.includes(permission.key as AgencyPermission)}
                            onChange={(e) => handlePermissionChange(permission.key as AgencyPermission, e.target.checked)}
                            className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                          />
                          <span className="ml-2 text-sm text-gray-700">{permission.label}</span>
                        </label>
                      ))}
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>

          {/* İş Bilgileri */}
          <div>
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h2 className="text-lg font-medium text-gray-900 mb-4 flex items-center">
                <Building2 className="h-5 w-5 mr-2" />
                İş Bilgileri
              </h2>
              
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Departman *
                  </label>
                  <select
                    value={formData.departmentId}
                    onChange={(e) => handleInputChange('departmentId', e.target.value)}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.departmentId ? 'border-red-300' : 'border-gray-300'
                    }`}
                  >
                    <option value="">Departman seçiniz</option>
                    {departments.map((dept) => (
                      <option key={dept.id} value={dept.id}>
                        {dept.name}
                      </option>
                    ))}
                  </select>
                  {errors.departmentId && (
                    <p className="mt-1 text-sm text-red-600 flex items-center">
                      <AlertCircle className="h-4 w-4 mr-1" />
                      {errors.departmentId}
                    </p>
                  )}
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Rol
                  </label>
                  <select
                    value={formData.role}
                    onChange={(e) => handleInputChange('role', e.target.value as UserRole)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    {roleOptions.map((role) => (
                      <option key={role.value} value={role.value}>
                        {role.label}
                      </option>
                    ))}
                  </select>
                </div>

                <div>
                  <label className="flex items-center">
                    <input
                      type="checkbox"
                      checked={formData.isActive}
                      onChange={(e) => handleInputChange('isActive', e.target.checked)}
                      className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                    />
                    <span className="ml-2 text-sm text-gray-700">Kullanıcı aktif</span>
                  </label>
                </div>
              </div>

              {/* Submit Button */}
              <div className="mt-6 pt-6 border-t border-gray-200">
                <button
                  type="submit"
                  disabled={loading}
                  className="w-full flex items-center justify-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {loading ? (
                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                  ) : (
                    <Save className="h-4 w-4 mr-2" />
                  )}
                  {loading ? 'Kaydediliyor...' : 'Kullanıcı Oluştur'}
                </button>
              </div>
            </div>
          </div>
        </div>
      </form>
    </div>
  );
}