'use client';

import React, { useState, useEffect } from 'react';
import {
  Users,
  Plus,
  Eye,
  Edit,
  Trash2,
  Shield,
  UserCheck,
  UserX,
  Building2,
  Crown,
  Clock,
  Activity,
  LayoutGrid,
  List,
  Filter
} from 'lucide-react';
import Link from 'next/link';
import DataTable, { type TableColumn, type TableAction } from '@/components/ui/DataTable';
import KanbanBoard from '@/components/ui/KanbanBoard';
import type { AgencyUser, Department } from '@/types';
import { agencyUserApi, departmentApi } from '@/services/api';
import PermissionGuard from '@/components/guards/PermissionGuard';

export default function UsersPage() {
  return (
    <PermissionGuard requiredPermissions={['users_view']}>
      <UsersPageContent />
    </PermissionGuard>
  );
}

function UsersPageContent() {
  const [users, setUsers] = useState<AgencyUser[]>([]);
  const [departments, setDepartments] = useState<Department[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [departmentFilter, setDepartmentFilter] = useState<string>('all');
  const [roleFilter, setRoleFilter] = useState<string>('all');
  const [statusFilter, setStatusFilter] = useState<string>('active'); // Varsayılan: sadece aktif kullanıcılar
  const [viewMode, setViewMode] = useState<'table' | 'kanban'>('table');

  useEffect(() => {
    fetchUsers();
    fetchDepartments();
  }, []);

  const fetchUsers = async () => {
    try {
      setLoading(true);

      // Fetch users from API
      const data = await agencyUserApi.getAgencyUsers();

      // Convert API response to frontend format
      const usersWithDates = data.map((user: {
        id: string;
        first_name: string;
        last_name: string;
        email: string;
        phone?: string;
        avatar?: string;
        department_id: string;
        department_name?: string;
        department_color?: string;
        role: string;
        permissions?: string | string[];
        is_active?: boolean;
        is_online?: boolean;
        last_login_at?: string;
        manager_id?: string;
        total_tickets?: number;
        total_revenue?: number;
        created_at: string;
        updated_at: string;
      }) => ({
        id: user.id,
        firstName: user.first_name,
        lastName: user.last_name,
        email: user.email,
        phone: user.phone,
        avatar: user.avatar,
        department: {
          id: user.department_id,
          name: user.department_name || '',
          color: user.department_color || 'bg-gray-500',
          description: ''
        },
        departmentId: user.department_id,
        role: user.role,
        permissions: user.permissions ? (typeof user.permissions === 'string' ? JSON.parse(user.permissions) : user.permissions) : [],
        isActive: Boolean(user.is_active),
        isOnline: Boolean(user.is_online),
        lastLoginAt: user.last_login_at ? new Date(user.last_login_at) : null,
        managerId: user.manager_id,
        totalTickets: user.total_tickets || 0,
        totalRevenue: user.total_revenue || 0,
        createdAt: new Date(user.created_at),
        updatedAt: new Date(user.updated_at)
      }));

      setUsers(usersWithDates);
    } catch (error) {
      console.error('Error fetching users:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchDepartments = async () => {
    try {
      // Fetch departments from API
      const deptData = await departmentApi.getDepartments();
      setDepartments(deptData);
    } catch (error) {
      console.error('Error fetching departments:', error);
    }
  };

  const handleUserStatusChange = async (userId: string, isActive: boolean) => {
    try {
      // Update user status via API
      await agencyUserApi.updateAgencyUser(userId, { is_active: isActive });

      // Update local state
      setUsers(prev => prev.map(user =>
        user.id === userId
          ? { ...user, isActive, updatedAt: new Date() }
          : user
      ));

      alert(isActive ? 'Kullanıcı aktifleştirildi' : 'Kullanıcı pasifleştirildi');
    } catch (error) {
      console.error('Error updating user status:', error);
      alert('Kullanıcı durumu güncellenirken bir hata oluştu');
    }
  };

  const handleUserDeactivate = async (userId: string) => {
    if (!confirm('Bu kullanıcıyı pasifleştirmek istediğinize emin misiniz? Kullanıcı giriş yapamayacak ancak veriler silinmeyecektir.')) {
      return;
    }

    try {
      // Deactivate user via API (soft delete)
      await agencyUserApi.deactivateAgencyUser(userId);

      // Update local state
      setUsers(prev => prev.map(user =>
        user.id === userId
          ? { ...user, isActive: false, updatedAt: new Date() }
          : user
      ));

      alert('Kullanıcı başarıyla pasifleştirildi');
    } catch (error) {
      console.error('Error deactivating user:', error);
      alert('Kullanıcı pasifleştirilirken bir hata oluştu: ' + (error.response?.data?.error || error.message));
    }
  };

  const handleUserDelete = async (userId: string) => {
    if (!confirm('⚠️ UYARI: Bu kullanıcı kalıcı olarak silinecek ve bu işlem geri alınamaz!\n\nDevam etmek istediğinize emin misiniz?')) {
      return;
    }

    try {
      // Permanently delete user via API (hard delete)
      await agencyUserApi.deleteAgencyUser(userId);

      // Remove from local state
      setUsers(prev => prev.filter(user => user.id !== userId));

      alert('Kullanıcı kalıcı olarak silindi');
    } catch (error) {
      console.error('Error deleting user:', error);
      alert('Kullanıcı silinirken bir hata oluştu: ' + (error.response?.data?.error || error.message));
    }
  };

  const renderUser = (user: AgencyUser) => (
    <div className="flex items-center">
      <div className="flex-shrink-0 h-10 w-10 rounded-full bg-gradient-to-r from-blue-600 to-purple-600 flex items-center justify-center">
        {user.avatar ? (
          <img src={user.avatar} alt={user.firstName} className="h-10 w-10 rounded-full" />
        ) : (
          <span className="text-sm font-medium text-white">
            {user.firstName.charAt(0)}{user.lastName.charAt(0)}
          </span>
        )}
      </div>
      <div className="ml-4">
        <div className="text-sm font-medium text-gray-900 flex items-center">
          {user.firstName} {user.lastName}
          {user.isOnline && (
            <div className="ml-2 h-2 w-2 bg-green-500 rounded-full"></div>
          )}
        </div>
        <div className="text-sm text-gray-500">{user.email}</div>
      </div>
    </div>
  );

  // Render departman
  const renderDepartment = (user: AgencyUser) => (
    <div className="flex items-center">
      <div className={`w-3 h-3 rounded-full ${user.department.color} mr-2`}></div>
      <div>
        <div className="text-sm font-medium text-gray-900">{user.department.name}</div>
        <div className="text-xs text-gray-500">{user.department.description}</div>
      </div>
    </div>
  );

  // Render rol
  const renderRole = (role: string) => {
    const roleConfig = {
      system_admin: { color: 'bg-red-100 text-red-800', label: 'Sistem Yöneticisi', icon: Crown },
      agency_manager: { color: 'bg-purple-100 text-purple-800', label: 'Acente Yöneticisi', icon: Shield },
      department_manager: { color: 'bg-blue-100 text-blue-800', label: 'Departman Yöneticisi', icon: Building2 },
      senior_specialist: { color: 'bg-green-100 text-green-800', label: 'Kıdemli Uzman', icon: UserCheck },
      specialist: { color: 'bg-indigo-100 text-indigo-800', label: 'Uzman', icon: Users },
      assistant: { color: 'bg-gray-100 text-gray-800', label: 'Uzman Yardımcısı', icon: Users },
      intern: { color: 'bg-yellow-100 text-yellow-800', label: 'Stajyer', icon: Clock }
    };

    const config = roleConfig[role as keyof typeof roleConfig] || roleConfig.specialist;
    const IconComponent = config.icon;

    return (
      <span className={`inline-flex items-center px-2 py-1 text-xs font-semibold rounded-full ${config.color}`}>
        <IconComponent className="h-3 w-3 mr-1" />
        {config.label}
      </span>
    );
  };

  // Render durum
  const renderStatus = (user: AgencyUser) => (
    <div>
      <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
        user.isActive 
          ? 'bg-green-100 text-green-800' 
          : 'bg-red-100 text-red-800'
      }`}>
        {user.isActive ? 'Aktif' : 'Pasif'}
      </span>
      {user.lastLoginAt && (
        <div className="text-xs text-gray-500 mt-1">
          Son giriş: {user.lastLoginAt.toLocaleDateString('tr-TR')}
        </div>
      )}
    </div>
  );

  // Render performans
  const renderPerformance = (user: AgencyUser) => (
    <div>
      {user.totalTickets && (
        <div className="text-sm text-gray-900">{user.totalTickets} bilet</div>
      )}
      {user.totalRevenue && (
        <div className="text-sm text-green-600 font-medium">
          {user.totalRevenue.toLocaleString()} TRY
        </div>
      )}
    </div>
  );

  // Tablo sütunları
  const columns: TableColumn<AgencyUser>[] = [
    {
      key: 'user',
      title: 'Kullanıcı',
      render: (_, record) => renderUser(record)
    },
    {
      key: 'department',
      title: 'Departman',
      render: (_, record) => renderDepartment(record)
    },
    {
      key: 'role',
      title: 'Rol',
      render: (_, record) => renderRole(record.role)
    },
    {
      key: 'permissions',
      title: 'Yetkiler',
      render: (_, record) => (
        <div className="text-sm text-gray-600">
          {record.permissions ? record.permissions.length : 0} yetki
        </div>
      )
    },
    {
      key: 'performance',
      title: 'Performans',
      render: (_, record) => renderPerformance(record)
    },
    {
      key: 'status',
      title: 'Durum',
      render: (_, record) => renderStatus(record)
    }
  ];

  // Tablo aksiyonları
  const actions: TableAction<AgencyUser>[] = [
    {
      key: 'view',
      label: 'Görüntüle',
      icon: Eye,
      onClick: (record) => window.location.href = `/users/${record.id}`,
      variant: 'primary'
    },
    {
      key: 'edit',
      label: 'Düzenle',
      icon: Edit,
      onClick: (record) => window.location.href = `/users/${record.id}/edit`,
      variant: 'secondary'
    },
    {
      key: 'permissions',
      label: 'Yetkiler',
      icon: Shield,
      onClick: (record) => window.location.href = `/users/${record.id}/permissions`,
      variant: 'secondary'
    },
    {
      key: 'activate',
      label: 'Aktifleştir',
      icon: UserCheck,
      onClick: (record) => handleUserStatusChange(record.id, true),
      variant: 'success',
      condition: (record) => !record.isActive
    },
    {
      key: 'deactivate',
      label: 'Pasifleştir',
      icon: UserX,
      onClick: (record) => handleUserDeactivate(record.id),
      variant: 'warning',
      condition: (record) => record.isActive
    },
    {
      key: 'delete',
      label: 'Kalıcı Sil',
      icon: Trash2,
      onClick: (record) => handleUserDelete(record.id),
      variant: 'danger'
    }
  ];

  // Filtreler
  const filters = [
    {
      key: 'department',
      label: 'Departman',
      value: departmentFilter,
      onChange: setDepartmentFilter,
      options: [
        { value: 'all', label: 'Tüm Departmanlar' },
        ...departments.map(dept => ({ value: dept.id, label: dept.name }))
      ]
    },
    {
      key: 'role',
      label: 'Rol',
      value: roleFilter,
      onChange: setRoleFilter,
      options: [
        { value: 'all', label: 'Tüm Roller' },
        { value: 'system_admin', label: 'Sistem Yöneticisi' },
        { value: 'agency_manager', label: 'Acente Yöneticisi' },
        { value: 'department_manager', label: 'Departman Yöneticisi' },
        { value: 'senior_specialist', label: 'Kıdemli Uzman' },
        { value: 'specialist', label: 'Uzman' },
        { value: 'assistant', label: 'Uzman Yardımcısı' },
        { value: 'intern', label: 'Stajyer' }
      ]
    },
    {
      key: 'status',
      label: 'Durum',
      value: statusFilter,
      onChange: setStatusFilter,
      options: [
        { value: 'all', label: 'Tüm Durumlar' },
        { value: 'active', label: 'Aktif' },
        { value: 'inactive', label: 'Pasif' },
        { value: 'online', label: 'Çevrimiçi' }
      ]
    }
  ];

  // Filtrelenmiş kullanıcılar
  const filteredUsers = users.filter(user => {
    const matchesSearch = 
      `${user.firstName} ${user.lastName}`.toLowerCase().includes(searchTerm.toLowerCase()) ||
      user.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
      user.department.name.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesDepartment = departmentFilter === 'all' || user.departmentId === departmentFilter;
    const matchesRole = roleFilter === 'all' || user.role === roleFilter;
    const matchesStatus = statusFilter === 'all' || 
      (statusFilter === 'active' && user.isActive) ||
      (statusFilter === 'inactive' && !user.isActive) ||
      (statusFilter === 'online' && user.isOnline);
    
    return matchesSearch && matchesDepartment && matchesRole && matchesStatus;
  });

  // Kanban için sütunlar (departmanlara göre)
  const kanbanColumns = departments.map(dept => ({
    id: dept.id,
    title: dept.name,
    color: dept.color,
    count: 0
  }));

  // Kanban kartları
  const kanbanCards = filteredUsers.map(user => ({
    id: user.id,
    title: `${user.firstName} ${user.lastName}`,
    subtitle: user.email,
    customer: user, // User bilgisi için
    status: user.departmentId,
    tags: [
      user.role === 'system_admin' ? 'Sistem Yöneticisi' :
      user.role === 'agency_manager' ? 'Acente Yöneticisi' :
      user.role === 'department_manager' ? 'Departman Yöneticisi' :
      user.role === 'senior_specialist' ? 'Kıdemli Uzman' :
      user.role === 'specialist' ? 'Uzman' :
      user.role === 'assistant' ? 'Uzman Yardımcısı' : 'Stajyer',
      user.isOnline ? 'Çevrimiçi' : 'Çevrimdışı',
      user.permissions ? `${user.permissions.length} yetki` : '0 yetki'
    ].filter(Boolean),
    onClick: () => window.location.href = `/users/${user.id}`,
    onEdit: () => window.location.href = `/users/${user.id}/edit`
  }));

  // View mode toggle
  const viewModeToggle = (
    <div className="flex items-center bg-gray-100 rounded-lg p-1">
      <button
        onClick={() => setViewMode('table')}
        className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
          viewMode === 'table'
            ? 'bg-white text-gray-900 shadow-sm'
            : 'text-gray-600 hover:text-gray-900'
        }`}
      >
        <List className="h-4 w-4 mr-2" />
        Tablo
      </button>
      <button
        onClick={() => setViewMode('kanban')}
        className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
          viewMode === 'kanban'
            ? 'bg-white text-gray-900 shadow-sm'
            : 'text-gray-600 hover:text-gray-900'
        }`}
      >
        <LayoutGrid className="h-4 w-4 mr-2" />
        Departmanlar
      </button>
    </div>
  );

  // İstatistikler
  const stats = {
    total: users.length,
    active: users.filter(u => u.isActive).length,
    online: users.filter(u => u.isOnline).length,
    departments: departments.length
  };

  if (viewMode === 'kanban') {
    return (
      <div className="p-6">
        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
          <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
            <div className="flex items-center">
              <Users className="h-8 w-8 text-blue-600" />
              <div className="ml-3">
                <p className="text-sm font-medium text-gray-500">Toplam Kullanıcı</p>
                <p className="text-2xl font-semibold text-gray-900">{stats.total}</p>
              </div>
            </div>
          </div>
          <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
            <div className="flex items-center">
              <UserCheck className="h-8 w-8 text-green-600" />
              <div className="ml-3">
                <p className="text-sm font-medium text-gray-500">Aktif</p>
                <p className="text-2xl font-semibold text-gray-900">{stats.active}</p>
              </div>
            </div>
          </div>
          <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
            <div className="flex items-center">
              <Activity className="h-8 w-8 text-orange-600" />
              <div className="ml-3">
                <p className="text-sm font-medium text-gray-500">Çevrimiçi</p>
                <p className="text-2xl font-semibold text-gray-900">{stats.online}</p>
              </div>
            </div>
          </div>
          <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
            <div className="flex items-center">
              <Building2 className="h-8 w-8 text-purple-600" />
              <div className="ml-3">
                <p className="text-sm font-medium text-gray-500">Departman</p>
                <p className="text-2xl font-semibold text-gray-900">{stats.departments}</p>
              </div>
            </div>
          </div>
        </div>

        {/* Header */}
        <div className="flex justify-between items-center mb-6">
          <div>
            <h1 className="text-2xl font-bold text-gray-900 flex items-center">
              <Users className="mr-3 h-8 w-8 text-blue-600" />
              Kullanıcı Yönetimi
            </h1>
            <p className="text-gray-600 mt-1">
              Acente kullanıcılarını ve departmanları yönetin
            </p>
          </div>
          <div className="flex items-center space-x-4">
            {viewModeToggle}
            <Link
              href="/departments"
              className="bg-purple-600 text-white px-4 py-2 rounded-lg hover:bg-purple-700 flex items-center gap-2"
            >
              <Building2 className="h-4 w-4" />
              Departmanlar
            </Link>
            <Link
              href="/users/new"
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
            >
              <Plus className="h-4 w-4" />
              Yeni Kullanıcı
            </Link>
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <input
                  type="text"
                  placeholder="Kullanıcı adı, email veya departman ara..."
                  className="w-full pl-4 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </div>
            </div>
            <div className="sm:w-48">
              <select
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                value={roleFilter}
                onChange={(e) => setRoleFilter(e.target.value)}
              >
                {filters[1].options.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
            <div className="sm:w-48">
              <select
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
              >
                {filters[2].options.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
          </div>
        </div>

        {/* Kanban Board */}
        <KanbanBoard
          columns={kanbanColumns}
          cards={kanbanCards}
          statusConfig={{
            dept1: { color: 'bg-blue-100 text-blue-800', label: 'Biletleme' },
            dept2: { color: 'bg-purple-100 text-purple-800', label: 'Konaklama' },
            dept3: { color: 'bg-indigo-100 text-indigo-800', label: 'Vize' },
            dept4: { color: 'bg-green-100 text-green-800', label: 'Muhasebe' },
            dept5: { color: 'bg-orange-100 text-orange-800', label: 'Transfer' }
          }}
          onCardClick={(card) => window.location.href = `/users/${card.id}`}
          onCardEdit={(card) => window.location.href = `/users/${card.id}/edit`}
          onStatusChange={(cardId, newDepartmentId) => {
            // Kullanıcının departmanını değiştir
            console.log('Department change:', cardId, newDepartmentId);
            setUsers(prev => prev.map(user => 
              user.id === cardId 
                ? { 
                    ...user, 
                    departmentId: newDepartmentId,
                    department: departments.find(d => d.id === newDepartmentId) || user.department,
                    updatedAt: new Date() 
                  }
                : user
            ));
          }}
          loading={loading}
        />
      </div>
    );
  }

  return (
    <div className="p-6">
      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Users className="h-8 w-8 text-blue-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Toplam Kullanıcı</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.total}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <UserCheck className="h-8 w-8 text-green-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Aktif</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.active}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Activity className="h-8 w-8 text-orange-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Çevrimiçi</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.online}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Building2 className="h-8 w-8 text-purple-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Departman</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.departments}</p>
            </div>
          </div>
        </div>
      </div>

      <DataTable
        title="Kullanıcı Yönetimi"
        description="Acente kullanıcılarını ve yetkilerini yönetin"
        data={filteredUsers}
        columns={columns}
        actions={actions}
        loading={loading}
        searchable={true}
        searchPlaceholder="Kullanıcı adı, email veya departman ara..."
        searchValue={searchTerm}
        onSearchChange={setSearchTerm}
        filters={filters}
        headerActions={
          <div className="flex items-center space-x-4">
            {viewModeToggle}
            <Link
              href="/departments"
              className="bg-purple-600 text-white px-4 py-2 rounded-lg hover:bg-purple-700 flex items-center gap-2"
            >
              <Building2 className="h-4 w-4" />
              Departmanlar
            </Link>
            <Link
              href="/users/new"
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
            >
              <Plus className="h-4 w-4" />
              Yeni Kullanıcı
            </Link>
          </div>
        }
        emptyTitle="Kullanıcı bulunamadı"
        emptyDescription="Henüz kullanıcı eklenmemiş."
        emptyIcon={Users}
      />
    </div>
  );
}