'use client';

import React, { useState, useEffect } from 'react';
import { CheckCircle, AlertTriangle, Clock, FileText, Calendar, User, Globe } from 'lucide-react';
import PermissionGuard from '@/components/guards/PermissionGuard';

interface VisaControlItem {
  id: string;
  customerName: string;
  passportNo: string;
  country: string;
  visaType: string;
  applicationDate: string;
  appointmentDate?: string;
  status: 'document_check' | 'appointment_scheduled' | 'interview_completed' | 'decision_pending' | 'visa_ready';
  priority: 'low' | 'medium' | 'high';
  nextAction: string;
  daysRemaining: number;
  notes?: string;
}

export default function VisaControlPage() {
  return (
    <PermissionGuard requiredPermissions={['visa_view', 'visa_approve']}>
      <VisaControlPageContent />
    </PermissionGuard>
  );
}

function VisaControlPageContent() {
  const [controlItems, setControlItems] = useState<VisaControlItem[]>([]);
  const [loading, setLoading] = useState(true);
  const [priorityFilter, setPriorityFilter] = useState<string>('all');
  const [statusFilter, setStatusFilter] = useState<string>('all');

  useEffect(() => {
    fetchControlItems();
  }, []);

  const fetchControlItems = async () => {
    try {
      setLoading(true);
      
      // Mock data for now
      const mockItems: VisaControlItem[] = [
        {
          id: '1',
          customerName: 'Mehmet Özkan',
          passportNo: 'T12345678',
          country: 'ABD',
          visaType: 'İş Vizesi',
          applicationDate: '2025-01-10',
          appointmentDate: '2025-01-20',
          status: 'appointment_scheduled',
          priority: 'high',
          nextAction: 'Randevu hatırlatması gönder',
          daysRemaining: 10,
          notes: 'Acil işlem - müşteri erken seyahat planlıyor'
        },
        {
          id: '2',
          customerName: 'Ayşe Demir',
          passportNo: 'T87654321',
          country: 'Schengen',
          visaType: 'İş Vizesi',
          applicationDate: '2025-01-08',
          status: 'document_check',
          priority: 'medium',
          nextAction: 'Eksik belgeleri talep et',
          daysRemaining: 15,
          notes: 'Banka hesap özeti eksik'
        },
        {
          id: '3',
          customerName: 'Can Yıldız',
          passportNo: 'T11223344',
          country: 'İngiltere',
          visaType: 'Turist Vizesi',
          applicationDate: '2025-01-12',
          status: 'visa_ready',
          priority: 'low',
          nextAction: 'Müşteriyi bilgilendir ve teslim et',
          daysRemaining: 0,
          notes: 'Vize hazır, teslim edilmeyi bekliyor'
        }
      ];
      
      setControlItems(mockItems);
    } catch (error) {
      console.error('Error fetching control items:', error);
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'document_check': return 'bg-yellow-100 text-yellow-800';
      case 'appointment_scheduled': return 'bg-blue-100 text-blue-800';
      case 'interview_completed': return 'bg-purple-100 text-purple-800';
      case 'decision_pending': return 'bg-orange-100 text-orange-800';
      case 'visa_ready': return 'bg-green-100 text-green-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusText = (status: string) => {
    switch (status) {
      case 'document_check': return 'Belge Kontrolü';
      case 'appointment_scheduled': return 'Randevu Alındı';
      case 'interview_completed': return 'Mülakat Tamamlandı';
      case 'decision_pending': return 'Karar Bekleniyor';
      case 'visa_ready': return 'Vize Hazır';
      default: return status;
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'high': return 'bg-red-100 text-red-800';
      case 'medium': return 'bg-yellow-100 text-yellow-800';
      case 'low': return 'bg-green-100 text-green-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getPriorityText = (priority: string) => {
    switch (priority) {
      case 'high': return 'Yüksek';
      case 'medium': return 'Orta';
      case 'low': return 'Düşük';
      default: return priority;
    }
  };

  const filteredItems = controlItems.filter(item => {
    const matchesPriority = priorityFilter === 'all' || item.priority === priorityFilter;
    const matchesStatus = statusFilter === 'all' || item.status === statusFilter;
    return matchesPriority && matchesStatus;
  });

  // Öncelik sırasına göre sırala
  const sortedItems = filteredItems.sort((a, b) => {
    const priorityOrder = { high: 3, medium: 2, low: 1 };
    return priorityOrder[b.priority] - priorityOrder[a.priority];
  });

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-1/4"></div>
          <div className="h-64 bg-gray-200 rounded"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      {/* Header */}
      <div className="mb-6">
        <h1 className="text-2xl font-bold text-gray-900 flex items-center">
          <CheckCircle className="mr-3 h-8 w-8 text-blue-600" />
          Vize Kontrol Listesi
        </h1>
        <p className="text-gray-600 mt-1">
          Vize başvuru süreçlerini takip edin ve kontrol edin
        </p>
      </div>

      {/* Quick Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <AlertTriangle className="h-8 w-8 text-red-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Yüksek Öncelik</p>
              <p className="text-2xl font-semibold text-gray-900">
                {controlItems.filter(item => item.priority === 'high').length}
              </p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Clock className="h-8 w-8 text-yellow-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Bu Hafta Randevu</p>
              <p className="text-2xl font-semibold text-gray-900">
                {controlItems.filter(item => item.appointmentDate && 
                  new Date(item.appointmentDate) <= new Date(Date.now() + 7 * 24 * 60 * 60 * 1000)
                ).length}
              </p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <FileText className="h-8 w-8 text-blue-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Belge Kontrolü</p>
              <p className="text-2xl font-semibold text-gray-900">
                {controlItems.filter(item => item.status === 'document_check').length}
              </p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <CheckCircle className="h-8 w-8 text-green-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Hazır Vizeler</p>
              <p className="text-2xl font-semibold text-gray-900">
                {controlItems.filter(item => item.status === 'visa_ready').length}
              </p>
            </div>
          </div>
        </div>
      </div>

      {/* Filters */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
        <div className="flex flex-col sm:flex-row gap-4">
          <div className="sm:w-48">
            <label className="block text-sm font-medium text-gray-700 mb-1">Öncelik</label>
            <select
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              value={priorityFilter}
              onChange={(e) => setPriorityFilter(e.target.value)}
            >
              <option value="all">Tüm Öncelikler</option>
              <option value="high">Yüksek</option>
              <option value="medium">Orta</option>
              <option value="low">Düşük</option>
            </select>
          </div>
          <div className="sm:w-48">
            <label className="block text-sm font-medium text-gray-700 mb-1">Durum</label>
            <select
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              value={statusFilter}
              onChange={(e) => setStatusFilter(e.target.value)}
            >
              <option value="all">Tüm Durumlar</option>
              <option value="document_check">Belge Kontrolü</option>
              <option value="appointment_scheduled">Randevu Alındı</option>
              <option value="interview_completed">Mülakat Tamamlandı</option>
              <option value="decision_pending">Karar Bekleniyor</option>
              <option value="visa_ready">Vize Hazır</option>
            </select>
          </div>
        </div>
      </div>

      {/* Control List */}
      <div className="space-y-4">
        {sortedItems.map((item) => (
          <div key={item.id} className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="flex items-start justify-between">
              <div className="flex-1">
                <div className="flex items-center space-x-4 mb-3">
                  <div>
                    <h3 className="text-lg font-semibold text-gray-900">{item.customerName}</h3>
                    <p className="text-sm text-gray-500">{item.passportNo} • {item.country} • {item.visaType}</p>
                  </div>
                  <div className="flex space-x-2">
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getPriorityColor(item.priority)}`}>
                      {getPriorityText(item.priority)}
                    </span>
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(item.status)}`}>
                      {getStatusText(item.status)}
                    </span>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                  <div className="flex items-center text-sm text-gray-600">
                    <Calendar className="h-4 w-4 mr-2" />
                    <span>Başvuru: {new Date(item.applicationDate).toLocaleDateString('tr-TR')}</span>
                  </div>
                  {item.appointmentDate && (
                    <div className="flex items-center text-sm text-gray-600">
                      <Clock className="h-4 w-4 mr-2" />
                      <span>Randevu: {new Date(item.appointmentDate).toLocaleDateString('tr-TR')}</span>
                    </div>
                  )}
                  <div className="flex items-center text-sm text-gray-600">
                    <User className="h-4 w-4 mr-2" />
                    <span>
                      {item.daysRemaining > 0 ? `${item.daysRemaining} gün kaldı` : 'Süre doldu'}
                    </span>
                  </div>
                </div>

                <div className="bg-blue-50 border border-blue-200 rounded-lg p-3 mb-3">
                  <p className="text-sm font-medium text-blue-900">Sonraki Aksiyon:</p>
                  <p className="text-sm text-blue-800">{item.nextAction}</p>
                </div>

                {item.notes && (
                  <div className="bg-gray-50 border border-gray-200 rounded-lg p-3">
                    <p className="text-sm font-medium text-gray-900">Notlar:</p>
                    <p className="text-sm text-gray-700">{item.notes}</p>
                  </div>
                )}
              </div>

              <div className="ml-6 flex flex-col space-y-2">
                <button className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 text-sm">
                  Güncelle
                </button>
                <button className="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 text-sm">
                  Tamamla
                </button>
                <button className="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 text-sm">
                  Detaylar
                </button>
              </div>
            </div>
          </div>
        ))}
      </div>

      {sortedItems.length === 0 && (
        <div className="text-center py-12 bg-white rounded-lg shadow-sm border border-gray-200">
          <Globe className="mx-auto h-12 w-12 text-gray-400" />
          <h3 className="mt-2 text-sm font-medium text-gray-900">Kontrol edilecek vize başvurusu bulunamadı</h3>
          <p className="mt-1 text-sm text-gray-500">
            Tüm vize başvuruları güncel durumda.
          </p>
        </div>
      )}
    </div>
  );
}