'use client';

import React, { useState, useEffect } from 'react';
import { Globe, Plus, Eye, Edit, CheckCircle, XCircle, FileText, LayoutGrid, List } from 'lucide-react';
import Link from 'next/link';
import DataTable, { type TableColumn, type TableAction } from '@/components/ui/DataTable';
import KanbanBoard from '@/components/ui/KanbanBoard';
import { 
  renderCustomer, 
  renderStatusBadge, 
  renderVisaInfo,
  renderVisaDates,
  renderVisaDocuments,
  renderCurrency,
  visaStatusConfig 
} from '@/components/ui/TableRenderers';
import type { VisaRequest } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

export default function VisaRequestsPage() {
  return (
    <PermissionGuard requiredPermissions={['visa_view']}>
      <VisaRequestsPageContent />
    </PermissionGuard>
  );
}

function VisaRequestsPageContent() {
  const [requests, setRequests] = useState<VisaRequest[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState<string>('all');
  const [countryFilter, setCountryFilter] = useState<string>('all');
  const [viewMode, setViewMode] = useState<'table' | 'kanban'>('table');

  useEffect(() => {
    fetchVisaRequests();
  }, []);

  const fetchVisaRequests = async () => {
    try {
      setLoading(true);
      
      // Mock data for now
      const mockRequests: VisaRequest[] = [
        {
          id: '1',
          customerId: 'c1',
          customer: {
            id: 'c1',
            type: 'individual',
            firstName: 'Mehmet',
            lastName: 'Özkan',
            phone: '+905551234567',
            email: 'mehmet@email.com',
            isVip: false,
            isActive: true,
            createdAt: new Date(),
            updatedAt: new Date()
          },
          applicant: {
            id: 'a1',
            firstName: 'Mehmet',
            lastName: 'Özkan',
            passportNumber: 'T12345678',
            passportExpiry: new Date('2028-05-15'),
            nationality: 'Turkish',
            birthDate: new Date('1985-03-20'),
            birthPlace: 'İstanbul',
            phone: '+905551234567',
            email: 'mehmet@email.com',
            occupation: 'Mühendis',
            employer: 'ABC Teknoloji'
          },
          country: 'ABD',
          visaType: 'business',
          applicationDate: new Date('2025-01-10'),
          travelDate: new Date('2025-03-15'),
          returnDate: new Date('2025-03-25'),
          status: 'processing',
          priority: 'high',
          documents: [
            {
              id: 'd1',
              name: 'Pasaport',
              type: 'passport',
              uploadedAt: new Date(),
              fileUrl: '/docs/passport.pdf',
              isRequired: true,
              status: 'approved'
            },
            {
              id: 'd2',
              name: 'Fotoğraf',
              type: 'photo',
              uploadedAt: new Date(),
              fileUrl: '/docs/photo.jpg',
              isRequired: true,
              status: 'approved'
            }
          ],
          requiredDocuments: ['Pasaport', 'Fotoğraf', 'Banka Hesap Özeti', 'Davet Mektubu'],
          appointmentDate: new Date('2025-01-20'),
          appointmentLocation: 'ABD Konsolosluğu İstanbul',
          applicationFee: 160,
          serviceFee: 500,
          totalAmount: 660,
          currency: 'USD',
          agentNotes: 'Acil işlem gerekli - iş seyahati',
          specialRequests: 'Hızlı randevu talep edildi',
          createdAt: new Date('2025-01-10'),
          updatedAt: new Date('2025-01-12')
        },
        {
          id: '2',
          customerId: 'c2',
          customer: {
            id: 'c2',
            type: 'corporate',
            companyName: 'XYZ İnşaat A.Ş.',
            contactPerson: 'Ayşe Demir',
            phone: '+905559876543',
            email: 'info@xyzinsaat.com',
            isVip: true,
            isActive: true,
            createdAt: new Date(),
            updatedAt: new Date()
          },
          requestedBy: {
            id: 'u1',
            customerId: 'c2',
            firstName: 'Ayşe',
            lastName: 'Demir',
            email: 'ayse.demir@xyzinsaat.com',
            department: 'İnsan Kaynakları',
            position: 'İK Uzmanı',
            permissions: ['create_visa_request'],
            isActive: true,
            createdAt: new Date(),
            updatedAt: new Date()
          },
          applicant: {
            id: 'a2',
            firstName: 'Ayşe',
            lastName: 'Demir',
            passportNumber: 'T87654321',
            passportExpiry: new Date('2027-12-10'),
            nationality: 'Turkish',
            birthDate: new Date('1990-07-15'),
            birthPlace: 'Ankara',
            phone: '+905559876543',
            email: 'ayse.demir@xyzinsaat.com',
            occupation: 'İK Uzmanı',
            employer: 'XYZ İnşaat A.Ş.'
          },
          country: 'Schengen',
          visaType: 'business',
          applicationDate: new Date('2025-01-08'),
          travelDate: new Date('2025-02-20'),
          returnDate: new Date('2025-02-27'),
          status: 'approved',
          priority: 'medium',
          documents: [
            {
              id: 'd3',
              name: 'Pasaport',
              type: 'passport',
              uploadedAt: new Date(),
              fileUrl: '/docs/passport2.pdf',
              isRequired: true,
              status: 'approved'
            },
            {
              id: 'd4',
              name: 'Davet Mektubu',
              type: 'invitation',
              uploadedAt: new Date(),
              fileUrl: '/docs/invitation.pdf',
              isRequired: true,
              status: 'approved'
            }
          ],
          requiredDocuments: ['Pasaport', 'Fotoğraf', 'Davet Mektubu', 'Sigorta'],
          decisionDate: new Date('2025-01-15'),
          visaNumber: 'SCH2025001',
          validUntil: new Date('2025-08-15'),
          applicationFee: 80,
          serviceFee: 300,
          totalAmount: 380,
          currency: 'EUR',
          agentNotes: 'Kurumsal müşteri - iş seyahati onaylandı',
          createdAt: new Date('2025-01-08'),
          updatedAt: new Date('2025-01-15')
        }
      ];
      
      setRequests(mockRequests);
    } catch (error) {
      console.error('Error fetching visa requests:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleStatusChange = async (requestId: string, newStatus: string) => {
    try {
      console.log('Status change:', requestId, newStatus);

      setRequests(prev => prev.map(request =>
        request.id === requestId
          ? { ...request, status: newStatus as VisaRequest['status'], updatedAt: new Date() }
          : request
      ));
    } catch (error) {
      console.error('Error updating status:', error);
    }
  };

  // Tablo sütunları
  const columns: TableColumn<VisaRequest>[] = [
    {
      key: 'customer',
      title: 'Müşteri/Başvuru Sahibi',
      render: (_, record) => renderCustomer(record.customer)
    },
    {
      key: 'visa',
      title: 'Vize Bilgileri',
      render: (_, record) => renderVisaInfo(
        record.country, 
        record.visaType, 
        record.applicant.passportNumber
      )
    },
    {
      key: 'dates',
      title: 'Tarihler',
      render: (_, record) => renderVisaDates(
        record.applicationDate,
        record.travelDate,
        record.appointmentDate
      )
    },
    {
      key: 'documents',
      title: 'Belgeler',
      render: (_, record) => renderVisaDocuments(
        record.documents,
        record.requiredDocuments
      )
    },
    {
      key: 'amount',
      title: 'Tutar',
      render: (_, record) => record.totalAmount ? renderCurrency(record.totalAmount, record.currency) : null
    },
    {
      key: 'status',
      title: 'Durum',
      render: (_, record) => renderStatusBadge(record.status, visaStatusConfig)
    }
  ];

  // Tablo aksiyonları
  const actions: TableAction<VisaRequest>[] = [
    {
      key: 'view',
      label: 'Görüntüle',
      icon: Eye,
      onClick: (record) => window.location.href = `/visa/requests/${record.id}`,
      variant: 'primary'
    },
    {
      key: 'edit',
      label: 'Düzenle',
      icon: Edit,
      onClick: (record) => console.log('Edit:', record.id),
      variant: 'secondary'
    },
    {
      key: 'documents',
      label: 'Belgeler',
      icon: FileText,
      onClick: (record) => console.log('Documents:', record.id),
      variant: 'secondary'
    },
    {
      key: 'approve',
      label: 'Onayla',
      icon: CheckCircle,
      onClick: (record) => handleStatusChange(record.id, 'approved'),
      variant: 'success',
      condition: (record) => record.status === 'processing'
    },
    {
      key: 'reject',
      label: 'Reddet',
      icon: XCircle,
      onClick: (record) => handleStatusChange(record.id, 'rejected'),
      variant: 'danger',
      condition: (record) => ['pending', 'processing'].includes(record.status)
    },
    {
      key: 'deliver',
      label: 'Teslim Et',
      onClick: (record) => handleStatusChange(record.id, 'delivered'),
      variant: 'success',
      condition: (record) => record.status === 'approved'
    }
  ];

  // Filtreler
  const filters = [
    {
      key: 'status',
      label: 'Durum',
      value: statusFilter,
      onChange: setStatusFilter,
      options: [
        { value: 'all', label: 'Tüm Durumlar' },
        { value: 'pending', label: 'Beklemede' },
        { value: 'document_review', label: 'Belge İncelemesi' },
        { value: 'appointment_scheduled', label: 'Randevu Alındı' },
        { value: 'processing', label: 'İşlemde' },
        { value: 'approved', label: 'Onaylandı' },
        { value: 'rejected', label: 'Reddedildi' },
        { value: 'delivered', label: 'Teslim Edildi' }
      ]
    },
    {
      key: 'country',
      label: 'Ülke',
      value: countryFilter,
      onChange: setCountryFilter,
      options: [
        { value: 'all', label: 'Tüm Ülkeler' },
        { value: 'ABD', label: 'ABD' },
        { value: 'Schengen', label: 'Schengen' },
        { value: 'İngiltere', label: 'İngiltere' },
        { value: 'Kanada', label: 'Kanada' },
        { value: 'Avustralya', label: 'Avustralya' }
      ]
    }
  ];

  // Filtrelenmiş talepler
  const filteredRequests = requests.filter(request => {
    const customerName = request.customer.type === 'individual' 
      ? `${request.customer.firstName} ${request.customer.lastName}`
      : request.customer.companyName || '';
    
    const applicantName = `${request.applicant.firstName} ${request.applicant.lastName}`;
    
    const requestedByName = request.requestedBy 
      ? `${request.requestedBy.firstName} ${request.requestedBy.lastName}`
      : '';
    
    const matchesSearch = 
      customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      applicantName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      requestedByName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      request.applicant.passportNumber.toLowerCase().includes(searchTerm.toLowerCase()) ||
      request.country.toLowerCase().includes(searchTerm.toLowerCase()) ||
      request.visaNumber?.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesStatus = statusFilter === 'all' || request.status === statusFilter;
    const matchesCountry = countryFilter === 'all' || request.country === countryFilter;
    
    return matchesSearch && matchesStatus && matchesCountry;
  });

  // Kanban için sütunlar
  const kanbanColumns = [
    { id: 'pending', title: 'Beklemede', color: 'bg-yellow-500', count: 0 },
    { id: 'document_review', title: 'Belge İncelemesi', color: 'bg-blue-500', count: 0 },
    { id: 'appointment_scheduled', title: 'Randevu Alındı', color: 'bg-purple-500', count: 0 },
    { id: 'processing', title: 'İşlemde', color: 'bg-orange-500', count: 0 },
    { id: 'approved', title: 'Onaylandı', color: 'bg-green-500', count: 0 },
    { id: 'rejected', title: 'Reddedildi', color: 'bg-red-500', count: 0 },
    { id: 'delivered', title: 'Teslim Edildi', color: 'bg-green-600', count: 0 }
  ];

  // Kanban kartları
  const kanbanCards = filteredRequests.map(request => ({
    id: request.id,
    title: `${request.country} Vizesi`,
    subtitle: `${request.applicant.firstName} ${request.applicant.lastName}`,
    customer: request.customer,
    status: request.status,
    amount: request.totalAmount,
    currency: request.currency,
    date: request.travelDate,
    tags: [
      request.visaType === 'business' ? 'İş' : 
      request.visaType === 'tourist' ? 'Turist' : 
      request.visaType === 'student' ? 'Öğrenci' : request.visaType,
      request.priority === 'urgent' ? 'Acil' : 
      request.priority === 'high' ? 'Yüksek' : 
      request.priority === 'medium' ? 'Orta' : 'Düşük',
      request.visaNumber || ''
    ].filter(Boolean),
    onClick: () => window.location.href = `/visa/requests/${request.id}`,
    onEdit: () => window.location.href = `/visa/requests/${request.id}/edit`
  }));

  // View mode toggle
  const viewModeToggle = (
    <div className="flex items-center bg-gray-100 rounded-lg p-1">
      <button
        onClick={() => setViewMode('table')}
        className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
          viewMode === 'table'
            ? 'bg-white text-gray-900 shadow-sm'
            : 'text-gray-600 hover:text-gray-900'
        }`}
      >
        <List className="h-4 w-4 mr-2" />
        Tablo
      </button>
      <button
        onClick={() => setViewMode('kanban')}
        className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
          viewMode === 'kanban'
            ? 'bg-white text-gray-900 shadow-sm'
            : 'text-gray-600 hover:text-gray-900'
        }`}
      >
        <LayoutGrid className="h-4 w-4 mr-2" />
        Kanban
      </button>
    </div>
  );

  if (viewMode === 'kanban') {
    return (
      <div className="p-6">
        {/* Header */}
        <div className="flex justify-between items-center mb-6">
          <div>
            <h1 className="text-2xl font-bold text-gray-900 flex items-center">
              <Globe className="mr-3 h-8 w-8 text-indigo-600" />
              Vize Talepleri
            </h1>
            <p className="text-gray-600 mt-1">
              Müşterilerden gelen vize başvurularını yönetin
            </p>
          </div>
          <div className="flex items-center space-x-4">
            {viewModeToggle}
            <Link
              href="/visa/requests/new"
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
            >
              <Plus className="h-4 w-4" />
              Yeni Başvuru
            </Link>
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <input
                  type="text"
                  placeholder="Müşteri adı, pasaport no, ülke veya vize numarası ara..."
                  className="w-full pl-4 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </div>
            </div>
            <div className="sm:w-48">
              <select
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
              >
                {filters[0].options.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
            <div className="sm:w-48">
              <select
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                value={countryFilter}
                onChange={(e) => setCountryFilter(e.target.value)}
              >
                {filters[1].options.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
          </div>
        </div>

        {/* Kanban Board */}
        <KanbanBoard
          columns={kanbanColumns}
          cards={kanbanCards}
          statusConfig={visaStatusConfig}
          onCardClick={(card) => window.location.href = `/visa/requests/${card.id}`}
          onCardEdit={(card) => window.location.href = `/visa/requests/${card.id}/edit`}
          onStatusChange={handleStatusChange}
          loading={loading}
        />
      </div>
    );
  }

  return (
    <div className="p-6">
      <DataTable
        title="Vize Talepleri"
        description="Müşterilerden gelen vize başvurularını yönetin"
        data={filteredRequests}
        columns={columns}
        actions={actions}
        loading={loading}
        searchable={true}
        searchPlaceholder="Müşteri adı, pasaport no, ülke veya vize numarası ara..."
        searchValue={searchTerm}
        onSearchChange={setSearchTerm}
        filters={filters}
        headerActions={
          <div className="flex items-center space-x-4">
            {viewModeToggle}
            <Link
              href="/visa/requests/new"
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
            >
              <Plus className="h-4 w-4" />
              Yeni Başvuru
            </Link>
          </div>
        }
        emptyTitle="Vize başvurusu bulunamadı"
        emptyDescription="Henüz vize başvurusu bulunmuyor."
        emptyIcon={Globe}
      />
    </div>
  );
}