import React from 'react';
import { Shield, CheckCircle2 } from 'lucide-react';
import type { AgencyPermission } from '@/types';
import { PERMISSION_CATEGORIES, DEPARTMENT_PERMISSION_TEMPLATES } from '@/utils/permissions';

interface PermissionSelectorProps {
  selectedPermissions: AgencyPermission[];
  onChange: (permissions: AgencyPermission[]) => void;
  departmentName?: string;
}

export default function PermissionSelector({
  selectedPermissions,
  onChange,
  departmentName
}: PermissionSelectorProps) {

  const handlePermissionToggle = (permission: AgencyPermission) => {
    const newPermissions = selectedPermissions.includes(permission)
      ? selectedPermissions.filter(p => p !== permission)
      : [...selectedPermissions, permission];
    onChange(newPermissions);
  };

  const handleCategoryToggle = (categoryPermissions: AgencyPermission[]) => {
    const allSelected = categoryPermissions.every(p => selectedPermissions.includes(p));

    if (allSelected) {
      // Tümünü kaldır
      const newPermissions = selectedPermissions.filter(
        p => !categoryPermissions.includes(p)
      );
      onChange(newPermissions);
    } else {
      // Tümünü ekle
      const newPermissions = [
        ...selectedPermissions,
        ...categoryPermissions.filter(p => !selectedPermissions.includes(p))
      ];
      onChange(newPermissions);
    }
  };

  const applyTemplate = (templateName: string) => {
    const template = DEPARTMENT_PERMISSION_TEMPLATES[templateName];
    if (template) {
      onChange(template);
    }
  };

  return (
    <div className="space-y-6">
      {/* Yetki Şablonları */}
      {departmentName && (
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
          <h3 className="text-sm font-medium text-blue-900 mb-3 flex items-center">
            <Shield className="h-4 w-4 mr-2" />
            Hızlı Şablonlar
          </h3>
          <div className="flex flex-wrap gap-2">
            {Object.keys(DEPARTMENT_PERMISSION_TEMPLATES).map((templateName) => (
              <button
                key={templateName}
                type="button"
                onClick={() => applyTemplate(templateName)}
                className="px-3 py-1.5 bg-white border border-blue-300 rounded-lg text-sm text-blue-700 hover:bg-blue-100 transition-colors"
              >
                {templateName} Yetkileri
              </button>
            ))}
          </div>
          <p className="mt-2 text-xs text-blue-700">
            Departman türüne uygun hazır yetki setlerinden birini seçebilirsiniz
          </p>
        </div>
      )}

      {/* Yetki Kategorileri */}
      <div className="space-y-4">
        {PERMISSION_CATEGORIES.map((category) => {
          const categoryPermissionValues = category.permissions.map(p => p.value);
          const allSelected = categoryPermissionValues.every(p => selectedPermissions.includes(p));
          const someSelected = categoryPermissionValues.some(p => selectedPermissions.includes(p));

          return (
            <div
              key={category.name}
              className="border border-gray-200 rounded-lg overflow-hidden"
            >
              {/* Kategori Başlığı */}
              <div
                className={`px-4 py-3 flex items-center justify-between cursor-pointer transition-colors ${
                  allSelected
                    ? 'bg-blue-50 border-b border-blue-200'
                    : someSelected
                    ? 'bg-gray-50 border-b border-gray-200'
                    : 'bg-gray-50 hover:bg-gray-100 border-b border-gray-200'
                }`}
                onClick={() => handleCategoryToggle(categoryPermissionValues)}
              >
                <div className="flex items-center">
                  <div className={`w-5 h-5 rounded border-2 mr-3 flex items-center justify-center transition-all ${
                    allSelected
                      ? 'bg-blue-600 border-blue-600'
                      : someSelected
                      ? 'bg-blue-100 border-blue-400'
                      : 'border-gray-300'
                  }`}>
                    {allSelected && <CheckCircle2 className="h-3.5 w-3.5 text-white" />}
                    {someSelected && !allSelected && <div className="w-2 h-2 bg-blue-600 rounded-sm" />}
                  </div>
                  <h3 className="font-medium text-gray-900">{category.label}</h3>
                </div>
                <span className="text-sm text-gray-500">
                  {categoryPermissionValues.filter(p => selectedPermissions.includes(p)).length}/{category.permissions.length}
                </span>
              </div>

              {/* Kategori Yetkileri */}
              <div className="p-4 bg-white space-y-2">
                {category.permissions.map((perm) => {
                  const isSelected = selectedPermissions.includes(perm.value);

                  return (
                    <label
                      key={perm.value}
                      className="flex items-start p-3 rounded-lg hover:bg-gray-50 cursor-pointer transition-colors"
                    >
                      <input
                        type="checkbox"
                        checked={isSelected}
                        onChange={() => handlePermissionToggle(perm.value)}
                        className="mt-0.5 h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                      />
                      <div className="ml-3 flex-1">
                        <div className="text-sm font-medium text-gray-900">
                          {perm.label}
                        </div>
                        {perm.description && (
                          <div className="text-xs text-gray-500 mt-0.5">
                            {perm.description}
                          </div>
                        )}
                      </div>
                    </label>
                  );
                })}
              </div>
            </div>
          );
        })}
      </div>

      {/* Seçili Yetki Özeti */}
      <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
        <h3 className="text-sm font-medium text-gray-900 mb-2">
          Toplam Yetki: {selectedPermissions.length}
        </h3>
        {selectedPermissions.length > 0 ? (
          <div className="flex flex-wrap gap-2">
            {selectedPermissions.map((perm) => (
              <span
                key={perm}
                className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800"
              >
                {PERMISSION_CATEGORIES.flatMap(c => c.permissions)
                  .find(p => p.value === perm)?.label || perm}
              </span>
            ))}
          </div>
        ) : (
          <p className="text-sm text-gray-500">Henüz yetki seçilmedi</p>
        )}
      </div>
    </div>
  );
}
