'use client';

import React, { useState, useRef, useEffect } from 'react';
import {
  Activity,
  Building2,
  Check,
  CheckCheck,
  Clock,
  Mail,
  MessageSquare,
  Paperclip,
  Phone,
  Send
} from 'lucide-react';

export interface ChatMessage {
  id: string;
  content: string;
  timestamp: Date;
  sender: {
    id: string;
    name: string;
    type: 'agent' | 'customer' | 'system';
    avatar?: string;
    isOnline?: boolean;
  };
  type: 'text' | 'system' | 'status_change' | 'file';
  status?: 'sending' | 'sent' | 'delivered' | 'read';
  isInternal?: boolean;
  fileUrl?: string;
  fileName?: string;
}

interface ChatPanelProps {
  messages: ChatMessage[];
  onSendMessage: (content: string, isInternal: boolean) => void;
  onFileUpload?: (file: File) => void;
  currentUser: {
    id: string;
    name: string;
    type: 'agent';
  };
  customer?: {
    id: string;
    name: string;
    type: 'individual' | 'corporate';
    isOnline?: boolean;
  };
  loading?: boolean;
}

export default function ChatPanel({
  messages,
  onSendMessage,
  onFileUpload,
  currentUser,
  customer,
  loading = false
}: ChatPanelProps) {
  const [newMessage, setNewMessage] = useState('');
  const [isInternal, setIsInternal] = useState(false);
  const [activeTab, setActiveTab] = useState<'chat' | 'activity'>('chat');
  const messagesEndRef = useRef<HTMLDivElement>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);

  useEffect(() => {
    scrollToBottom();
  }, [messages]);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  const handleSendMessage = () => {
    if (newMessage.trim()) {
      onSendMessage(newMessage.trim(), isInternal);
      setNewMessage('');
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      handleSendMessage();
    }
  };

  const handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file && onFileUpload) {
      onFileUpload(file);
    }
  };

  const formatTime = (date: Date) => {
    return date.toLocaleTimeString('tr-TR', { 
      hour: '2-digit', 
      minute: '2-digit' 
    });
  };

  const formatDate = (date: Date) => {
    const today = new Date();
    const yesterday = new Date(today);
    yesterday.setDate(yesterday.getDate() - 1);

    if (date.toDateString() === today.toDateString()) {
      return 'Bugün';
    } else if (date.toDateString() === yesterday.toDateString()) {
      return 'Dün';
    } else {
      return date.toLocaleDateString('tr-TR');
    }
  };

  const getMessageStatusIcon = (status?: string) => {
    switch (status) {
      case 'sending':
        return <Clock className="h-3 w-3 text-gray-400" />;
      case 'sent':
        return <Check className="h-3 w-3 text-gray-400" />;
      case 'delivered':
        return <CheckCheck className="h-3 w-3 text-gray-400" />;
      case 'read':
        return <CheckCheck className="h-3 w-3 text-blue-500" />;
      default:
        return null;
    }
  };

  const renderMessage = (message: ChatMessage, index: number) => {
    const isFromCurrentUser = message.sender.id === currentUser.id;
    const isSystem = message.sender.type === 'system';
    const prevMessage = index > 0 ? messages[index - 1] : null;
    const nextMessage = index < messages.length - 1 ? messages[index + 1] : null;
    
    // Tarih ayırıcısı göster
    const showDateSeparator = !prevMessage || 
      formatDate(message.timestamp) !== formatDate(prevMessage.timestamp);

    // Mesaj gruplandırması için
    const isFirstInGroup = !prevMessage || 
      prevMessage.sender.id !== message.sender.id ||
      message.timestamp.getTime() - prevMessage.timestamp.getTime() > 300000; // 5 dakika

    const isLastInGroup = !nextMessage || 
      nextMessage.sender.id !== message.sender.id ||
      nextMessage.timestamp.getTime() - message.timestamp.getTime() > 300000; // 5 dakika

    if (isSystem) {
      return (
        <div key={message.id}>
          {showDateSeparator && (
            <div className="flex justify-center my-4">
              <span className="bg-gray-100 text-gray-600 text-xs px-3 py-1 rounded-full">
                {formatDate(message.timestamp)}
              </span>
            </div>
          )}
          <div className="flex justify-center my-2">
            <div className="bg-gray-100 text-gray-600 text-xs px-3 py-1 rounded-full max-w-xs text-center">
              <Activity className="h-3 w-3 inline mr-1" />
              {message.content}
            </div>
          </div>
        </div>
      );
    }

    return (
      <div key={message.id}>
        {showDateSeparator && (
          <div className="flex justify-center my-4">
            <span className="bg-gray-100 text-gray-600 text-xs px-3 py-1 rounded-full">
              {formatDate(message.timestamp)}
            </span>
          </div>
        )}
        
        <div className={`flex mb-1 ${isFromCurrentUser ? 'justify-end' : 'justify-start'}`}>
          <div className={`flex max-w-xs lg:max-w-md ${isFromCurrentUser ? 'flex-row-reverse' : 'flex-row'}`}>
            {/* Avatar - sadece grup başında göster */}
            {!isFromCurrentUser && isFirstInGroup && (
              <div className="flex-shrink-0 mr-2">
                <div className="w-8 h-8 rounded-full bg-gradient-to-r from-blue-500 to-purple-500 flex items-center justify-center">
                  {message.sender.avatar ? (
                    <img 
                      src={message.sender.avatar} 
                      alt={message.sender.name}
                      className="w-8 h-8 rounded-full"
                    />
                  ) : (
                    <span className="text-xs font-medium text-white">
                      {message.sender.name.charAt(0).toUpperCase()}
                    </span>
                  )}
                </div>
              </div>
            )}
            
            {/* Boşluk - avatar olmadığında */}
            {!isFromCurrentUser && !isFirstInGroup && (
              <div className="w-8 mr-2"></div>
            )}

            <div className={`flex flex-col ${isFromCurrentUser ? 'items-end' : 'items-start'}`}>
              {/* Gönderen adı - sadece grup başında ve karşı taraftan geliyorsa */}
              {!isFromCurrentUser && isFirstInGroup && (
                <div className="flex items-center mb-1">
                  <span className="text-xs text-gray-600 font-medium">
                    {message.sender.name}
                  </span>
                  {customer?.type === 'corporate' && (
                    <Building2 className="h-3 w-3 ml-1 text-gray-400" />
                  )}
                  {message.sender.isOnline && (
                    <div className="w-2 h-2 bg-green-500 rounded-full ml-1"></div>
                  )}
                </div>
              )}

              {/* Mesaj balonu */}
              <div
                className={`px-3 py-2 rounded-2xl relative ${
                  isFromCurrentUser
                    ? message.isInternal
                      ? 'bg-yellow-500 text-white' // İç notlar sarı
                      : 'bg-blue-500 text-white'   // Acente mesajları mavi
                    : 'bg-gray-100 text-gray-900'   // Müşteri mesajları gri
                } ${
                  // Köşe yuvarlaklığı - WhatsApp tarzı
                  isFromCurrentUser
                    ? isLastInGroup
                      ? 'rounded-br-md'
                      : 'rounded-br-2xl'
                    : isLastInGroup
                      ? 'rounded-bl-md'
                      : 'rounded-bl-2xl'
                }`}
              >
                {/* İç not etiketi */}
                {message.isInternal && (
                  <div className="text-xs opacity-75 mb-1">
                    🔒 İç Not
                  </div>
                )}

                {/* Dosya mesajı */}
                {message.type === 'file' && (
                  <div className="flex items-center space-x-2">
                    <Paperclip className="h-4 w-4" />
                    <span className="text-sm">{message.fileName}</span>
                  </div>
                )}

                {/* Metin mesajı */}
                {message.type === 'text' && (
                  <div className="text-sm whitespace-pre-wrap break-words">
                    {message.content}
                  </div>
                )}

                {/* Zaman ve durum - sadece grup sonunda */}
                {isLastInGroup && (
                  <div className={`flex items-center justify-end mt-1 space-x-1 ${
                    isFromCurrentUser ? 'text-white' : 'text-gray-500'
                  }`}>
                    <span className="text-xs opacity-75">
                      {formatTime(message.timestamp)}
                    </span>
                    {isFromCurrentUser && getMessageStatusIcon(message.status)}
                  </div>
                )}
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  };

  const chatMessages = messages.filter(m => m.type !== 'system' && m.type !== 'status_change');
  const activityMessages = messages.filter(m => m.type === 'system' || m.type === 'status_change');

  return (
    <div className="w-96 bg-white border-l border-gray-200 flex flex-col h-full">
      {/* Header */}
      <div className="border-b border-gray-200 p-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            {customer && (
              <>
                <div className="w-10 h-10 rounded-full bg-gradient-to-r from-blue-500 to-purple-500 flex items-center justify-center">
                  <span className="text-sm font-medium text-white">
                    {customer.name.charAt(0).toUpperCase()}
                  </span>
                </div>
                <div>
                  <h3 className="font-medium text-gray-900 flex items-center">
                    {customer.name}
                    {customer.type === 'corporate' && (
                      <Building2 className="h-4 w-4 ml-1 text-gray-400" />
                    )}
                  </h3>
                  <p className="text-xs text-gray-500 flex items-center">
                    {customer.isOnline ? (
                      <>
                        <div className="w-2 h-2 bg-green-500 rounded-full mr-1"></div>
                        Çevrimiçi
                      </>
                    ) : (
                      'Çevrimdışı'
                    )}
                  </p>
                </div>
              </>
            )}
          </div>
          
          {/* Quick Actions */}
          <div className="flex space-x-1">
            <button className="p-2 text-gray-400 hover:text-gray-600 rounded-lg hover:bg-gray-100">
              <Phone className="h-4 w-4" />
            </button>
            <button className="p-2 text-gray-400 hover:text-gray-600 rounded-lg hover:bg-gray-100">
              <Mail className="h-4 w-4" />
            </button>
          </div>
        </div>
      </div>

      {/* Tab Headers */}
      <div className="border-b border-gray-200">
        <nav className="flex">
          <button
            onClick={() => setActiveTab('chat')}
            className={`flex-1 px-4 py-3 text-sm font-medium border-b-2 ${
              activeTab === 'chat'
                ? 'border-blue-500 text-blue-600'
                : 'border-transparent text-gray-500 hover:text-gray-700'
            }`}
          >
            <MessageSquare className="h-4 w-4 inline mr-2" />
            Sohbet ({chatMessages.length})
          </button>
          <button
            onClick={() => setActiveTab('activity')}
            className={`flex-1 px-4 py-3 text-sm font-medium border-b-2 ${
              activeTab === 'activity'
                ? 'border-blue-500 text-blue-600'
                : 'border-transparent text-gray-500 hover:text-gray-700'
            }`}
          >
            <Activity className="h-4 w-4 inline mr-2" />
            Aktivite ({activityMessages.length})
          </button>
        </nav>
      </div>

      {/* Messages Area */}
      <div className="flex-1 overflow-y-auto p-4 space-y-1">
        {activeTab === 'chat' ? (
          <>
            {chatMessages.map((message, index) => renderMessage(message, index))}
            {loading && (
              <div className="flex justify-start">
                <div className="bg-gray-100 rounded-2xl px-4 py-2">
                  <div className="flex space-x-1">
                    <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce"></div>
                    <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '0.1s' }}></div>
                    <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '0.2s' }}></div>
                  </div>
                </div>
              </div>
            )}
            <div ref={messagesEndRef} />
          </>
        ) : (
          <>
            {activityMessages.map((message, index) => renderMessage(message, index))}
            {activityMessages.length === 0 && (
              <div className="text-center py-8">
                <Activity className="h-8 w-8 mx-auto text-gray-400 mb-2" />
                <p className="text-sm text-gray-500">Henüz aktivite yok</p>
              </div>
            )}
          </>
        )}
      </div>

      {/* Message Input - Only for chat tab */}
      {activeTab === 'chat' && (
        <div className="border-t border-gray-200 p-4">
          <div className="space-y-3">
            {/* İç not checkbox */}
            <div className="flex items-center space-x-2">
              <input
                type="checkbox"
                id="internal"
                checked={isInternal}
                onChange={(e) => setIsInternal(e.target.checked)}
                className="rounded border-gray-300 text-yellow-600 focus:ring-yellow-500"
              />
              <label htmlFor="internal" className="text-sm text-gray-600">
                🔒 İç not (sadece ekip görebilir)
              </label>
            </div>
            
            {/* Message input */}
            <div className="flex items-end space-x-2">
              <div className="flex-1">
                <textarea
                  value={newMessage}
                  onChange={(e) => setNewMessage(e.target.value)}
                  onKeyPress={handleKeyPress}
                  placeholder={isInternal ? "İç not yazın..." : "Mesaj yazın..."}
                  className={`w-full px-3 py-2 border rounded-2xl focus:ring-2 focus:border-transparent resize-none ${
                    isInternal 
                      ? 'border-yellow-300 focus:ring-yellow-500' 
                      : 'border-gray-300 focus:ring-blue-500'
                  }`}
                  rows={1}
                  style={{ minHeight: '40px', maxHeight: '120px' }}
                />
              </div>
              
              {/* Action buttons */}
              <div className="flex space-x-1">
                <input
                  type="file"
                  ref={fileInputRef}
                  onChange={handleFileSelect}
                  className="hidden"
                  accept="image/*,.pdf,.doc,.docx"
                />
                <button
                  onClick={() => fileInputRef.current?.click()}
                  className="p-2 text-gray-400 hover:text-gray-600 rounded-full hover:bg-gray-100"
                >
                  <Paperclip className="h-5 w-5" />
                </button>
                <button
                  onClick={handleSendMessage}
                  disabled={!newMessage.trim()}
                  className={`p-2 rounded-full text-white transition-colors ${
                    newMessage.trim()
                      ? isInternal
                        ? 'bg-yellow-500 hover:bg-yellow-600'
                        : 'bg-blue-500 hover:bg-blue-600'
                      : 'bg-gray-300 cursor-not-allowed'
                  }`}
                >
                  <Send className="h-5 w-5" />
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}