'use client';

import React, { useState } from 'react';
import { Eye, Edit, MoreHorizontal, Plus } from 'lucide-react';
import { renderCustomer, renderStatusBadge } from './TableRenderers';

interface KanbanColumn {
  id: string;
  title: string;
  color: string;
  count: number;
}

interface KanbanCard {
  id: string;
  title: string;
  subtitle?: string;
  customer: Record<string, unknown>;
  status: string;
  priority?: 'low' | 'medium' | 'high';
  amount?: number;
  currency?: string;
  date?: Date;
  tags?: string[];
  onClick?: () => void;
  onEdit?: () => void;
}

interface KanbanBoardProps {
  columns: KanbanColumn[];
  cards: KanbanCard[];
  statusConfig: Record<string, { color: string; label: string }>;
  onCardClick?: (card: KanbanCard) => void;
  onCardEdit?: (card: KanbanCard) => void;
  onStatusChange?: (cardId: string, newStatus: string) => void;
  loading?: boolean;
}

export default function KanbanBoard({
  columns,
  cards,
  statusConfig,
  onCardClick,
  onCardEdit,
  onStatusChange,
  loading = false
}: KanbanBoardProps) {
  const [draggedCard, setDraggedCard] = useState<KanbanCard | null>(null);
  const [dragOverColumn, setDragOverColumn] = useState<string | null>(null);

  const handleDragStart = (e: React.DragEvent, card: KanbanCard) => {
    setDraggedCard(card);
    e.dataTransfer.effectAllowed = 'move';
  };

  const handleDragOver = (e: React.DragEvent, columnId: string) => {
    e.preventDefault();
    e.dataTransfer.dropEffect = 'move';
    setDragOverColumn(columnId);
  };

  const handleDragLeave = () => {
    setDragOverColumn(null);
  };

  const handleDrop = (e: React.DragEvent, columnId: string) => {
    e.preventDefault();
    if (draggedCard && onStatusChange && draggedCard.status !== columnId) {
      onStatusChange(draggedCard.id, columnId);
    }
    setDraggedCard(null);
    setDragOverColumn(null);
  };

  const getCardsForColumn = (columnId: string) => {
    return cards.filter(card => card.status === columnId);
  };

  const getPriorityColor = (priority?: string) => {
    switch (priority) {
      case 'high': return 'bg-red-100 text-red-800';
      case 'medium': return 'bg-yellow-100 text-yellow-800';
      case 'low': return 'bg-green-100 text-green-800';
      default: return '';
    }
  };

  if (loading) {
    return (
      <div className="flex space-x-6 overflow-x-auto pb-6">
        {[1, 2, 3, 4].map(i => (
          <div key={i} className="flex-shrink-0 w-80">
            <div className="animate-pulse">
              <div className="h-6 bg-gray-200 rounded mb-4 w-24"></div>
              <div className="space-y-3">
                {[1, 2, 3].map(j => (
                  <div key={j} className="h-32 bg-gray-200 rounded"></div>
                ))}
              </div>
            </div>
          </div>
        ))}
      </div>
    );
  }

  return (
    <div className="flex space-x-6 overflow-x-auto pb-6">
      {columns.map((column) => {
        const columnCards = getCardsForColumn(column.id);
        const isDragOver = dragOverColumn === column.id;
        
        return (
          <div
            key={column.id}
            className="flex-shrink-0 w-80"
            onDragOver={(e) => handleDragOver(e, column.id)}
            onDragLeave={handleDragLeave}
            onDrop={(e) => handleDrop(e, column.id)}
          >
            {/* Column Header */}
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center space-x-2">
                <div className={`w-3 h-3 rounded-full ${column.color}`}></div>
                <h3 className="font-semibold text-gray-900">{column.title}</h3>
                <span className="bg-gray-100 text-gray-600 text-xs px-2 py-1 rounded-full">
                  {columnCards.length}
                </span>
              </div>
              <button className="text-gray-400 hover:text-gray-600">
                <Plus className="h-4 w-4" />
              </button>
            </div>

            {/* Column Content */}
            <div
              className={`min-h-[200px] space-y-3 transition-colors ${
                isDragOver ? 'bg-blue-50 rounded-lg p-2' : ''
              }`}
            >
              {columnCards.map((card) => (
                <div
                  key={card.id}
                  draggable
                  onDragStart={(e) => handleDragStart(e, card)}
                  className="bg-white rounded-lg border border-gray-200 p-4 shadow-sm hover:shadow-md transition-shadow cursor-move group"
                  onClick={() => onCardClick?.(card)}
                >
                  {/* Card Header */}
                  <div className="flex items-start justify-between mb-3">
                    <div className="flex-1">
                      <h4 className="font-medium text-gray-900 text-sm leading-tight">
                        {card.title}
                      </h4>
                      {card.subtitle && (
                        <p className="text-xs text-gray-500 mt-1">{card.subtitle}</p>
                      )}
                    </div>
                    <div className="opacity-0 group-hover:opacity-100 transition-opacity">
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          onCardEdit?.(card);
                        }}
                        className="text-gray-400 hover:text-gray-600"
                      >
                        <MoreHorizontal className="h-4 w-4" />
                      </button>
                    </div>
                  </div>

                  {/* Customer Info */}
                  <div className="mb-3 scale-90 origin-left">
                    {renderCustomer(card.customer)}
                  </div>

                  {/* Tags */}
                  {card.tags && card.tags.length > 0 && (
                    <div className="flex flex-wrap gap-1 mb-3">
                      {card.tags.map((tag, index) => (
                        <span
                          key={index}
                          className="px-2 py-1 text-xs bg-gray-100 text-gray-600 rounded"
                        >
                          {tag}
                        </span>
                      ))}
                    </div>
                  )}

                  {/* Priority */}
                  {card.priority && (
                    <div className="mb-3">
                      <span className={`px-2 py-1 text-xs rounded-full ${getPriorityColor(card.priority)}`}>
                        {card.priority === 'high' ? 'Yüksek' : 
                         card.priority === 'medium' ? 'Orta' : 'Düşük'} Öncelik
                      </span>
                    </div>
                  )}

                  {/* Amount */}
                  {card.amount && (
                    <div className="mb-3">
                      <span className="text-sm font-medium text-green-600">
                        {card.amount.toLocaleString()} {card.currency}
                      </span>
                    </div>
                  )}

                  {/* Date */}
                  {card.date && (
                    <div className="mb-3">
                      <span className="text-xs text-gray-500">
                        {new Date(card.date).toLocaleDateString('tr-TR')}
                      </span>
                    </div>
                  )}

                  {/* Status Badge */}
                  <div className="flex items-center justify-between">
                    {renderStatusBadge(card.status, statusConfig)}
                    <div className="flex space-x-1">
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          onCardClick?.(card);
                        }}
                        className="opacity-0 group-hover:opacity-100 transition-opacity text-blue-600 hover:text-blue-800"
                      >
                        <Eye className="h-3 w-3" />
                      </button>
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          onCardEdit?.(card);
                        }}
                        className="opacity-0 group-hover:opacity-100 transition-opacity text-gray-600 hover:text-gray-800"
                      >
                        <Edit className="h-3 w-3" />
                      </button>
                    </div>
                  </div>
                </div>
              ))}

              {/* Empty State */}
              {columnCards.length === 0 && (
                <div className="text-center py-8 text-gray-400">
                  <div className="text-sm">Bu durumda talep yok</div>
                </div>
              )}
            </div>
          </div>
        );
      })}
    </div>
  );
}