'use client';

import React, { useState } from 'react';
import { Plane, Calendar, Clock, Phone, User, Plus, Mail, Users, X } from 'lucide-react';
import { IMaskInput } from 'react-imask';

interface PassengerData {
  id: number;
  pnrOrTicket: string;
  lastName: string;
  // firstName backend'den gelecek, form'da istenmeyecek
}

interface ManualCheckinFormProps {
  onSubmit: (data: {
    airline: string;
    passengerCount: number;
    passengers: PassengerData[];
    flightDate: string;
    flightTime: string;
    phone: string;
    email: string;
    immediate: boolean;
    seatSelection: string; // Tüm havayolları için koltuk seçimi
  }) => Promise<void>;
  onCancel: () => void;
  loading?: boolean;
}

interface FlightInfo {
  pnr: string;
  airline: string;
  date: string | null;
  departureTime: string | null;
  arrivalTime: string | null;
  origin: string | null;
  destination: string | null;
  originCity: string | null;
  destinationCity: string | null;
  flightNumber: string | null;
  dayOfWeek: string | null;
  checkinAvailable: boolean;
  checkinMessage: string;
  hoursUntilFlight: number | null;
}

export default function ManualCheckinForm({ onSubmit, onCancel, loading = false }: ManualCheckinFormProps) {
  const [formData, setFormData] = useState({
    airline: 'THY',
    passengerCount: 1,
    flightDate: '',
    flightTime: '',
    phone: '',
    email: '',
    immediate: false,
    seatSelection: 'with_seat' // 'with_seat' veya 'without_seat'
  });

  const [passengers, setPassengers] = useState<PassengerData[]>([
    { id: 1, pnrOrTicket: '', lastName: '' }
  ] as PassengerData[]);

  const [errors, setErrors] = useState<Record<string, string>>({});
  const [flightInfo, setFlightInfo] = useState<FlightInfo | null>(null);
  const [loadingFlightInfo, setLoadingFlightInfo] = useState(false);
  const [flightInfoError, setFlightInfoError] = useState<string | null>(null);

  // Uçuş bilgilerini getir
  const fetchFlightInfo = async (pnr: string, lastName: string, airline: string) => {
    if (!pnr || !lastName || pnr.length < 6 || lastName.length < 2) {
      return; // Geçerli bilgi girilmeden sorgu atma
    }

    setLoadingFlightInfo(true);
    setFlightInfoError(null);
    setFlightInfo(null);

    try {
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/checkin/flight-info`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ pnr, lastName, airline }),
      });

      const data = await response.json();

      if (data.success && data.flightInfo) {
        setFlightInfo(data.flightInfo);

        // Uçuş bilgisinden tarih ve saati otomatik doldur
        if (data.flightInfo.date && data.flightInfo.departureTime) {
          // "11 Ekim 2025" → "2025-10-11" formatına çevir
          const dateStr = convertTurkishDateToISO(data.flightInfo.date);
          if (dateStr) {
            setFormData(prev => ({
              ...prev,
              flightDate: dateStr,
              flightTime: data.flightInfo.departureTime || '',
              immediate: data.flightInfo.checkinAvailable // Check-in açıksa immediate=true
            }));
          }
        }
      } else {
        setFlightInfoError(data.error || 'Uçuş bilgisi alınamadı');
      }
    } catch (error) {
      console.error('Flight info fetch error:', error);
      setFlightInfoError('Sunucu hatası, uçuş bilgisi sorgulanamadı');
    } finally {
      setLoadingFlightInfo(false);
    }
  };

  // Türkçe tarih → ISO formatına çevir (11 Ekim 2025 → 2025-10-11)
  const convertTurkishDateToISO = (dateStr: string): string | null => {
    const monthMap: Record<string, string> = {
      'Ocak': '01', 'Oca': '01',
      'Şubat': '02', 'Şub': '02',
      'Mart': '03', 'Mar': '03',
      'Nisan': '04', 'Nis': '04',
      'Mayıs': '05', 'May': '05',
      'Haziran': '06', 'Haz': '06',
      'Temmuz': '07', 'Tem': '07',
      'Ağustos': '08', 'Ağu': '08',
      'Eylül': '09', 'Eyl': '09',
      'Ekim': '10', 'Eki': '10',
      'Kasım': '11', 'Kas': '11',
      'Aralık': '12', 'Ara': '12'
    };

    try {
      const parts = dateStr.split(' ');
      if (parts.length === 3) {
        const day = parts[0].padStart(2, '0');
        const month = monthMap[parts[1]];
        const year = parts[2];
        if (month) {
          return `${year}-${month}-${day}`;
        }
      }
    } catch (error) {
      console.error('Tarih çevirme hatası:', error);
    }
    return null;
  };

  const validateForm = () => {
    const newErrors: Record<string, string> = {};

    // Yolcu bilgileri kontrolü
    passengers.forEach((passenger, index) => {
      if (!passenger.pnrOrTicket.trim()) {
        const fieldName = formData.airline === 'THY' && formData.passengerCount > 1
          ? 'Bilet numarası'
          : 'PNR kodu';
        newErrors[`passenger_${index}_pnrOrTicket`] = `${fieldName} zorunludur`;
      } else if (passenger.pnrOrTicket.length < 6) {
        const fieldName = formData.airline === 'THY' && formData.passengerCount > 1
          ? 'Bilet numarası'
          : 'PNR kodu';
        newErrors[`passenger_${index}_pnrOrTicket`] = `${fieldName} en az 6 karakter olmalıdır`;
      }

      // firstName backend'den gelecek, form'da validate etmeye gerek yok

      if (!passenger.lastName.trim()) {
        newErrors[`passenger_${index}_lastName`] = 'Soyisim zorunludur';
      } else if (passenger.lastName.length < 2) {
        newErrors[`passenger_${index}_lastName`] = 'Soyisim en az 2 karakter olmalıdır';
      }
    });

    // Hemen yap seçili değilse tarih ve saat zorunlu
    if (!formData.immediate) {
      if (!formData.flightDate.trim()) {
        newErrors.flightDate = 'Uçuş tarihi zorunludur';
      }

      if (!formData.flightTime.trim()) {
        newErrors.flightTime = 'Uçuş saati zorunludur';
      }
    }

    // THY için telefon, AJET için hem telefon hem email, diğerleri için email zorunlu
    if (formData.airline === 'THY') {
      if (!formData.phone || formData.phone.trim() === '') {
        newErrors.phone = 'GSM numarası zorunludur';
      } else {
        // IMaskInput unmask=true olduğu için phone zaten temiz gelir (sadece rakamlar)
        const cleanPhone = formData.phone.replace(/\D/g, ''); // Tüm rakam olmayanları temizle
        // Türk telefon numaraları 11 haneli (0 ile başlar: 05XX XXX XX XX)
        if (cleanPhone.length !== 11) {
          newErrors.phone = 'Geçerli bir GSM numarası girin (11 haneli, 0 ile başlamalı)';
        } else if (!cleanPhone.startsWith('0')) {
          newErrors.phone = 'GSM numarası 0 ile başlamalıdır';
        }
      }
    } else if (formData.airline === 'AJET') {
      // AJET için hem email hem telefon zorunlu
      if (!formData.email.trim()) {
        newErrors.email = 'AJET için e-posta adresi zorunludur';
      } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) {
        newErrors.email = 'Geçerli bir e-posta adresi girin';
      }

      if (!formData.phone || formData.phone.trim() === '') {
        newErrors.phone = 'AJET için telefon numarası zorunludur';
      } else {
        // IMaskInput unmask=true olduğu için phone zaten temiz gelir (sadece rakamlar)
        const cleanPhone = formData.phone.replace(/\D/g, ''); // Tüm rakam olmayanları temizle
        // Türk telefon numaraları 11 haneli (0 ile başlar: 05XX XXX XX XX)
        if (cleanPhone.length !== 11) {
          newErrors.phone = 'Geçerli bir GSM numarası girin (11 haneli, 0 ile başlamalı)';
        } else if (!cleanPhone.startsWith('0')) {
          newErrors.phone = 'GSM numarası 0 ile başlamalıdır';
        }
      }
    } else {
      if (!formData.email.trim()) {
        newErrors.email = 'E-posta adresi zorunludur';
      } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) {
        newErrors.email = 'Geçerli bir e-posta adresi girin';
      }
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!validateForm()) {
      console.log('Form validation failed:', errors);
      return;
    }

    console.log('Form data being submitted:', formData, passengers);

    try {
      await onSubmit({
        ...formData,
        passengers: passengers.map(p => ({
          ...p,
          pnrOrTicket: p.pnrOrTicket.toUpperCase(),
          lastName: p.lastName.toUpperCase()
          // firstName backend'den gelecek
        }))
      });

      // Form'u reset etme - modal parent tarafından kapatılacak
      console.log('✅ Check-in başarıyla başlatıldı');

    } catch (error) {
      console.error('Form submission error:', error);
      // Hata durumunda form açık kalmalı
    }
  };

  const handleInputChange = (field: string, value: string | boolean) => {
    setFormData(prev => ({ ...prev, [field]: value }));

    // Clear error when user starts typing
    if (errors[field]) {
      setErrors(prev => ({ ...prev, [field]: '' }));
    }
  };

  const handlePassengerCountChange = (count: number) => {
    setFormData(prev => ({ ...prev, passengerCount: count }));

    // Yolcu sayısına göre passenger array'ini güncelle
    const newPassengers: PassengerData[] = [];
    for (let i = 0; i < count; i++) {
      newPassengers.push({
        id: i + 1,
        pnrOrTicket: passengers[i]?.pnrOrTicket || '',
        lastName: passengers[i]?.lastName || ''
      });
    }
    setPassengers(newPassengers);
  };

  const handlePassengerChange = (index: number, field: keyof PassengerData, value: string) => {
    const updatedPassengers = passengers.map((passenger, i) =>
      i === index ? { ...passenger, [field]: value } : passenger
    );
    setPassengers(updatedPassengers);

    // Clear error when user starts typing
    const errorKey = `passenger_${index}_${field}`;
    if (errors[errorKey]) {
      setErrors(prev => ({ ...prev, [errorKey]: '' }));
    }

    // Otomatik sorgu kaldırıldı - sadece "Listeye Ekle" butonuna basıldığında backend'de kontrol yapılacak
  };

  const getFieldLabel = () => {
    return formData.airline === 'THY' && formData.passengerCount > 1
      ? 'Bilet Numarası'
      : 'PNR Kodu';
  };

  const getFieldPlaceholder = () => {
    return formData.airline === 'THY' && formData.passengerCount > 1
      ? '1234567890123'
      : 'ABC123';
  };

  return (
    <div className="fixed inset-0 bg-gray-900/20 backdrop-blur-sm flex items-center justify-center p-4 z-50">
      <div className="bg-white rounded-xl shadow-2xl w-full max-w-4xl max-h-[90vh] overflow-y-auto border border-gray-200">
        <div className="px-6 py-4 border-b border-gray-200 flex items-center justify-between bg-gradient-to-r from-blue-600 to-blue-700 rounded-t-xl">
          <div className="flex items-center">
            <Plane className="h-5 w-5 text-white mr-2" />
            <h3 className="text-lg font-semibold text-white">Manuel Check-in Ekle</h3>
          </div>
          <button
            onClick={onCancel}
            className="p-2 hover:bg-blue-500 rounded-full transition-colors"
            disabled={loading}
          >
            <X className="h-5 w-5 text-white" />
          </button>
        </div>

        <form onSubmit={handleSubmit} className="p-6">
          <div className="space-y-6">
            {/* Havayolu ve Yolcu Sayısı - en üstte */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div>
                <label htmlFor="airline" className="block text-sm font-medium text-gray-700 mb-2">
                  <Plane className="h-4 w-4 inline mr-1" />
                  Havayolu *
                </label>
                <select
                  id="airline"
                  className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors ${
                    errors.airline ? 'border-red-300' : 'border-gray-300'
                  }`}
                  value={formData.airline}
                  onChange={(e) => handleInputChange('airline', e.target.value)}
                  disabled={loading}
                >
                  <option value="THY">Turkish Airlines (THY)</option>
                  <option value="PC">Pegasus Airlines (PC)</option>
                  <option value="AJET">AnadoluJet (AJET)</option>
                </select>
                {errors.airline && (
                  <p className="text-red-500 text-sm mt-1">{errors.airline}</p>
                )}
              </div>

              {/* Tüm havayolları için check-in tipi */}
              <div>
                <label htmlFor="seatSelection" className="block text-sm font-medium text-gray-700 mb-2">
                  <Plane className="h-4 w-4 inline mr-1" />
                  Check-in Tipi *
                </label>
                <select
                  id="seatSelection"
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                  value={formData.seatSelection}
                  onChange={(e) => handleInputChange('seatSelection', e.target.value)}
                  disabled={loading}
                >
                  <option value="without_seat">Koltuksuz Check-in</option>
                  <option value="with_seat">Koltuklu Check-in</option>
                </select>
                <p className="text-xs text-gray-500 mt-1">
                  {formData.seatSelection === 'without_seat'
                    ? 'Koltuk otomatik atanır, e-posta ile gönderilir'
                    : 'Koltuk zaten seçilmiş biletler için'
                  }
                </p>
              </div>

              <div>
                <label htmlFor="passengerCount" className="block text-sm font-medium text-gray-700 mb-2">
                  <Users className="h-4 w-4 inline mr-1" />
                  Yolcu Sayısı *
                </label>
                <select
                  id="passengerCount"
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                  value={formData.passengerCount}
                  onChange={(e) => handlePassengerCountChange(parseInt(e.target.value))}
                  disabled={loading}
                >
                  {[1,2,3,4,5,6,7,8,9].map(num => (
                    <option key={num} value={num}>{num} Yolcu</option>
                  ))}
                </select>
              </div>
            </div>

            {/* Yolcu Bilgileri */}
            <div className="space-y-4">
              <h4 className="text-md font-medium text-gray-900 border-b pb-2">
                Yolcu Bilgileri
                {formData.airline === 'THY' && formData.passengerCount > 1 && (
                  <span className="text-sm text-orange-600 font-normal ml-2">
                    ⚠️ Birden fazla yolcu için bilet numaralarını girin
                  </span>
                )}
              </h4>

              {passengers.map((passenger, index) => (
                <div key={passenger.id} className="grid grid-cols-1 md:grid-cols-2 gap-4 p-4 bg-gray-50 rounded-lg">
                  <div>
                    <label htmlFor={`pnrOrTicket_${index}`} className="block text-sm font-medium text-gray-700 mb-2">
                      <User className="h-4 w-4 inline mr-1" />
                      {getFieldLabel()} {index + 1} *
                    </label>
                    <input
                      type="text"
                      id={`pnrOrTicket_${index}`}
                      required
                      className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors ${
                        errors[`passenger_${index}_pnrOrTicket`] ? 'border-red-300' : 'border-gray-300'
                      }`}
                      placeholder={getFieldPlaceholder()}
                      value={passenger.pnrOrTicket}
                      onChange={(e) => handlePassengerChange(index, 'pnrOrTicket', e.target.value.toUpperCase())}
                      disabled={loading}
                    />
                    {errors[`passenger_${index}_pnrOrTicket`] && (
                      <p className="text-red-500 text-sm mt-1">{errors[`passenger_${index}_pnrOrTicket`]}</p>
                    )}
                  </div>

                  <div>
                    <label htmlFor={`lastName_${index}`} className="block text-sm font-medium text-gray-700 mb-2">
                      <User className="h-4 w-4 inline mr-1" />
                      Soyisim {index + 1} *
                      <span className="text-xs text-gray-500 ml-2">(İsim otomatik gelecek)</span>
                    </label>
                    <input
                      type="text"
                      id={`lastName_${index}`}
                      required
                      className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors ${
                        errors[`passenger_${index}_lastName`] ? 'border-red-300' : 'border-gray-300'
                      }`}
                      placeholder="YILMAZ"
                      value={passenger.lastName}
                      onChange={(e) => handlePassengerChange(index, 'lastName', e.target.value.toUpperCase())}
                      disabled={loading}
                    />
                    {errors[`passenger_${index}_lastName`] && (
                      <p className="text-red-500 text-sm mt-1">{errors[`passenger_${index}_lastName`]}</p>
                    )}
                  </div>
                </div>
              ))}
            </div>

            {/* Uçuş Bilgileri Bölümü - AJET/PC için */}
            {(formData.airline === 'AJET' || formData.airline === 'PC') && (
              <div className="space-y-3">
                {/* Loading */}
                {loadingFlightInfo && (
                  <div className="flex items-center justify-center p-4 bg-blue-50 rounded-lg border border-blue-200">
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-blue-600 mr-3"></div>
                    <span className="text-sm text-blue-900">Uçuş bilgileri sorgulanıyor...</span>
                  </div>
                )}

                {/* Error */}
                {flightInfoError && !loadingFlightInfo && (
                  <div className="p-4 bg-red-50 rounded-lg border border-red-200">
                    <p className="text-sm text-red-800">⚠️ {flightInfoError}</p>
                  </div>
                )}

                {/* Flight Info Success */}
                {flightInfo && !loadingFlightInfo && (
                  <div className="space-y-3">
                    {/* Rota Bilgisi */}
                    <div className="p-4 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg border border-blue-200">
                      <div className="flex items-center justify-between">
                        <div className="text-center">
                          <p className="text-xs text-gray-600 mb-1">Kalkış</p>
                          <p className="text-2xl font-bold text-blue-900">{flightInfo.origin || '—'}</p>
                          <p className="text-xs text-gray-600">{flightInfo.departureTime || '—'}</p>
                        </div>
                        <div className="flex-1 px-4">
                          <div className="flex items-center justify-center">
                            <div className="flex-1 h-px bg-blue-300"></div>
                            <Plane className="h-5 w-5 text-blue-600 mx-2 transform rotate-90" />
                            <div className="flex-1 h-px bg-blue-300"></div>
                          </div>
                          {flightInfo.flightNumber && (
                            <p className="text-xs text-center text-gray-600 mt-1">{flightInfo.flightNumber}</p>
                          )}
                        </div>
                        <div className="text-center">
                          <p className="text-xs text-gray-600 mb-1">Varış</p>
                          <p className="text-2xl font-bold text-blue-900">{flightInfo.destination || '—'}</p>
                          <p className="text-xs text-gray-600">{flightInfo.arrivalTime || '—'}</p>
                        </div>
                      </div>
                      {flightInfo.date && (
                        <div className="mt-3 pt-3 border-t border-blue-200">
                          <p className="text-sm text-center text-gray-700">
                            <Calendar className="h-4 w-4 inline mr-1" />
                            {flightInfo.date}
                            {flightInfo.dayOfWeek && ` (${flightInfo.dayOfWeek})`}
                          </p>
                        </div>
                      )}
                    </div>

                    {/* Check-in Durumu */}
                    <div className={`p-4 rounded-lg border ${
                      flightInfo.checkinAvailable
                        ? 'bg-green-50 border-green-200'
                        : 'bg-yellow-50 border-yellow-200'
                    }`}>
                      <div className="flex items-start">
                        <div className={`flex-shrink-0 w-10 h-10 rounded-full flex items-center justify-center ${
                          flightInfo.checkinAvailable ? 'bg-green-100' : 'bg-yellow-100'
                        }`}>
                          {flightInfo.checkinAvailable ? (
                            <span className="text-2xl">✓</span>
                          ) : (
                            <Clock className="h-5 w-5 text-yellow-600" />
                          )}
                        </div>
                        <div className="ml-3 flex-1">
                          <h4 className={`text-sm font-semibold ${
                            flightInfo.checkinAvailable ? 'text-green-900' : 'text-yellow-900'
                          }`}>
                            {flightInfo.checkinAvailable ? 'Check-in Açık!' : 'Check-in Henüz Açılmadı'}
                          </h4>
                          <p className={`text-xs mt-1 ${
                            flightInfo.checkinAvailable ? 'text-green-700' : 'text-yellow-700'
                          }`}>
                            {flightInfo.checkinMessage}
                          </p>
                          {flightInfo.checkinAvailable && (
                            <p className="text-xs mt-2 text-green-600">
                              💡 &quot;Hemen Check-in Yap&quot; otomatik seçildi
                            </p>
                          )}
                        </div>
                      </div>
                    </div>
                  </div>
                )}
              </div>
            )}

            {/* Hemen Yap Seçeneği */}
            <div className="flex items-center space-x-3 p-4 bg-blue-50 rounded-lg border border-blue-200">
              <input
                type="checkbox"
                id="immediate"
                checked={formData.immediate}
                onChange={(e) => handleInputChange('immediate', e.target.checked)}
                className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                disabled={loading || (flightInfo !== null && !flightInfo.checkinAvailable)}
              />
              <label htmlFor="immediate" className="text-sm font-medium text-blue-900">
                🚀 Hemen Check-in Yap
                {flightInfo && !flightInfo.checkinAvailable && (
                  <span className="text-xs text-yellow-700 ml-2">(Check-in açılana kadar bekleyecek)</span>
                )}
              </label>
            </div>

            {/* İletişim Bilgileri */}
            <div>
              <h4 className="text-md font-medium text-gray-900 border-b pb-2 mb-4">İletişim Bilgileri</h4>
              <div className="space-y-4">
                {formData.airline === 'THY' ? (
                  <>
                    {/* THY - Sadece Telefon */}
                    <div>
                      <label htmlFor="phone" className="block text-sm font-medium text-gray-700 mb-2">
                        <Phone className="h-4 w-4 inline mr-1" />
                        GSM Numarası *
                      </label>
                      <IMaskInput
                        mask="0000 000 00 00"
                        value={formData.phone}
                        unmask={true}
                        onAccept={(value) => handleInputChange('phone', value)}
                        placeholder="05XX XXX XX XX"
                        disabled={loading}
                        type="tel"
                        id="phone"
                        required
                        className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors ${
                          errors.phone ? 'border-red-300' : 'border-gray-300'
                        }`}
                      />
                      <p className="text-xs text-gray-500 mt-1">Türkiye (TR +90)</p>
                      {errors.phone && (
                        <p className="text-red-500 text-sm mt-1">{errors.phone}</p>
                      )}
                    </div>
                  </>
                ) : formData.airline === 'AJET' ? (
                  <>
                    {/* AJET - Hem Email Hem Telefon */}
                    <div>
                      <label htmlFor="email" className="block text-sm font-medium text-gray-700 mb-2">
                        <Mail className="h-4 w-4 inline mr-1" />
                        E-posta Adresi *
                      </label>
                      <input
                        type="email"
                        id="email"
                        required
                        className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors ${
                          errors.email ? 'border-red-300' : 'border-gray-300'
                        }`}
                        placeholder="ornek@email.com"
                        value={formData.email}
                        onChange={(e) => handleInputChange('email', e.target.value)}
                        disabled={loading}
                      />
                      {errors.email && (
                        <p className="text-red-500 text-sm mt-1">{errors.email}</p>
                      )}
                    </div>
                    <div>
                      <label htmlFor="phone" className="block text-sm font-medium text-gray-700 mb-2">
                        <Phone className="h-4 w-4 inline mr-1" />
                        Telefon Numarası *
                      </label>
                      <IMaskInput
                        mask="0000 000 00 00"
                        value={formData.phone}
                        unmask={true}
                        onAccept={(value) => handleInputChange('phone', value)}
                        placeholder="05XX XXX XX XX"
                        disabled={loading}
                        type="tel"
                        id="phone"
                        required
                        className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors ${
                          errors.phone ? 'border-red-300' : 'border-gray-300'
                        }`}
                      />
                      <p className="text-xs text-gray-500 mt-1">Türkiye (TR +90)</p>
                      {errors.phone && (
                        <p className="text-red-500 text-sm mt-1">{errors.phone}</p>
                      )}
                    </div>
                  </>
                ) : (
                  <>
                    {/* Diğer Havayolları - Sadece Email */}
                    <div>
                      <label htmlFor="email" className="block text-sm font-medium text-gray-700 mb-2">
                        <Mail className="h-4 w-4 inline mr-1" />
                        E-posta Adresi *
                      </label>
                      <input
                        type="email"
                        id="email"
                        required
                        className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors ${
                          errors.email ? 'border-red-300' : 'border-gray-300'
                        }`}
                        placeholder="ornek@email.com"
                        value={formData.email}
                        onChange={(e) => handleInputChange('email', e.target.value)}
                        disabled={loading}
                      />
                      {errors.email && (
                        <p className="text-red-500 text-sm mt-1">{errors.email}</p>
                      )}
                    </div>
                  </>
                )}
              </div>
            </div>

            {/* Uçuş Tarihi ve Saati - Sadece hemen yap seçili değilse göster */}
            {!formData.immediate && (
              <div>
                <h4 className="text-md font-medium text-gray-900 border-b pb-2 mb-4">Uçuş Bilgileri</h4>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label htmlFor="flightDate" className="block text-sm font-medium text-gray-700 mb-2">
                      <Calendar className="h-4 w-4 inline mr-1" />
                      Uçuş Tarihi *
                    </label>
                    <input
                      type="date"
                      id="flightDate"
                      required
                      className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors ${
                        errors.flightDate ? 'border-red-300' : 'border-gray-300'
                      }`}
                      value={formData.flightDate}
                      onChange={(e) => handleInputChange('flightDate', e.target.value)}
                      disabled={loading}
                    />
                    {errors.flightDate && (
                      <p className="text-red-500 text-sm mt-1">{errors.flightDate}</p>
                    )}
                  </div>

                  <div>
                    <label htmlFor="flightTime" className="block text-sm font-medium text-gray-700 mb-2">
                      <Clock className="h-4 w-4 inline mr-1" />
                      Uçuş Saati *
                    </label>
                    <input
                      type="time"
                      id="flightTime"
                      required
                      className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors ${
                        errors.flightTime ? 'border-red-300' : 'border-gray-300'
                      }`}
                      value={formData.flightTime}
                      onChange={(e) => handleInputChange('flightTime', e.target.value)}
                      disabled={loading}
                    />
                    {errors.flightTime && (
                      <p className="text-red-500 text-sm mt-1">{errors.flightTime}</p>
                    )}
                  </div>
                </div>
              </div>
            )}

          </div>

          {/* Action buttons */}
          <div className="flex justify-end space-x-3 pt-6 border-t border-gray-200 mt-6">
            <button
              type="button"
              onClick={onCancel}
              disabled={loading}
              className="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
            >
              İptal
            </button>
            <button
              type="submit"
              disabled={loading}
              className="px-6 py-2 text-sm font-medium text-white bg-blue-600 border border-transparent rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed transition-colors flex items-center"
            >
              {loading ? (
                <>
                  <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                  İşleniyor...
                </>
              ) : (
                <>
                  <Plus className="h-4 w-4 mr-2" />
                  Listeye Ekle
                </>
              )}
            </button>
          </div>
        </form>
      </div>
    </div>
  );
}