import React from 'react';
import { Building2, User, Crown, Plane, Hotel, Globe, Car, FileText } from 'lucide-react';
import type { Customer } from '@/types';

// Müşteri bilgisi render edici
export const renderCustomer = (customer?: Customer) => {
    if (!customer) {
        return (
            <div className="flex items-center">
                <div className="flex-shrink-0 h-10 w-10 rounded-full flex items-center justify-center bg-gray-100">
                    <User className="h-5 w-5 text-gray-400" />
                </div>
                <div className="ml-4">
                    <div className="text-sm font-medium text-gray-500">
                        Müşteri bilgisi yok
                    </div>
                </div>
            </div>
        );
    }

    return (
        <div className="flex items-center">
            <div className={`flex-shrink-0 h-10 w-10 rounded-full flex items-center justify-center ${customer?.type === 'corporate' ? 'bg-purple-100' : 'bg-blue-100'
                }`}>
                {customer?.type === 'corporate' ? (
                    <Building2 className="h-5 w-5 text-purple-600" />
                ) : (
                    <User className="h-5 w-5 text-blue-600" />
                )}
            </div>
            <div className="ml-4">
                <div className="text-sm font-medium text-gray-900 flex items-center">
                    {customer?.type === 'individual'
                        ? `***** ${customer?.lastName || ''}`
                        : `***** ${customer?.companyName || 'Şirket'}`
                }
                {customer?.isVip && (
                    <Crown className="ml-2 h-4 w-4 text-yellow-500" />
                )}
            </div>
            <div className="text-sm text-gray-500">{customer?.phone || ''}</div>
        </div>
    </div>
    );
};

// Durum badge render edici
export const renderStatusBadge = (status: string, statusConfig: Record<string, { color: string; label: string }>) => {
    const config = statusConfig[status] || { color: 'bg-gray-100 text-gray-800', label: status };
    return (
        <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${config.color}`}>
            {config.label}
        </span>
    );
};

// Para birimi render edici
export const renderCurrency = (amount: number, currency: string = 'TRY') => (
    <span className="text-sm font-medium text-gray-900">
        {amount.toLocaleString()} {currency}
    </span>
);

// Tarih render edici
export const renderDate = (date: Date | string) => {
    const dateObj = typeof date === 'string' ? new Date(date) : date;
    return (
        <div className="text-sm text-gray-900">
            {dateObj.toLocaleDateString('tr-TR')}
        </div>
    );
};

// Hizmet türü render edici
export const renderServiceType = (type: string, details?: string) => {
    const getServiceIcon = (serviceType: string) => {
        switch (serviceType) {
            case 'ticket': return <Plane className="h-4 w-4 text-blue-600" />;
            case 'accommodation': return <Hotel className="h-4 w-4 text-purple-600" />;
            case 'visa': return <Globe className="h-4 w-4 text-indigo-600" />;
            case 'transfer': return <Car className="h-4 w-4 text-green-600" />;
            default: return <FileText className="h-4 w-4 text-gray-600" />;
        }
    };

    const getServiceTypeText = (serviceType: string) => {
        switch (serviceType) {
            case 'ticket': return 'Bilet';
            case 'accommodation': return 'Konaklama';
            case 'visa': return 'Vize';
            case 'transfer': return 'Transfer';
            default: return serviceType;
        }
    };

    return (
        <div className="flex items-center">
            {getServiceIcon(type)}
            <div className="ml-3">
                <div className="text-sm font-medium text-gray-900">
                    {getServiceTypeText(type)}
                </div>
                {details && (
                    <div className="text-sm text-gray-500">{details}</div>
                )}
            </div>
        </div>
    );
};

// Yolcu bilgisi render edici
export const renderPassenger = (firstName?: string, lastName?: string, phone?: string) => {
    if (!firstName || !lastName) return null;

    return (
        <div>
            <div className="text-sm font-medium text-gray-900">
                {firstName} {lastName}
            </div>
            {phone && (
                <div className="text-sm text-gray-500">{phone}</div>
            )}
        </div>
    );
};

// Uçuş bilgisi render edici
export const renderFlightInfo = (airline: string, flightNumber?: string, route?: string, pnr?: string) => (
    <div>
        <div className="text-sm text-gray-900">
            {airline} {flightNumber}
        </div>
        {route && (
            <div className="text-sm text-gray-500">{route}</div>
        )}
        {pnr && (
            <div className="text-xs text-blue-600 font-mono">
                PNR: {pnr}
            </div>
        )}
    </div>
);

// Tarih ve saat render edici
export const renderDateTime = (date: Date | string, time?: string) => {
    const dateObj = typeof date === 'string' ? new Date(date) : date;
    return (
        <div>
            <div className="text-sm text-gray-900">
                {dateObj.toLocaleDateString('tr-TR')}
            </div>
            {time && (
                <div className="text-sm text-gray-500">{time}</div>
            )}
        </div>
    );
};

// Durum ikonları
export const renderStatusIcon = (status: string) => {
    switch (status) {
        case 'completed':
        case 'paid':
            return <div className="h-2 w-2 bg-green-500 rounded-full"></div>;
        case 'pending':
        case 'waiting':
            return <div className="h-2 w-2 bg-yellow-500 rounded-full"></div>;
        case 'failed':
        case 'cancelled':
            return <div className="h-2 w-2 bg-red-500 rounded-full"></div>;
        case 'processing':
            return <div className="h-2 w-2 bg-blue-500 rounded-full animate-pulse"></div>;
        default:
            return <div className="h-2 w-2 bg-gray-500 rounded-full"></div>;
    }
};

// Durum konfigürasyonları
export const ticketStatusConfig = {
    pending: { color: 'bg-yellow-100 text-yellow-800', label: 'Bekliyor' },
    confirmed: { color: 'bg-blue-100 text-blue-800', label: 'Onaylandı' },
    ticketed: { color: 'bg-green-100 text-green-800', label: 'Biletlendi' },
    cancelled: { color: 'bg-red-100 text-red-800', label: 'İptal' },
    rejected: { color: 'bg-gray-100 text-gray-800', label: 'Reddedildi' }
};

export const checkinStatusConfig = {
    waiting: { color: 'bg-gray-100 text-gray-800', label: 'Bekliyor' },
    ready: { color: 'bg-blue-100 text-blue-800', label: 'Hazır' },
    processing: { color: 'bg-yellow-100 text-yellow-800', label: 'İşleniyor' },
    completed: { color: 'bg-green-100 text-green-800', label: 'Tamamlandı' },
    failed: { color: 'bg-red-100 text-red-800', label: 'Başarısız' },
    expired: { color: 'bg-orange-100 text-orange-800', label: 'Süresi Geçti' },
    cancelled: { color: 'bg-gray-100 text-gray-800', label: 'İptal' }
};

export const accommodationStatusConfig = {
    pending: { color: 'bg-yellow-100 text-yellow-800', label: 'Beklemede' },
    searching: { color: 'bg-blue-100 text-blue-800', label: 'Aranıyor' },
    quoted: { color: 'bg-purple-100 text-purple-800', label: 'Fiyat Verildi' },
    confirmed: { color: 'bg-green-100 text-green-800', label: 'Onaylandı' },
    booked: { color: 'bg-green-100 text-green-800', label: 'Rezerve Edildi' },
    cancelled: { color: 'bg-red-100 text-red-800', label: 'İptal Edildi' },
    completed: { color: 'bg-gray-100 text-gray-800', label: 'Tamamlandı' }
};

export const visaStatusConfig = {
    pending: { color: 'bg-yellow-100 text-yellow-800', label: 'Beklemede' },
    processing: { color: 'bg-blue-100 text-blue-800', label: 'İşlemde' },
    approved: { color: 'bg-green-100 text-green-800', label: 'Onaylandı' },
    rejected: { color: 'bg-red-100 text-red-800', label: 'Reddedildi' },
    delivered: { color: 'bg-purple-100 text-purple-800', label: 'Teslim Edildi' }
};

export const transferStatusConfig = {
    pending: { color: 'bg-yellow-100 text-yellow-800', label: 'Beklemede' },
    confirmed: { color: 'bg-blue-100 text-blue-800', label: 'Onaylandı' },
    assigned: { color: 'bg-purple-100 text-purple-800', label: 'Şoför Atandı' },
    completed: { color: 'bg-green-100 text-green-800', label: 'Tamamlandı' },
    cancelled: { color: 'bg-red-100 text-red-800', label: 'İptal Edildi' }
};

export const invoiceStatusConfig = {
    not_invoiced: { color: 'bg-gray-100 text-gray-800', label: 'Faturalanmadı' },
    invoiced: { color: 'bg-blue-100 text-blue-800', label: 'Faturalandı' },
    paid: { color: 'bg-green-100 text-green-800', label: 'Ödendi' }
};

export const paymentStatusConfig = {
    unpaid: { color: 'bg-red-100 text-red-800', label: 'Ödenmedi' },
    partial: { color: 'bg-yellow-100 text-yellow-800', label: 'Kısmi Ödendi' },
    paid: { color: 'bg-green-100 text-green-800', label: 'Ödendi' },
    refunded: { color: 'bg-purple-100 text-purple-800', label: 'İade Edildi' }
};

// Konaklama bilgisi render edici
export const renderAccommodationInfo = (destination: string, checkIn: Date, checkOut: Date, nights: number) => (
    <div>
        <div className="text-sm font-medium text-gray-900 flex items-center">
            <Hotel className="h-4 w-4 text-purple-600 mr-2" />
            {destination}
        </div>
        <div className="text-sm text-gray-500">
            {checkIn.toLocaleDateString('tr-TR')} - {checkOut.toLocaleDateString('tr-TR')}
        </div>
        <div className="text-xs text-blue-600">
            {nights} gece
        </div>
    </div>
);

// Misafir bilgisi render edici
export const renderGuestInfo = (totalGuests: number, roomCount: number, roomType: string) => {
    const getRoomTypeText = (type: string) => {
        switch (type) {
            case 'single': return 'Tek Kişilik';
            case 'double': return 'Çift Kişilik';
            case 'twin': return 'İki Yataklı';
            case 'triple': return 'Üç Kişilik';
            case 'family': return 'Aile Odası';
            case 'suite': return 'Suit';
            case 'connecting': return 'Bağlantılı Odalar';
            default: return type;
        }
    };

    return (
        <div>
            <div className="text-sm font-medium text-gray-900 flex items-center">
                <User className="h-4 w-4 text-blue-600 mr-2" />
                {totalGuests} misafir
            </div>
            <div className="text-sm text-gray-500">
                {roomCount} oda • {getRoomTypeText(roomType)}
            </div>
        </div>
    );
};

// Bütçe render edici
export const renderBudget = (budgetMin?: number, budgetMax?: number, currency: string = 'TRY') => {
    if (!budgetMin && !budgetMax) return null;

    return (
        <div className="text-sm text-gray-900">
            {budgetMin && budgetMax ? (
                `${budgetMin.toLocaleString()} - ${budgetMax.toLocaleString()} ${currency}`
            ) : budgetMin ? (
                `${budgetMin.toLocaleString()}+ ${currency}`
            ) : (
                `${budgetMax?.toLocaleString()} ${currency} altı`
            )}
        </div>
    );
};

// Otel bilgisi render edici
export const renderHotelInfo = (hotelName?: string, confirmationNumber?: string, totalPrice?: number, currency: string = 'TRY') => {
    if (!hotelName) return null;

    return (
        <div>
            <div className="text-sm font-medium text-gray-900">
                {hotelName}
            </div>
            {confirmationNumber && (
                <div className="text-xs text-blue-600 font-mono">
                    Rezervasyon: {confirmationNumber}
                </div>
            )}
            {totalPrice && (
                <div className="text-sm text-green-600 font-medium">
                    {totalPrice.toLocaleString()} {currency}
                </div>
            )}
        </div>
    );
};

// Vize bilgisi render edici
export const renderVisaInfo = (country: string, visaType: string, passportNumber: string) => (
    <div>
        <div className="text-sm font-medium text-gray-900 flex items-center">
            <Globe className="h-4 w-4 text-indigo-600 mr-2" />
            {country}
        </div>
        <div className="text-sm text-gray-500">
            {getVisaTypeText(visaType)}
        </div>
        <div className="text-xs text-blue-600 font-mono">
            Pasaport: {passportNumber}
        </div>
    </div>
);

// Vize tarihleri render edici
export const renderVisaDates = (applicationDate: Date, travelDate: Date, appointmentDate?: Date) => (
    <div>
        <div className="text-sm text-gray-900">
            Başvuru: {applicationDate.toLocaleDateString('tr-TR')}
        </div>
        <div className="text-sm text-gray-500">
            Seyahat: {travelDate.toLocaleDateString('tr-TR')}
        </div>
        {appointmentDate && (
            <div className="text-xs text-purple-600">
                Randevu: {appointmentDate.toLocaleDateString('tr-TR')}
            </div>
        )}
    </div>
);

// Vize belgeleri render edici
export const renderVisaDocuments = (documents: Array<{ status: string }>, requiredDocuments: string[]) => (
    <div>
        <div className="text-sm text-gray-900">
            {documents.length}/{requiredDocuments.length} belge
        </div>
        <div className="text-xs text-gray-500">
            {documents.filter(d => d.status === 'approved').length} onaylı
        </div>
        {documents.length < requiredDocuments.length && (
            <div className="text-xs text-red-600">
                {requiredDocuments.length - documents.length} eksik
            </div>
        )}
    </div>
);

// Transfer bilgisi render edici
export const renderTransferInfo = (transferType: string, pickupLocation: string, dropoffLocation: string) => (
    <div>
        <div className="text-sm font-medium text-gray-900 flex items-center">
            <Car className="h-4 w-4 text-green-600 mr-2" />
            {getTransferTypeText(transferType)}
        </div>
        <div className="text-sm text-gray-500">
            <div className="flex items-center">
                <span className="text-green-600">●</span>
                <span className="ml-1">{pickupLocation}</span>
            </div>
            <div className="flex items-center">
                <span className="text-red-600">●</span>
                <span className="ml-1">{dropoffLocation}</span>
            </div>
        </div>
    </div>
);

// Transfer araç bilgisi render edici
export const renderVehicleInfo = (vehicleType: string, totalPassengers: number, vehicleCount: number = 1) => (
    <div>
        <div className="text-sm font-medium text-gray-900">
            {getVehicleTypeText(vehicleType)}
        </div>
        <div className="text-sm text-gray-500 flex items-center">
            <User className="h-3 w-3 mr-1" />
            {totalPassengers} yolcu
        </div>
        {vehicleCount > 1 && (
            <div className="text-xs text-blue-600">
                {vehicleCount} araç
            </div>
        )}
    </div>
);

// Şoför bilgisi render edici
export const renderDriverInfo = (
    driver?: { firstName: string; lastName: string; phone: string },
    vehicle?: { plateNumber: string }
) => {
    if (!driver) {
        return <span className="text-sm text-gray-400">Atanmadı</span>;
    }

    return (
        <div>
            <div className="text-sm font-medium text-gray-900">
                {driver.firstName} {driver.lastName}
            </div>
            <div className="text-sm text-gray-500">
                {driver.phone}
            </div>
            {vehicle && (
                <div className="text-xs text-blue-600">
                    {vehicle.plateNumber}
                </div>
            )}
        </div>
    );
};

// Yardımcı fonksiyonlar
const getVisaTypeText = (type: string) => {
    switch (type) {
        case 'tourist': return 'Turist Vizesi';
        case 'business': return 'İş Vizesi';
        case 'student': return 'Öğrenci Vizesi';
        case 'work': return 'Çalışma Vizesi';
        case 'transit': return 'Transit Vize';
        case 'family': return 'Aile Birleşimi';
        case 'medical': return 'Tedavi Vizesi';
        default: return type;
    }
};

const getTransferTypeText = (type: string) => {
    switch (type) {
        case 'airport_pickup': return 'Havalimanı Karşılama';
        case 'airport_dropoff': return 'Havalimanı Uğurlama';
        case 'hotel_transfer': return 'Otel Transferi';
        case 'city_tour': return 'Şehir Turu';
        case 'point_to_point': return 'Nokta-Nokta';
        case 'hourly': return 'Saatlik';
        case 'daily': return 'Günlük';
        default: return type;
    }
};

const getVehicleTypeText = (type: string) => {
    switch (type) {
        case 'economy': return 'Ekonomi';
        case 'comfort': return 'Konfor';
        case 'luxury': return 'Lüks';
        case 'minibus': return 'Minibüs';
        case 'bus': return 'Otobüs';
        case 'vip': return 'VIP';
        default: return type;
    }
};