
export interface Customer {
  id: string;
  type: 'individual' | 'corporate';

  // Bireysel müşteri bilgileri
  firstName?: string;
  lastName?: string;

  // Kurumsal müşteri bilgileri
  companyName?: string;
  taxNumber?: string;
  taxOffice?: string;

  // Ortak bilgiler
  email?: string;
  phone: string;
  address?: string;
  city?: string;
  country?: string;
  notes?: string;
  isVip: boolean;
  isActive: boolean;

  // Kurumsal müşteri için
  parentCompanyId?: string; // Alt şirket ise
  contactPerson?: string;
  department?: string;

  createdAt: Date;
  updatedAt: Date;
}

export interface CorporateUser {
  id: string;
  customerId: string; // Hangi kurumsal müşteriye ait
  firstName: string;
  lastName: string;
  email: string;
  phone?: string;
  department?: string;
  position?: string;
  permissions: UserPermission[];
  isActive: boolean;
  lastLoginAt?: Date;
  createdAt: Date;
  updatedAt: Date;
}

export type UserPermission =
  | 'create_ticket_request'
  | 'create_accommodation_request'
  | 'create_visa_request'
  | 'create_transfer_request'
  | 'view_all_requests'
  | 'view_own_requests'
  | 'manage_users';

// Acente kullanıcı yönetimi için yeni types
export interface AgencyUser {
  id: string;
  firstName: string;
  lastName: string;
  email: string;
  phone?: string;
  avatar?: string;
  
  // Departman
  departmentId: string;
  department: Department;

  // Yetkilendirme
  role: UserRole;
  permissions: AgencyPermission[];
  
  // Durum
  isActive: boolean;
  isOnline?: boolean;
  lastLoginAt?: Date;
  
  // Yönetici bilgileri
  managerId?: string;
  manager?: AgencyUser;
  
  // İstatistikler
  totalTickets?: number;
  totalRevenue?: number;
  
  // Meta
  createdAt: Date;
  updatedAt: Date;
  createdBy?: string;
}

export interface Department {
  id: string;
  name: string;
  description?: string;
  color: string;

  // Hiyerarşi
  parentDepartmentId?: string;
  parentDepartment?: Department;

  // Yönetici
  managerId?: string;
  manager?: AgencyUser;

  // Yetkiler
  permissions?: AgencyPermission[];

  // Durum
  isActive: boolean;

  // İstatistikler
  userCount: number;

  // Meta
  createdAt: Date;
  updatedAt: Date;
}

export type UserRole =
  | 'system_admin'        // Sistem Yöneticisi
  | 'agency_manager'      // Acente Yöneticisi
  | 'department_manager'  // Departman Yöneticisi
  | 'senior_specialist'   // Kıdemli Uzman
  | 'specialist'          // Uzman
  | 'assistant'           // Uzman Yardımcısı
  | 'intern';             // Stajyer

export type AgencyPermission =
  // Bilet işlemleri
  | 'tickets_view_all'
  | 'tickets_view_own'
  | 'tickets_create'
  | 'tickets_edit'
  | 'tickets_delete'
  | 'tickets_approve'
  | 'tickets_issue'
  
  // Check-in işlemleri
  | 'checkin_view_all'
  | 'checkin_view_own'
  | 'checkin_manage'
  | 'checkin_retry'
  
  // Konaklama işlemleri
  | 'accommodation_view_all'
  | 'accommodation_view_own'
  | 'accommodation_create'
  | 'accommodation_edit'
  | 'accommodation_delete'
  | 'accommodation_quote'
  | 'accommodation_book'
  
  // Vize işlemleri
  | 'visa_view_all'
  | 'visa_view_own'
  | 'visa_create'
  | 'visa_edit'
  | 'visa_delete'
  | 'visa_approve'
  | 'visa_documents'
  
  // Transfer işlemleri
  | 'transfer_view_all'
  | 'transfer_view_own'
  | 'transfer_create'
  | 'transfer_edit'
  | 'transfer_delete'
  | 'transfer_assign'
  | 'transfer_track'
  
  // Müşteri yönetimi
  | 'customers_view_all'
  | 'customers_view_own'
  | 'customers_create'
  | 'customers_edit'
  | 'customers_delete'
  | 'customers_export'
  
  // Muhasebe
  | 'accounting_view_all'
  | 'accounting_view_own'
  | 'accounting_create'
  | 'accounting_edit'
  | 'accounting_invoice'
  | 'accounting_payment'
  | 'accounting_reports'
  
  // Kullanıcı yönetimi
  | 'users_view_all'
  | 'users_create'
  | 'users_edit'
  | 'users_delete'
  | 'users_permissions'
  
  // Departman yönetimi
  | 'departments_view'
  | 'departments_create'
  | 'departments_edit'
  | 'departments_delete'
  
  // Raporlar
  | 'reports_view'
  | 'reports_export'
  | 'reports_advanced'
  
  // Sistem ayarları
  | 'settings_view'
  | 'settings_edit'
  | 'settings_system';

export interface UserActivity {
  id: string;
  userId: string;
  user: AgencyUser;
  action: string;
  description: string;
  entityType?: string; // 'ticket', 'customer', 'user' etc.
  entityId?: string;
  ipAddress?: string;
  userAgent?: string;
  createdAt: Date;
}

export interface UserSession {
  id: string;
  userId: string;
  user: AgencyUser;
  token: string;
  ipAddress: string;
  userAgent: string;
  isActive: boolean;
  expiresAt: Date;
  createdAt: Date;
  lastActivityAt: Date;
}

// Yolcu bilgileri - bilet ve check-in için ortak kullanım
export interface PassengerInfo {
  id: string;
  firstName: string;
  lastName: string;
  phone?: string;
  email?: string;
  
  // Kurumsal müşteri çalışanları için
  employeeId?: string;
  department?: string;
  position?: string;
  
  // Çocuk yolcu bilgileri
  age?: number;
  isChild?: boolean;
  
  // Özel ihtiyaçlar
  specialNeeds?: string;
}

// Bilet Talebi - Müşterilerden gelen talepler
export interface TicketRequest {
  id: string;
  customerId: string;
  customer: Customer;

  // Kurumsal müşteri için
  requestedBy?: CorporateUser; // Hangi kullanıcı tarafından talep edildi

  // Passenger Details - çoklu yolcu desteği
  passengers: PassengerInfo[];

  // Flight Details
  airline: 'THY' | 'PC' | 'AJET' | 'OTHER';
  flightNumber?: string;
  departureAirport: string;
  arrivalAirport: string;
  departureDate: Date;
  departureTime?: string;

  // Request Details
  status: TicketRequestStatus;
  requestDate: Date;
  ticketPrice?: number;
  currency?: 'TRY' | 'USD' | 'EUR';

  // Agent Notes
  agentNotes?: string;
  pnr?: string; // Biletlendiğinde eklenir

  // Meta
  createdAt: Date;
  updatedAt: Date;
}

export type TicketRequestStatus =
  | 'pending' // Yeni talep
  | 'confirmed' // Onaylandı, biletlenecek
  | 'ticketed' // Biletlendi
  | 'cancelled' // İptal edildi
  | 'rejected'; // Reddedildi

// Check-in için biletlenmiş uçuşlar
export interface CheckinTicket {
  id: string;
  ticketRequestId: string; // Hangi talepten geldiği
  pnr: string;
  customerId: string;
  customer: Customer;

  // Kurumsal müşteri için
  requestedBy?: CorporateUser; // Hangi kullanıcı tarafından talep edildi

  // Passenger Details - çoklu yolcu desteği
  passengers: PassengerInfo[];

  // Flight Details
  airline: 'THY' | 'PC' | 'AJET' | 'OTHER';
  flightNumber: string;
  departureAirport: string;
  arrivalAirport: string;
  departureDate: Date;
  departureTime: string;

  // Check-in Details
  checkinStatus: CheckInStatus;
  checkinOpenTime?: Date; // 24 saat öncesi
  checkinCompletedAt?: Date;
  checkinAttempts: number;
  lastCheckinAttempt?: Date;

  // Passenger Results - her yolcu için ayrı sonuç
  passengerResults: PassengerCheckinResult[];

  // SMS ve genel bilgiler
  smsStatus: 'pending' | 'sent' | 'failed';
  emailStatus?: 'pending' | 'sent' | 'failed'; // AJET ve Pegasus için
  errorMessage?: string;

  // Meta
  createdAt: Date;
  updatedAt: Date;
  isAutoCheckinEnabled: boolean;
  notes?: string;
}

// Yolcu check-in sonucu
export interface PassengerCheckinResult {
  passengerId: string;
  checkinStatus: CheckInStatus;
  seat?: string;
  boardingPassUrl?: string;
  errorMessage?: string;
}

export type CheckInStatus =
  | 'waiting' // Henüz check-in zamanı gelmedi
  | 'ready' // Check-in açık, bekliyor
  | 'processing' // Check-in yapılıyor
  | 'completed' // Başarıyla tamamlandı
  | 'failed' // Başarısız
  | 'expired' // Check-in süresi geçti
  | 'cancelled'; // İptal edildi

export interface AutoCheckinJob {
  id: string;
  checkinTicketId: string;
  checkinTicket: CheckinTicket;
  scheduledFor: Date;
  status: 'scheduled' | 'processing' | 'completed' | 'failed' | 'cancelled';
  attempts: number;
  lastAttempt?: Date;
  nextRetry?: Date;
  createdAt: Date;
  updatedAt: Date;
}

export interface DashboardStats {
  todayCheckIns: number;
  todayScheduled: number;
  successRate: number;
  totalTicketRequests: number;
  pendingRequests: number;
  ticketedToday: number;
  upcomingFlights: number;
  failedCheckIns: number;
}

// Muhasebe için fatura ve ödeme yönetimi
export interface Invoice {
  id: string;
  invoiceNumber: string;
  customerId: string;
  customer: Customer;
  
  // Fatura detayları
  issueDate: Date;
  dueDate: Date;
  amount: number;
  currency: 'TRY' | 'USD' | 'EUR';
  taxAmount: number;
  totalAmount: number;
  
  // Fatura kalemleri
  items: InvoiceItem[];
  
  // Durum
  status: InvoiceStatus;
  paymentStatus: PaymentStatus;
  
  // Ödeme bilgileri
  paidAmount: number;
  paidDate?: Date;
  paymentMethod?: PaymentMethod;
  
  // Notlar
  notes?: string;
  
  // Meta
  createdAt: Date;
  updatedAt: Date;
}

export interface InvoiceItem {
  id: string;
  serviceType: 'ticket' | 'accommodation' | 'visa' | 'transfer';
  serviceId: string; // İlgili hizmetin ID'si
  description: string;
  quantity: number;
  unitPrice: number;
  totalPrice: number;
  taxRate: number;
  taxAmount: number;
}

export type InvoiceStatus = 
  | 'draft' // Taslak
  | 'sent' // Gönderildi
  | 'viewed' // Görüntülendi
  | 'overdue' // Vadesi geçti
  | 'cancelled'; // İptal edildi

export type PaymentStatus = 
  | 'unpaid' // Ödenmedi
  | 'partial' // Kısmi ödendi
  | 'paid' // Tamamen ödendi
  | 'refunded'; // İade edildi

export type PaymentMethod = 
  | 'cash' // Nakit
  | 'bank_transfer' // Havale/EFT
  | 'credit_card' // Kredi kartı
  | 'check' // Çek
  | 'other'; // Diğer

// Muhasebe için hizmet özeti
export interface AccountingService {
  id: string;
  type: 'ticket' | 'accommodation' | 'visa' | 'transfer';
  customerId: string;
  customer: Customer;
  
  // Hizmet bilgileri
  serviceDetails: string;
  completedDate: Date;
  amount: number;
  currency: 'TRY' | 'USD' | 'EUR';
  
  // Faturalama durumu
  invoiceStatus: 'not_invoiced' | 'invoiced' | 'paid';
  invoiceId?: string;
  
  // Referans bilgileri
  referenceId: string; // Orijinal talep ID'si
  
  // Meta
  createdAt: Date;
  updatedAt: Date;
}

// Muhasebe için gelir kayıtları
export interface RevenueRecord {
  id: string;
  type: 'ticket' | 'accommodation' | 'visa' | 'transfer';
  customerId: string;
  customer: Customer;
  requestedBy?: CorporateUser;

  // İşlem detayları
  description: string;
  amount: number;
  currency: 'TRY' | 'USD' | 'EUR';
  commission?: number; // Komisyon oranı
  netAmount?: number; // Net kazanç

  // Referans bilgileri
  referenceId: string; // Bilet PNR, rezervasyon no, vs.
  referenceType: string; // "Bilet", "Otel Rezervasyonu", "Vize Başvurusu", vs.

  // Durum bilgileri
  status: RevenueStatus;
  paymentStatus: PaymentStatus;
  invoiceNumber?: string;

  // Tarihler
  serviceDate: Date; // Hizmet tarihi (uçuş tarihi, konaklama tarihi vs.)
  completedAt?: Date; // İşlem tamamlanma tarihi
  invoicedAt?: Date; // Faturalandırma tarihi
  paidAt?: Date; // Ödeme tarihi

  // Meta
  notes?: string;
  createdAt: Date;
  updatedAt: Date;
}

export type RevenueStatus =
  | 'pending' // Beklemede
  | 'confirmed' // Onaylandı
  | 'completed' // Tamamlandı
  | 'cancelled' // İptal edildi
  | 'refunded'; // İade edildi

export type PaymentStatus =
  | 'unpaid' // Ödenmedi
  | 'partial' // Kısmi ödendi
  | 'paid' // Ödendi
  | 'overdue' // Vadesi geçti
  | 'refunded'; // İade edildi

// Konaklama Talebi
export interface AccommodationRequest {
  id: string;
  customerId: string;
  customer: Customer;

  // Kurumsal müşteri için
  requestedBy?: CorporateUser;

  // Konaklama detayları
  destination: string;
  checkInDate: Date;
  checkOutDate: Date;
  nights: number;
  
  // Misafir bilgileri
  guests: AccommodationGuest[];
  totalGuests: number;
  
  // Oda tercihleri
  roomType: RoomType;
  roomCount: number;
  
  // Bütçe ve tercihler
  budgetMin?: number;
  budgetMax?: number;
  currency: 'TRY' | 'USD' | 'EUR';
  
  // Özel istekler
  specialRequests?: string;
  mealPlan?: MealPlan;
  hotelCategory?: HotelCategory;
  
  // Durum
  status: AccommodationStatus;
  
  // Rezervasyon bilgileri (onaylandığında)
  hotelName?: string;
  hotelAddress?: string;
  confirmationNumber?: string;
  totalPrice?: number;
  
  // Notlar
  agentNotes?: string;
  
  // Meta
  createdAt: Date;
  updatedAt: Date;
}

export interface AccommodationGuest {
  id: string;
  firstName: string;
  lastName: string;
  age?: number;
  isChild: boolean;
  specialNeeds?: string;
}

export type RoomType = 
  | 'single' // Tek kişilik
  | 'double' // Çift kişilik
  | 'twin' // İki yataklı
  | 'triple' // Üç kişilik
  | 'family' // Aile odası
  | 'suite' // Suit
  | 'connecting'; // Bağlantılı odalar

export type MealPlan = 
  | 'room_only' // Sadece oda
  | 'breakfast' // Kahvaltı dahil
  | 'half_board' // Yarım pansiyon
  | 'full_board' // Tam pansiyon
  | 'all_inclusive'; // Her şey dahil

export type HotelCategory = 
  | 'budget' // Ekonomik
  | 'standard' // Standart
  | 'comfort' // Konforlu
  | 'luxury' // Lüks
  | 'boutique'; // Butik

export type AccommodationStatus = 
  | 'pending' // Beklemede
  | 'searching' // Aranıyor
  | 'quoted' // Fiyat verildi
  | 'confirmed' // Onaylandı
  | 'booked' // Rezerve edildi
  | 'cancelled' // İptal edildi
  | 'completed'; // Tamamlandı

// Vize Talebi
export interface VisaRequest {
  id: string;
  customerId: string;
  customer: Customer;

  // Kurumsal müşteri için
  requestedBy?: CorporateUser;

  // Başvuru sahibi bilgileri
  applicant: VisaApplicant;

  // Vize detayları
  country: string;
  visaType: VisaType;
  applicationDate: Date;
  travelDate: Date;
  returnDate?: Date;
  
  // Başvuru bilgileri
  status: VisaStatus;
  priority: VisaPriority;
  
  // Belgeler
  documents: VisaDocument[];
  requiredDocuments: string[];
  
  // Randevu bilgileri
  appointmentDate?: Date;
  appointmentLocation?: string;
  
  // Sonuç bilgileri
  decisionDate?: Date;
  visaNumber?: string;
  validUntil?: Date;
  
  // Fiyat bilgileri
  applicationFee?: number;
  serviceFee?: number;
  totalAmount?: number;
  currency: 'TRY' | 'USD' | 'EUR';
  
  // Notlar
  agentNotes?: string;
  specialRequests?: string;
  
  // Meta
  createdAt: Date;
  updatedAt: Date;
}

export interface VisaApplicant {
  id: string;
  firstName: string;
  lastName: string;
  passportNumber: string;
  passportExpiry: Date;
  nationality: string;
  birthDate: Date;
  birthPlace: string;
  phone?: string;
  email?: string;
  occupation?: string;
  employer?: string;
}

export interface VisaDocument {
  id: string;
  name: string;
  type: string;
  uploadedAt: Date;
  fileUrl: string;
  isRequired: boolean;
  status: 'pending' | 'approved' | 'rejected';
}

export type VisaType = 
  | 'tourist' // Turist
  | 'business' // İş
  | 'student' // Öğrenci
  | 'work' // Çalışma
  | 'transit' // Transit
  | 'family' // Aile birleşimi
  | 'medical'; // Tedavi

export type VisaStatus = 
  | 'pending' // Beklemede
  | 'document_review' // Belge incelemesi
  | 'appointment_scheduled' // Randevu alındı
  | 'interview_completed' // Mülakat tamamlandı
  | 'processing' // İşlemde
  | 'approved' // Onaylandı
  | 'rejected' // Reddedildi
  | 'delivered' // Teslim edildi
  | 'cancelled'; // İptal edildi

export type VisaPriority = 'low' | 'medium' | 'high' | 'urgent';

// Transfer Talebi
export interface TransferRequest {
  id: string;
  customerId: string;
  customer: Customer;

  // Kurumsal müşteri için
  requestedBy?: CorporateUser;

  // Transfer detayları
  transferType: TransferType;
  pickupLocation: string;
  dropoffLocation: string;
  pickupDate: Date;
  pickupTime: string;
  
  // Yolcu bilgileri
  passengers: TransferPassenger[];
  totalPassengers: number;
  
  // Araç tercihleri
  vehicleType: VehicleType;
  vehicleCount: number;
  
  // Özel istekler
  specialRequests?: string;
  hasLuggage: boolean;
  needsChildSeat: boolean;
  isWheelchairAccessible: boolean;
  
  // Durum
  status: TransferStatus;
  
  // Atama bilgileri
  assignedDriver?: TransferDriver;
  assignedVehicle?: TransferVehicle;
  
  // Fiyat bilgileri
  estimatedPrice?: number;
  finalPrice?: number;
  currency: 'TRY' | 'USD' | 'EUR';
  
  // Notlar
  agentNotes?: string;
  driverNotes?: string;
  
  // Meta
  createdAt: Date;
  updatedAt: Date;
}

export interface TransferPassenger {
  id: string;
  firstName: string;
  lastName: string;
  phone?: string;
  isChild: boolean;
  age?: number;
  specialNeeds?: string;
}

export interface TransferDriver {
  id: string;
  firstName: string;
  lastName: string;
  phone: string;
  licenseNumber: string;
  rating?: number;
  isActive: boolean;
}

export interface TransferVehicle {
  id: string;
  brand: string;
  model: string;
  year: number;
  plateNumber: string;
  capacity: number;
  type: VehicleType;
  features: string[];
  isActive: boolean;
}

export type TransferType = 
  | 'airport_pickup' // Havalimanı karşılama
  | 'airport_dropoff' // Havalimanı uğurlama
  | 'hotel_transfer' // Otel transferi
  | 'city_tour' // Şehir turu
  | 'point_to_point' // Nokta-nokta
  | 'hourly' // Saatlik
  | 'daily'; // Günlük

export type VehicleType = 
  | 'economy' // Ekonomi (4 kişi)
  | 'comfort' // Konfor (4 kişi)
  | 'luxury' // Lüks (4 kişi)
  | 'minibus' // Minibüs (8-14 kişi)
  | 'bus' // Otobüs (15+ kişi)
  | 'vip'; // VIP araç

export type TransferStatus = 
  | 'pending' // Beklemede
  | 'quoted' // Fiyat verildi
  | 'confirmed' // Onaylandı
  | 'assigned' // Şoför atandı
  | 'in_progress' // Devam ediyor
  | 'completed' // Tamamlandı
  | 'cancelled'; // İptal edildi

export interface AccountingStats {
  totalRevenue: number;
  monthlyRevenue: number;
  pendingPayments: number;
  completedTransactions: number;
  ticketRevenue: number;
  accommodationRevenue: number;
  visaRevenue: number;
  transferRevenue: number;
}